package com.aliyun.drc.client.impl;

import com.aliyun.drc.client.*;
import com.aliyun.drc.client.enums.DBType;
import com.aliyun.drc.client.enums.LogLevel;
import com.aliyun.drc.client.enums.Status;
import com.aliyun.drc.client.message.DataMessage;
import com.aliyun.drc.client.message.Message;
import com.aliyun.drc.client.message.RedirectMessage;
import com.aliyun.drc.utils.DataFilterUtil;

import java.io.IOException;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Implement the DRCClient, using ManagerCommunicator and ServerCommunicator
 * to connect with manager and daemon server respectively, receive data, parse
 * and notify listeners.
 */
public class DRCClientImpl implements DRCClient, Runnable {


    /* Filter for db, table and columns. */
    private DataFilterBase filter = null;

    /* Direct the client to quit or still run. */
    private volatile boolean quit = false;

    /* Local persistent file for drc client's internal use. */
    private LocalityFile binlogfile = null;

    private int oneHbEverySeconds = 1;

    /* Running status. */
    private Status status = Status.RAW;

    /* The server to be communicated with. */
    private ServerProxy server;

    /* All DRC client configures and user-defined parameters. */
    private final DRCConfig config;

    /* Service thread. */
    private Thread serviceThread = null;

    /* User-registered listeners. */
    private List<Listener> listeners;

    /* User-defined period to detect notify() consumed time. */
    private long lastNotifiedTime;

    private long notifiedPeriod = 60; /* 1 minute */

    private String blackList;
    
    private String storeIpAndPort = null;

    private int accumulatedHeartbeats;

    private int retriedTimes;

    private long accumulatedTime;

    private long count;

    private RecordsCache cache;

    private int checkpointPeriod;

    private int recordCheckpointTime;

    private volatile boolean suspend = false;


    private boolean useDrcnet = true;

    private DBType dbType = null;

    public static final String TOPIC_MATCH_REGEX = "^\\w+\\-[0-9]+\\-[0-9]+$";

    /**
     * Constructor of the DRCClientImpl.
     *
     * @param propertiesFilename is the file containing DRC client configures.
     * @throws IOException
     */
    public DRCClientImpl(final String propertiesFilename)
            throws IOException {
        config = new DRCConfig(propertiesFilename);
        listeners = new ArrayList<Listener>();
    }

    /**
     * Constructor of the DRCClientImpl.
     *
     * @param reader is the reader providing DRC client onfigures.
     * @throws IOException
     */
    public DRCClientImpl(final Reader reader)
            throws IOException {
        config = new DRCConfig(reader);
        listeners = new ArrayList<Listener>();
    }

    /**
     * Constructor of the DRCClientImpl.
     *
     * @param properties
     * @throws IOException
     */
    public DRCClientImpl(final Properties properties)
            throws IOException {
        config = new DRCConfig(properties);
        listeners = new ArrayList<Listener>();
    }

    /**
     * Add drc client required properties.
     *
     * @param name  is the name of the property.
     * @param value is the value of the property.
     */
    @Override
    public void addDRCConfigure(final String name, final String value) {
        config.addConfigure(name, value);
    }

    /**
     * Get all configures of drc client.
     *
     * @return all configures of drc client.
     */
    @Override
    public Map<String, String> getDRCConfigures() {
        return config.getConfigures();
    }

    /**
     * Get one configure for drc client.
     *
     * @param name is the name of the configure.
     * @return the value of the configure.
     */
    @Override
    public String getDRCConfigure(final String name) {
        return config.getConfigure(name);
    }

    /**
     * Add user-defined parameter which is sent to the server.
     *
     * @param name  is the name of the parameter.
     * @param value is the value of the parameter.
     */
    @Override
    public void addUserParameter(final String name, final String value) {
        config.addParam(name, value);
    }

    /**
     * Get all parameters added by the user.
     *
     * @return all parameters.
     */
    @Override
    public Map<String, String> getUserParameters() {
        return config.getParams();
    }

    /**
     * Get one parameter added by the user.
     *
     * @param name is the name of the parameter.
     * @return the value of the parameter.
     */
    @Override
    public String getUserParameter(final String name) {
        return config.getParam(name);
    }

    /**
     * Used by users to initialize the service by gmtModified.
     *
     * @param groupName      is the name of the users' group.
     * @param dbName         is the partial database name.
     * @param identification is the user's password.
     * @param startingPoint  is either "checkpoint" or "gmt_modified".
     * @param localFilename  path to binary log file.
     * @throws IOException log file related IO errors.
     */
    @Override
    public void initService(final String groupName,
                            final String dbName,
                            final String identification,
                            final String startingPoint,
                            final String localFilename)
            throws Exception {
        initService(groupName, dbName, identification, startingPoint,
                Location.PRIM_META, Location.PRIM_TKNM, Location.PRIM_INST, localFilename);
    }

    /**
     * Used by users to reconnect according to the log fle.
     *
     * @param groupName      is the name of the users' group.
     * @param dbName         is the partial database name.
     * @param identification is the user's password.
     * @param localFilename  is path to binary log file.
     * @throws DRCClientException if the log file not exists.
     * @throws IOException        if the log file is not readable.
     */
    @Override
    public void initService(final String groupName,
                            final String dbName,
                            final String identification,
                            final String localFilename)
            throws Exception {

        final Checkpoint location = getLocalInfo(localFilename);
        initService(groupName, dbName, identification, location.getPosition().equals("@") ? location.getTimestamp() : location.getPosition(),
                "0", "", location.getServerId(), localFilename);
    }


    public static boolean isValidTopicName(String topicName) {
        Pattern p = Pattern.compile(TOPIC_MATCH_REGEX);
        Matcher m = p.matcher(topicName);
        return m.matches();
    }

    /**
     * Initialize the service by manually setting restarted checkpoint.
     * Notice that the user should also provide the respective metadata version
     * and assigned task name.
     *
     * @param groupName      is the name of the users' group.
     * @param dbName         is the partial database name.
     * @param identification is the user's password.
     * @param startingPoint  is either "checkpoint" or "gmt_modified".
     * @param metaVersion    is the metadata version.
     * @param taskName       is the task name.
     * @param instance       is the address of the last database instance.
     * @param localFilename  is path to binary log file.
     * @throws IOException if the log file cannot be created.
     */
    @Override
    public synchronized void initService(final String groupName,
                                         final String dbName,
                                         final String identification,
                                         final String startingPoint,
                                         final String metaVersion,
                                         final String taskName,
                                         final String instance,
                                         final String localFilename) throws Exception {
        if (status != Status.RAW) {
            throw new Exception("server state is error,current state:" + status);
        }
        config.setGroupName(groupName);
        config.setDbname(dbName);
        config.setIdentification(identification);
        config.setStartingPoint(startingPoint);
        config.setInstance(instance);

        if (dbName != null) {
            if(!isValidTopicName(dbName)) {
                // db is a branched db name
                filter.setBranchDb(dbName);
            }
        }
        config.setBlackList(blackList);
        config.setDataFilter(filter);
        config.setRequiredTablesAndColumns(filter.toString());

        if (localFilename != null) {
            config.setBinlogFilename(localFilename);
            binlogfile = new LocalityFile(localFilename, 0, 104857600/*100M*/);
        } else {
            binlogfile = null;
        }
        status = Status.INIT;
        sendLog2Listeners(LogLevel.INFO, "Initialize the service with starting point: " + startingPoint);
    }

    @Override
    public synchronized void initService(final String groupName,
                                         final String dbName,
                                         final String identification,
                                         final Checkpoint checkpoint,
                                         final String localFilename) throws Exception {
        if (status != Status.RAW) {
            throw new Exception("server state is error,current state:" + status);
        }
        config.setGroupName(groupName);
        config.setDbname(dbName);
        config.setIdentification(identification);
        config.setCheckpoint(checkpoint);
        if (dbName != null) {
            if (!isValidTopicName(dbName)) {
                // db is a branched db name
                filter.setBranchDb(dbName);
            }
        }
        config.setBlackList(blackList);
        config.setDataFilter(filter);
        config.setRequiredTablesAndColumns(filter.toString());

        if (localFilename != null) {
            config.setBinlogFilename(localFilename);
            binlogfile = new LocalityFile(localFilename, 0, 104857600/*100M*/);
        } else {
            binlogfile = null;
        }
        status = Status.INIT;
        sendLog2Listeners(LogLevel.INFO,
                "Initialize the service with starting point: " + checkpoint.toString());
    }

    /**
     * Connect to one of the cluster managers, retried if previous one failed.
     *
     * @throws Exception
     */
    private void connectClusterManager(final String urls) throws Exception {
        ClusterManagers managers = new ClusterManagers(urls);
        server = managers.findStore(config, this);
        if (server != null) {
        	//fomat may be http://ip:port/topic, we use match to get the dst string
        	String storeUrl = server.getUrl();
        	Pattern pattern = Pattern.compile("(\\d+\\.\\d+\\.\\d+\\.\\d+):(\\d+)");
    		Matcher matcher = pattern.matcher(storeUrl);
    		//if not found the pattern, just ignore.
    		// after all. this is the special interface for TT
    		if(matcher.find()) {			 
    			storeIpAndPort = matcher.group(0);;
    		}
            sendLog2Listeners(LogLevel.INFO, "Connect to " + server.getUrl() + " successfully");
        }
    }

    private void resetServer() throws Exception {
        if (status != Status.STOP) {
            throw new Exception("server state is error,current state:" + status);
        }
        if (server != null) {
            server.close();
            server = null;
        }
        sendLog2Listeners(LogLevel.INFO, "Reset Server");
        status = Status.INIT;
    }


    public void run() {

        /* Tried times to reconnect to the server. */

        lastNotifiedTime = System.currentTimeMillis();

        /* Received message and message id. */
        Message message;
        MessageId messageId = new MessageId(config.getDbname());

        /* Every how many messages to record one checkpoint. */
        checkpointPeriod = config.getCheckpointPeriod();
        recordCheckpointTime = checkpointPeriod;

        if (config.isTxnRequiredCompleted()) {
            cache = new RecordsCache(config);
        }
        useDrcnet = config.getUseDrcNet();
        while (!isQuit()) {
            try {
                if (suspend) {
                    TimeUnit.SECONDS.sleep(1);
                    continue;
                }

                if(!useDrcnet) {
                	message =  server.getResponse(config.isBinaryFormat());
                } else {
                	message = server.getDrcNetResponse(config.isBinaryFormat());
                }

                messageId.meet(message.getMid());

                switch (message.getType()) {
                    case 300:
                        processRedirectMessage(message);
                        messageId.reset(config.getDbname());
                        break;
                    case 100:
                        processDataMessage(message);
                        notifyDataMessage(message);
                        persistCheckpoint();
                        break;
                    default:
                        throw new DRCClientException("Wrong DRCMessage type " + message.getType());
                }
            } catch (Exception e) {
                if (!isQuit()) {
                    String errMessage = e.getMessage();
                    if (errMessage == null)
                        errMessage = "server closed unexpectedly";
                    String urlMessage;
                    if (server != null)
                        urlMessage = server.getUrl();
                    else
                        urlMessage = new String("unknown url");
                    sendLog2Listeners(LogLevel.WARN, "Connect to " + urlMessage +
                            " failed because " + errMessage);
                    StackTraceElement[] stack = e.getStackTrace();
                    for (StackTraceElement element : stack) {
                        sendLog2Listeners(LogLevel.WARN, element.toString());
                    }
                    e.printStackTrace();
                    if (retriedTimes++ < config.getMaxRetriedTimes()) {
                        messageId.reset(config.getDbname());
                        RedirectMessage redirectMessage = new RedirectMessage();
                        redirectMessage.setId(0);
                        redirectMessage.setType(300);
                        redirectMessage.setDelayed(10);
                        sendLog2Listeners(LogLevel.WARN,
                                "After broken, retry cluster manager " + retriedTimes + " out of " + retriedTimes);
                        if (server == null)
                            server = new ServerProxy("retry", config);
                        if (server.setResponse(redirectMessage) != true) {
                            sendLog2Listeners(LogLevel.ERROR, "Fail to retry");
                            throw new RuntimeException(e);
                        }
                    } else {
                        throw new RuntimeException(e);
                    }
                } else {
                    sendLog2Listeners(LogLevel.WARN, "client interrupted for quitting");
                }
            }
        }
        status = Status.STOP;
        try {
            resetServer();   //called only in this place
            sendLog2Listeners(LogLevel.INFO, "Out of the thread loop");
            if (binlogfile != null) {
                binlogfile.close();
                sendLog2Listeners(LogLevel.INFO, "Closed log file");
            }
        } catch (Exception e) {
            System.err.print("Close local file failed");
        }
    }

    private void persistCheckpoint() throws IOException {
        if (recordCheckpointTime >= checkpointPeriod) {
            final StringBuilder pb = new StringBuilder();
            pb.append(DRCConfig.POSITION_INFO);
            pb.append(config.getCheckpoint().toString());
            if (binlogfile != null)
                binlogfile.writeLine(pb.toString());
            recordCheckpointTime = 0;
        } else {
            recordCheckpointTime++;
        }
    }

    private void notifyDataMessage(Message message) {
        DataMessage dataMessage = (DataMessage) message;
        for (Listener listener : listeners) {
            try {
                long t_prev = System.currentTimeMillis();
                //the ip and port information is set only for DataMessage which will be notified to users
                //the string is put to the message's attribute --- SOURCEIPANDPORT.
                if (cache == null) {
                	dataMessage.addAttribute(Message.SOURCEIPANDPORT, storeIpAndPort);
                    listener.notify(dataMessage);
                } else {
                    if (cache.isReady()) {
                    	//for cached message, restore the data message, so we should set the ip and port again
                    	DataMessage cachedMessage = cache.getReadyRecords();
                    	cachedMessage.addAttribute(Message.SOURCEIPANDPORT, storeIpAndPort);
                        listener.notify(cachedMessage);
                    }
                }
                long t_after = System.currentTimeMillis();
                if (t_after - lastNotifiedTime > notifiedPeriod * 1000) {
                    listener.notifyRuntimeLog("Info", "notify consume: " + (double) accumulatedTime / (double) count + " ms");
                    lastNotifiedTime = t_after;
                    accumulatedTime = 0;
                    count = 0;
                } else {
                    accumulatedTime += (t_after - t_prev);
                    count++;
                }
            } catch (Exception e) {
                listener.handleException(e);
            }
        }
    }

    private void processDataMessage(Message message) throws IOException {
        DataMessage dataMessage = (DataMessage) message;
        Iterator<DataMessage.Record> rit = dataMessage.getRecordList().iterator();
        while (rit.hasNext()) {
            DataMessage.Record r = rit.next();
            processColumnFilter(r);
            Checkpoint checkpoint = config.getCheckpoint();
            switch (r.getOpt()) {
                case HEARTBEAT:
                    if (++accumulatedHeartbeats < oneHbEverySeconds) {
                        rit.remove();
                    } else {
                        final String tm = r.getTimestamp();
                        if (checkpoint.getTimestamp() != null && !checkpoint.getTimestamp().equals(tm))
                            if (retriedTimes != 0)
                                retriedTimes = 0;
                        checkpoint.setTimestamp(tm);
                        accumulatedHeartbeats = 0;
                        if (cache != null)
                            cache.addRecord(r);
                    }
                    break;
                case BEGIN:
                case COMMIT:
                    final String pos = r.getCheckpoint();
                    if (checkpoint.equals(pos))
                        if (retriedTimes != 0)
                            retriedTimes = 0;
                    config.setInstance(r.getServerId());
                    if (cache != null)
                        cache.addRecord(r);
                    break;
                default:
                    final String pos1 = r.getCheckpoint();
                    if (!checkpoint.equals(pos1))
                        if (retriedTimes != 0)
                            retriedTimes = 0;
                    config.setInstance(r.getServerId());
                    if (cache != null)
                        cache.addRecord(r);
                    break;
            }
        }
    }

    private void processColumnFilter(DataMessage.Record record) {
    	if(filter.getIsAllMatch() == false) {
	        List<String> s = DataFilterUtil.getColNamesWithMapping(record.getDbname(), record.getTablename(), filter);
	        if (s != null) {
	            List<DataMessage.Record.Field> l = record.getFieldList();
	            Iterator<DataMessage.Record.Field> it = l.iterator();
	            while (it.hasNext()) {
	                DataMessage.Record.Field f = it.next();
	                if (!DataFilterUtil.isColInArray(f.getFieldname(), s))
	                    it.remove();
	            }
	        }
    	}
    }

    private void processRedirectMessage(Message message) throws Exception {
        sendLog2Listeners(LogLevel.INFO, "Redirect to taskName: " + config.getDbname());
        RedirectMessage redirectMessage = (RedirectMessage) message;
        if (redirectMessage.getDelayed() > 0)
            TimeUnit.SECONDS.sleep(redirectMessage.getDelayed());
        connectClusterManager(config.getClusterManagerAddresses());
    }

    /**
     * Send request to daemon server and get data continuously. The method will
     * not be blocked after the thread starts.
     *
     * @return The started service thread.
     * @throws DRCClientException DRCClient internal exception.
     * @throws IOException
     */
    @Override
    public synchronized Thread startService() throws Exception {
        if (status != Status.INIT) {
            throw new Exception("server state is error,current state:" + status);
        }
        connectClusterManager(config.getClusterManagerAddresses());
        sendLog2Listeners(LogLevel.INFO, "start service successfully");
        clearParams();
        serviceThread = new Thread(this);
        serviceThread.start();
        sendLog2Listeners(LogLevel.INFO, "getting data from daemon server");
        status = Status.START;
        return serviceThread;
    }

    private void clearParams() {
        accumulatedHeartbeats = 0;
        retriedTimes = 0;
        accumulatedTime = 0;
        count = 0;
        cache = null;
        checkpointPeriod = 0;
        recordCheckpointTime = 0;
    }

    /**
     * Closed active connections and set the status as not quited.
     *
     * @throws Exception
     */
    @Override
    synchronized public void resetService() throws Exception {
        if (status != Status.STOP) {
            throw new Exception("server state is error,current state:" + status);
        }
        setQuit(false);
        status = Status.INIT;
    }

    /**
     * Stop the service (thread), after stopped, users need initialize the
     * service again then the service can be started. It seems that the status
     * STOPPED is the same with UNAVAILABLE, but I leave it here for further
     * extension (e.g. restart without calling initService.
     */
    @Override
    public synchronized void stopService() throws Exception {
        if (status != Status.START) {
            throw new Exception("server state is error,current state:" + status);
        }
        sendLog2Listeners(LogLevel.INFO, "Set quit true");
        setQuit(true);
        sendLog2Listeners(LogLevel.INFO, "Judge if thread is null");
        if (serviceThread != null && serviceThread.isAlive() &&
                serviceThread != Thread.currentThread()) {

            sendLog2Listeners(LogLevel.INFO, "Wait for the thread stopped");
            serviceThread.interrupt();
            serviceThread.join();
        }
        status = Status.STOP;
        storeIpAndPort = null;
        sendLog2Listeners(LogLevel.INFO, "Service stopped.");
    }

    /**
     * Add listeners.
     */
    @Override
    public final void addListener(Listener listener) {
        listeners.add(listener);
    }

    @Override
    public String getInstance() {
        return config.getInstance();
    }

    @Override
    public void trimLongType() {
        config.trimLongType();
    }

    private final Checkpoint getLocalInfo(final String filename)
            throws IOException, DRCClientException {

        Checkpoint checkpoint = null;
        LocalityFile localfile = new LocalityFile(filename, 0, 0);
        String line = null, location = null;
        while ((line = localfile.readLine()) != null) {
            if (isLocationInfo(line)) {
                location = extractLocationInfo(line);
            }
        }
        if (location == null || location.isEmpty()) {
            throw new DRCClientException
                    ("Local file " + filename + " not exists or is empty");
        }

        checkpoint = new Checkpoint();
        final String items[] = location.split(":");
        checkpoint.setServerId(items[1] + "-" + items[2]);
        checkpoint.setPosition(items[4] + "@" + items[3]);
        checkpoint.setTimestamp(items[5]);
        if (items.length > 6)
            checkpoint.setRecordId(items[6]);
        return checkpoint;
    }

    private static boolean isLocationInfo(final String line) {

        if (line.contains(DRCConfig.POSITION_INFO)) {
            return true;
        } else {
            return false;
        }
    }

    private static final String extractLocationInfo(final String string) {
        if (string == null) {
            return null;
        }
        String[] items = string.split(" ");
        return items[2];
    }

    @Override
    public final String getDbName() {
        return config.getDbname();
    }

    @Deprecated
    @Override
    public void addDataFilter(DataFilter filter) {
        this.filter = filter;
    }

    @Override
    public final void addDataFilter(DataFilterBase filter) {
        this.filter = filter;
    }

    @Deprecated
    public final void addWhereFilter(DataFilterBase filter) {
        config.setWhereFilters(filter.toString());
    }

    @Deprecated
    public final void useStrictFilter() {
        config.setFilterUnchangedRecords();
    }

    @Override
    public String getTaskName() {
        return "0";
    }

    /**
     * Set the frequency of heartbeat records received, one heartbeat every seconds.
     *
     * @param everySeconds after how many seconds, has been sent one heartbeat.
     */
    @Override
    public void setHeartbeatFrequency(int everySeconds) {
        oneHbEverySeconds = everySeconds;
    }


    /**
     * Simple util to send a string to all listeners.
     *
     * @param msg is the logged data.
     */
    private void sendLog2Listeners(LogLevel level, final String msg) {

        for (Listener listener : listeners) {
            try {
                switch (level) {
                    case ERROR:
                        listener.notifyRuntimeLog("ERROR", msg);
                        break;
                    case WARN:
                        listener.notifyRuntimeLog("WARN", msg);
                        break;
                    case INFO:
                        listener.notifyRuntimeLog("INFO", msg);
                        break;
                    default:
                        break;
                }
            } catch (Exception e) {
                listener.handleException(e);
            }
        }
    }

    private void setQuit(boolean isQuit) {
        quit = isQuit;
    }

    private boolean isQuit() {
        return quit;
    }

    /**
     * Set whether requiring record type begin/commit.
     *
     * @param required ture if required, false otherwise, default is true.
     */
    public void requireTxnMark(boolean required) {
        config.requireTxnMark(required);
    }

    @Override
    public void setNotifyRuntimePeriodInSec(long sec) {
        notifiedPeriod = sec;
    }

    @Override
    public void setNumOfRecordsPerBatch(int threshold) {
        config.setNumOfRecordsPerBatch(threshold);
    }

    @Override
    public DBType getDatabaseType()
            throws DRCClientException {

        ClusterManagers managers = new ClusterManagers(config.getClusterManagerAddresses());
        DBType type;
        try {
            type = managers.getDatabaseType(config);
        } catch (UnknownHostException ue) {
            throw new DRCClientException(ue.getMessage());
        } catch (MalformedURLException me) {
            throw new DRCClientException(me.getMessage());
        }

        return type;
    }

    @Override
    public void setGroup(String group) {
        config.setGroup(group);
    }

    @Override
    public void setSubGroup(String subGroup) {
        config.setSubGroup(subGroup);
    }

    @Override
    public void setDrcMark(String mark) {
        config.setDRCMark(mark);
    }

    @Override
    public void askSelfUnit() {
        config.useDrcMark();
    }

    public void setBlackList(String blackList) {
        this.blackList = blackList;
    }

    @Override
    public void suspend() {
        suspend = true;
    }

    @Override
    public void resume() {
        suspend = false;
    }

    @Override
    public void usePublicIp() {
        config.usePublicIp();
    }

    public void useCaseSensitive() {
        config.useCaseSensitive();
    }

    @Override
    public void useCRC32Check() {
        config.setUseCheckCRC(true);
    }

    @Override
    public DBType getDBType() {
        return this.dbType;
    }

    public void setDbType(DBType dbType) {
        this.dbType = dbType;
    }

    public boolean validateDataFilter() throws DRCClientException{
        return this.filter.validateFilter(this.dbType);
    }
}
