package com.aliyun.drc.client.impl;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.params.CoreConnectionPNames;

import com.aliyun.drc.client.DRCClientException;
import com.aliyun.drc.client.HttpBadResponseException;
import com.aliyun.drc.client.message.Builder;
import com.aliyun.drc.client.message.Message;

import javax.net.ssl.SSLContext;

/**
 * HTTP operations handler. Currently only support POST request.
 */
public class HttpHandler {

    /* Basic http url not including parameters after ? */
    private final String url;

    /* Url encoding, can be changed. */
    private String urlEncoding = "UTF-8";

    /* Post form parameters p1=a&p2=b */
    private final List<NameValuePair> formParams;

    /* Core Java class. */
    private HttpClient httpClient;

    /* Several kinds of types of httpd returned results.*/
    private HttpResponse httpResponse;
    private DataInputStream inStream;
    private Message message;
    private Builder builder;
    private String protocolVersion;
    private int socketTimeout = CONNECTION_TIMEOUT;
    private int connectionTimeout = CONNECTION_TIMEOUT;

    /* 
     * Cache for receiving stream from the socket,
     * results are cached in byte[].
     */
    private final static int BUFFSIZE = 128 * 1024; /* 128K */
    private final byte[] buffer = new byte[BUFFSIZE];
    private final static int CONNECTION_TIMEOUT = 120; /* 2 minute */

    HttpHandler(final String urlString, boolean useHTTPS) {
        url = urlString;
        formParams = new ArrayList<NameValuePair>();
        builder = new Builder();
        protocolVersion = "2.0";
        if (useHTTPS) {
            initHTTPSClient();
        } else {
            httpClient = new DefaultHttpClient();
        }
    }

    private void initHTTPSClient() {
        RegistryBuilder<ConnectionSocketFactory> registryBuilder = RegistryBuilder.<ConnectionSocketFactory>create();
        ConnectionSocketFactory plainSF = new PlainConnectionSocketFactory();
        registryBuilder.register("http", plainSF);
        try {
            KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
            TrustStrategy anyTrustStrategy = new TrustStrategy() {
                @Override
                public boolean isTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
                    return true;
                }
            };
            SSLContext sslContext = SSLContexts.custom().useTLS().loadTrustMaterial(trustStore, anyTrustStrategy)
                    .build();
            LayeredConnectionSocketFactory sslSF = new SSLConnectionSocketFactory(sslContext,
                    SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
            registryBuilder.register("https", sslSF);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        Registry<ConnectionSocketFactory> registry = registryBuilder.build();
        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(registry);
        httpClient = HttpClientBuilder.create().setConnectionManager(connManager).build();
    }

    final String getProtocolVersion() {
        return protocolVersion;
    }

    final String getUrl() {
        return url;
    }

    final void addFormParam(final String param, final String value) {
        formParams.add(new BasicNameValuePair(param, value));
    }

    final List<NameValuePair> getFormParams() {
        return formParams;
    }

    final void setUrlEncoding(final String encoding) {
        urlEncoding = encoding;
    }

    final String getUrlEncoding() {
        return urlEncoding;
    }

    void setSocketTimeout(int timeout) {
        socketTimeout = timeout;
    }

    void setConnectionTimeout(int timeout) {
        connectionTimeout = timeout;
    }

    private void sendRequest(StringEntity entity)
            throws ClientProtocolException, IOException, HttpBadResponseException {
        HttpPost method = new HttpPost(url);
        method.getParams().setParameter(
                CoreConnectionPNames.SO_TIMEOUT, socketTimeout * 1000);
        method.getParams().setParameter
                (CoreConnectionPNames.CONNECTION_TIMEOUT, connectionTimeout * 1000);
        method.setEntity(entity);
        httpResponse = httpClient.execute(method);
        checkResponse(httpResponse);
    }

    void sendEncodedRequest() throws IOException, HttpBadResponseException {

        sendRequest(new UrlEncodedFormEntity(formParams, urlEncoding));
    }


    /**
     * Send HTTP request, only for POST. Response is not consumed, but only be
     * checked. The response can be accessed in different formats.
     */
    void sendPlainRequest()
            throws IOException, HttpBadResponseException {

        /* Do not use url encoded forms, send ; and @ directly. */
        // UrlEncodedFormEntity entity =
        // new UrlEncodedFormEntity(formParams, urlEncoding);

        StringBuilder builder = new StringBuilder();
        for (NameValuePair pair : formParams) {
            builder.append(pair.getName() + "=" + pair.getValue());
            builder.append("&");
        }

        /* Remove the last redundant & to avoid server crash. */
        if (builder.length() != 0)
            builder.deleteCharAt(builder.length() - 1);
        sendRequest(new StringEntity(builder.toString()));
    }

    /**
     * Get one complete HTTP response.
     */
    HttpResponse recvResponse() {
        return httpResponse;
    }

    /**
     * Get the input stream of the response.
     *
     * @return Input stream
     * @throws IOException
     * @throws DRCClientException
     */
    InputStream recvInputStream()
            throws IOException, DRCClientException {

        if (inStream == null) {
            if (httpResponse == null) {
                throw new DRCClientException
                        ("HttpHanlder can not find http response and inputStream.");
            }

            HttpEntity responsedEntity = httpResponse.getEntity();
            if (responsedEntity != null) {
                inStream = new DataInputStream(responsedEntity.getContent());
            }

        }

        return inStream;
    }

    /**
     * Get one complete DRC-P package which is defined and generated by
     * protobuf.
     */
    @SuppressWarnings("deprecation")
    Message recvDRCPResponse(DRCConfig drcConfig)
            throws IOException, DRCClientException {

        if (inStream == null) {
            if (httpResponse == null) {
                throw new DRCClientException
                        ("HttpHanlder can not find http response and inputStream.");
            }

            HttpEntity responsedEntity = httpResponse.getEntity();
            if (responsedEntity != null) {
                inStream = new DataInputStream(responsedEntity.getContent());
                protocolVersion = inStream.readLine();
            }
        }

        /*
         * Every time only get one complete message, if incomplete, wait, if
         * too many, return and leave remaining in the stream buffer.
         */
        message = builder.build(inStream,drcConfig);
        return message;
    }

    /**
     * Get one complete DRC-P package which is defined and generated by
     * protobuf.
     */
    Message recvDRCPBinaryResponse(DRCConfig drcConfig)
            throws Exception, DRCClientException {

        if (inStream == null) {
            if (httpResponse == null) {
                throw new DRCClientException
                        ("HttpHanlder can not find http response and inputStream.");
            }

            HttpEntity responsedEntity = httpResponse.getEntity();
            if (responsedEntity != null) {
                inStream = new DataInputStream(responsedEntity.getContent());
            }
        }

        /*
         * Every time only get one complete message, if incomplete, wait, if
         * too many, return and leave remaining in the stream buffer.
         */
        try {
            message = builder.buildBinaryBinlog(inStream, drcConfig);
        } catch (Exception e) {
            // get another 1024 buff to echo
            byte []buff = new byte[1024];
            int read_effective_count = inStream.read(buff, 0, 1024);
            Exception toThrow = new Exception("raw buf:" + new String(buff, 0, read_effective_count), e.getCause());
            toThrow.setStackTrace(e.getStackTrace());
            throw toThrow;
        }
        return message;
    }

    /**
     * Get one complete DRC-P package which is defined and generated by
     * drcnet
     * @throws Exception
     */
    Message recvDRCNetBinaryResponse(byte buff[],DRCConfig drcConfig)
            throws Exception {

        message = builder.buildDRCNetBinaryBinlog(buff,drcConfig);
        return message;
    }

    /**
     * Get one complete DRC-P package which is defined and generated by
     * drcnet.
     * @throws Exception
     */
    Message recvDRCNetTextResponse(byte buff[],DRCConfig drcConfig)
            throws Exception {

        message = builder.buildDRCNetTextBinlog(buff,drcConfig);
        return message;
    }

    /**
     * Receive a block of bytes from the long-term http link, will block if no
     * response.
     *
     * @return the length of returned string is less than or equal to BuffSize.
     */
    byte[] recv() throws IOException, DRCClientException {
        if (inStream == null) {
            if (httpResponse == null) {
                throw new DRCClientException
                        ("HttpHanlder can not find http response and inputStream.");
            }

            HttpEntity responsedEntity = httpResponse.getEntity();
            if (responsedEntity != null) {
                inStream = new DataInputStream(responsedEntity.getContent());
            }
        }

        int readBytes;
        if ((readBytes = inStream.read(buffer, 0, BUFFSIZE)) != -1) {
            byte[] data = new byte[readBytes];
            System.arraycopy(buffer, 0, data, 0, readBytes);
            return data;
        } else {
            return null;
        }
    }

    /**
     * Release occupied resource. No matter stopped normally or exceptions are
     * thrown, close should be called.
     */
    final void close() {
        if (inStream != null && httpClient != null) {
            httpClient.getConnectionManager().shutdown();
        }
    }

    /**
     * Check response code, if not 200 throw exceptions.
     */
    void checkResponse(final HttpResponse response)
            throws HttpBadResponseException, IOException {

        int statusCode = response.getStatusLine().getStatusCode();
        if (statusCode != 200) {
            switch (statusCode) {
                case 201:
                case 202:
                case 203:
                case 204:
                case 205:
                case 206:
                case 207:
                case 208:
                case 209:
                case 210:
                case 211:
                case 400:
                    InputStream in = response.getEntity().getContent();
                    BufferedReader reader = new BufferedReader(new InputStreamReader(in));
                    String line = null;
                    StringBuilder builder = new StringBuilder();
                    while ((line = reader.readLine()) != null) {
                        builder.append(line);
                    }
                    throw new HttpBadResponseException
                            (statusCode, "HTTP response code: " + statusCode + " " + builder.toString());
                case 403:
                    throw new HttpBadResponseException
                            (statusCode, "HTTP response code: " + statusCode + " Forbidden.");
                case 404:
                    throw new HttpBadResponseException
                            (statusCode, "HTTP response code: " + statusCode + " Not Found.");
                case 500:
                    throw new HttpBadResponseException
                            (statusCode, "HTTP response code: " + statusCode + " Internal Error.");
                default:
                    throw new HttpBadResponseException
                            (statusCode, "HTTP response code: " + statusCode + " Unexpected code.");
            }
        }
    }
}
