package com.aliyun.drc.client.message;

import java.io.DataInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class RedirectMessage extends Message {
    
    private String url;
    private int delayed=10;
    private String version;
    private final List<String> persistent;
    private final List<String> requires;
    private final List<String> optional;
    private final Map<String, String> userParameters;
    private static final String delimiter = " ";

    /**
     * Constructor of RedirectMessage.
     */
    public RedirectMessage() {
        super();
        delayed = 0;
        version = new String("2.0"); // default version.
        persistent = new ArrayList<String>();
        requires = new ArrayList<String>();
        optional = new ArrayList<String>();
        userParameters = new HashMap<String, String>();
    }

    /**
     * Set the redirect url.
     * @param url the redirect url.
     */
    public void setUrl(final String url) {
        this.url = url;
    }

    /**
     * Get the redirect url.
     * @return the redirect url.
     */
    public String getUrl() {
        return url;
    }

    /**
     * Set the delayed seconds.
     * @param delayed is the delayed seconds.
     */
    public void setDelayed(final int delayed) {
        this.delayed = delayed;
    }

    /**
     * Get the delayed seconds.
     * @return the delayed seconds.
     */
    public int getDelayed() {
    	final String delay = userParameters.get("delay");
    	if (delay != null && !delay.isEmpty()) {
    		delayed = Integer.parseInt(delay);
    	}
        return delayed;
    }

    /**
     * Add requires.
     * @param require is the required parameters.
     */
    public void addRequire(final String require) {
        requires.add(require);
    }

    /**
     * Get persistent-required attributes' names.
     * @return
     */
    public List<String> getPersists()
    {
    	return persistent;
    }
    
    /**
     * Get required parameters.
     * @return teh requried parameters.
     */
    public List<String> getRequires() {
        return requires;
    }

    /**
     * Add optional parameters.
     * @param optionalParameter is the optional parameter's name.
     */
    public void addOptional(final String optionalParameter) {
        optional.add(optionalParameter);
    }

    /**
     * Get all optional parameters.
     * @return optional parameters.
     */
    public List<String> getOptional() {
        return optional;
    }

    /**
     * Add extra user parameters.
     * @param name is the name of the new parameter.
     * @param value is the value of the new parameter.
     */
    public void addUserParameters(final String name, final String value) {
        userParameters.put(name, value);
    }

    /**
     * Get all new user parameters.
     * @return the user parameters.
     */
    public Map<String, String> getUserParameters() {
        return userParameters;
    }

    /**
     * Get version of the protocol.
     * @return the version of the protocol.
     */
    public String getVersion() {
        return version;
    }

    /**
     * Construct a RedirectMessage from DataInputStream.
     * @param in is the input stream.
     * @throws IOException.
     */
    @SuppressWarnings("deprecation")
    public void mergeFrom(final DataInputStream in) throws IOException {
        String line = in.readLine();
        String[] params = line.split(delimiter);
        if (params.length < 1) {
            throw new IOException("Parse monitor message " + line + " error");
        }
        
        url = params[0];
        for (int i = 1; i < params.length; i++) {
            handleRedirecMessageItem(params[i]);
        }
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append(super.toString());
        builder.append(url + delimiter);
        for (String req : requires) {
            builder.append(req + ":");
            final String value = userParameters.get(req);
            if (value != null) {
                builder.append(value);
            }
            builder.append(",required" + delimiter);
        }

        for (String opt : optional) {
            builder.append(opt + ":");
            final String value = userParameters.get(opt);
            if (value != null) {
                builder.append(value);
            }
            builder.append(",optional" + delimiter);
        }
        builder.append(System.getProperty("line.separator"));
        return builder.toString();
    }
    /**
     * Handle defined structure argument_name:[argument_value][",required"].
     * @param item is the string with defined structure.
     */
    private void handleRedirecMessageItem(final String item) {
        final String[] pair = item.split(":", -1);
        final String name = pair[0];
        final String params = pair[1];
        if (params.isEmpty()) {
            optional.add(name);
            return;
        }
        final String[] attrs = params.split(",", -1);
        final String value = attrs[0];
        if (!value.isEmpty()) {
            userParameters.put(name, value);
        }
        if (attrs.length <= 1) {
        	return;
        }
        final String require = attrs[1];
        if (require.equals("persistent")) {
        	persistent.add(name);
    	} else if (require.equals("required") ) {
            requires.add(name);
        } else if (require.equals("optional")){
            optional.add(name);
        }
    }
}