package com.aliyun.drc.client.message.drcmessage;

import com.aliyun.drc.client.message.DataMessage;
import com.aliyun.drc.client.enums.DBType;
import com.aliyun.drc.client.impl.DRCClientRunTimeException;
import com.aliyun.drc.client.message.ByteString;
import com.aliyun.drc.utils.BinaryMessageUtils;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;

import java.io.*;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.CRC32;

public class BinlogRecord extends DataMessage.Record {

    private static final String DEFAULT_ENCODING = "ASCII";
    private static final String SEP = System.getProperty("line.separator");

    //old version header length
    private static final int OLD_VERSION_2_HEADER_LEN = 88;

    //new version header length
    private static final int NEW_VERSION_2_HEADER_LEN = 96;

    //v3 header length
    private static final int VERSION_3_HEADER_LEN = 104;

    private static final int PREFIX_LENGTH = 12;

    private int m_brVersion = (byte) 0xff;
    private int m_srcType = (byte) 0xff;
    private int m_op = (byte) 0xff;
    private int m_lastInLogevent = (byte) 0xff;
    private long m_srcCategory = MAX_LONG;
    private long m_id = MAX_LONG;
    private long m_timestamp = MAX_LONG;

    private long m_encoding = MAX_LONG;
    private long m_instanceOffset = MAX_LONG;
    private long m_timemarkOffset = MAX_LONG;
    private long m_dbNameOffset = MAX_LONG;
    private long m_tbNameOffset = MAX_LONG;
    private long m_colNamesOffset = MAX_LONG;
    private long m_colTypesOffset = MAX_LONG;
    private long m_fileNameOffset = MAX_LONG;
    private long m_fileOffset = MAX_LONG;
    private long m_oldColsOffset = MAX_LONG;
    private long m_newColsOffset = MAX_LONG;
    private long m_pkValOffset = MAX_LONG;
    private long m_pkKeysOffset = MAX_LONG;
    private long m_ukColsOffset = MAX_LONG;
    private long m_filterRuleValOffset = MAX_LONG;
    private long m_tailOffset = MAX_LONG;


    /* version 2 */
    private long m_colsEncsOffset = MAX_LONG;

    private VarAreaMessage message=new VarAreaMessage();

    private static final long MAX_LONG = 0xFFFFFFFFFFFFFFFFL;

    /**
     * type bitmap,get array type bytes by type index
     * array first byte : 1,  array element is unsigned byte
     * 2,  array element is unsigned short
     * 4,  array element is unsigned int
     * 8,  array element is long
     */
    private final static int[] elementArray = {0, 1, 1, 2, 2, 4, 4, 8, 8};

    private final static int BYTE_SIZE = 1;

    private final static int INT_SIZE = Integer.SIZE / Byte.SIZE;

    private final  CRC32 crc32 = new CRC32();

    private boolean isCheckCRC=false;

    public BinlogRecord(boolean crcCheck){
        this.isCheckCRC=crcCheck;

    }

    @Override
    public byte[] getRawData() {
        return message.getRawData();
    }

    @Override
    public void parse(final byte[] data) throws IOException {
        DataInputStream is = new DataInputStream(new ByteArrayInputStream(data));
        this.parse(is);
    }

    public void parse(final DataInputStream is) throws IOException {
        message.parse(is);
        List<Integer> bytes = message.getArray(0);
        byte[] bytesArray = new byte[bytes.size()];
        for (int i = 0; i < bytes.size(); i++)
            bytesArray[i] = bytes.get(i).byteValue();
        boolean idOldVersion2;
        switch (bytes.size()) {
            case OLD_VERSION_2_HEADER_LEN:
                idOldVersion2 = true;
                break;
            case VERSION_3_HEADER_LEN:
            case NEW_VERSION_2_HEADER_LEN:
                idOldVersion2 = false;
                break;
            default:
                throw new IOException("Parsed bytes header size not recogonized: " + bytes.size());
        }

        DataInputStream hs = new DataInputStream(new ByteArrayInputStream(bytesArray));
        m_brVersion = hs.readUnsignedByte();
        m_srcType = hs.readUnsignedByte();
        m_op = hs.readUnsignedByte();
        m_lastInLogevent = hs.readUnsignedByte();
        m_srcCategory = CIOUtil.readUnsignedInt(hs);
        m_id = CIOUtil.readLong(hs);
        m_timestamp = CIOUtil.readLong(hs);
        m_encoding = CIOUtil.readUnsignedInt(hs);
        m_instanceOffset = CIOUtil.readUnsignedInt(hs);
        m_timemarkOffset = CIOUtil.readUnsignedInt(hs);
        m_dbNameOffset = CIOUtil.readUnsignedInt(hs);
        m_tbNameOffset = CIOUtil.readUnsignedInt(hs);
        m_colNamesOffset = CIOUtil.readUnsignedInt(hs);
        m_colTypesOffset = CIOUtil.readUnsignedInt(hs);

        //process old version
        if (idOldVersion2 == false) {
            m_pkValOffset = CIOUtil.readUnsignedInt(hs);
            m_fileNameOffset = CIOUtil.readLong(hs);
            m_fileOffset = CIOUtil.readLong(hs);
            if (m_fileNameOffset < -1 || m_fileOffset < -1) {
                throw new IOException("f: " + m_fileNameOffset + " and o: " +
                        m_fileOffset + " should both be signed integer");
            }
            m_oldColsOffset = CIOUtil.readUnsignedInt(hs);
            m_newColsOffset = CIOUtil.readUnsignedInt(hs);
        } else {
            //process new version
            m_fileNameOffset = CIOUtil.readUnsignedInt(hs);
            m_fileOffset = CIOUtil.readUnsignedInt(hs);
            m_oldColsOffset = CIOUtil.readUnsignedInt(hs);
            m_newColsOffset = CIOUtil.readUnsignedInt(hs);
            m_pkValOffset = CIOUtil.readUnsignedInt(hs);
        }

        m_pkKeysOffset = CIOUtil.readUnsignedInt(hs);
        m_ukColsOffset = CIOUtil.readUnsignedInt(hs);

        if (m_brVersion > 1) {
            m_colsEncsOffset = CIOUtil.readLong(hs);
        }

        //version 3
        if (m_brVersion == 3) {
            m_filterRuleValOffset = CIOUtil.readUnsignedInt(hs);
            m_tailOffset = CIOUtil.readUnsignedInt(hs);
        }

        //timestamp,process heartbeat between tx
        timestamp = Long.toString(m_timestamp);
        Type type = Type.valueOf(m_op);
        if(getDbType()==DBType.OCEANBASE1){
            gloalSafeTimestamp.set(String.valueOf(m_fileNameOffset));
        }else {
            if (type == Type.BEGIN) {
                gloalSafeTimestamp.set(timestamp);
                txEnd.set(false);
            }
            if (txEnd.get() == true) {
                gloalSafeTimestamp.set(timestamp);
            }
            //set txEnd
            if (type == Type.COMMIT || type == Type.ROLLBACK) {
                txEnd.set(true);
            }
        }
        safeTimestamp=new String(gloalSafeTimestamp.get());
        if(isCheckCRC) {
            checkCRC();
        }
    }

    private void checkCRC() throws IOException {
        long value = getCRCValue();
        if (value == 0l) {
            return;
        }
        crc32.update(getRawData(), 0, getRawData().length - 4);
        long actual = crc32.getValue();
        crc32.reset();
        if (value != actual) {
            throw new IOException("crc 32 check failed,expect:"+value+",actual:"+actual);
        }
    }

    public int getVersion() {
        return m_brVersion;
    }

    @Override
    public DBType getDbType() {
        switch (m_srcType) {
            case 0:
                return DBType.MYSQL;
            case 1:
                return DBType.OCEANBASE;
            case 2:
                return DBType.HBASE;
            case 3:
                return DBType.ORACLE;
            case 4:
                return DBType.OCEANBASE1;
            default:
                return DBType.UNKNOWN;
        }
    }

    @Override
    public boolean isQueryBack() {
        switch ((int) m_srcCategory) {
            case 0:
                return false;
            case 1:
                return true;
            case 2:
                return false;
            case 3:
                return false;
            default:
                return false;
        }
    }

    @Override
    public boolean isFirstInLogevent() {
        return (m_lastInLogevent == 1) ? true : false;
    }

    @Override
    public void mergeFrom(final DataInputStream is) throws IOException {
        parse(is);
    }

    @Override
    public Type getOpt() {
        return Type.valueOf(m_op);
    }

    @Override
    public String getId() {
        return Long.toString(m_id);
    }

    @Override
    public String getDbname() {
        try {
            if ((int) m_dbNameOffset < 0) {
                return null;
            } else {
                return message.getString((int) m_dbNameOffset, DEFAULT_ENCODING);
            }
        } catch (IOException e) {
            throw new DRCClientRunTimeException(e.getMessage(), e.getCause());
        }
    }

    @Override
    public final String getTablename() {
        try {
            if ((int) m_tbNameOffset < 0)
                return null;
            else
                return message.getString((int) m_tbNameOffset, DEFAULT_ENCODING);
        } catch (IOException e) {
            throw new DRCClientRunTimeException(e.getMessage(), e.getCause());
        }
    }

    @Override
    public String getCheckpoint() {
        return m_fileOffset + "@" + m_fileNameOffset;
    }

    @Deprecated
    @Override
    public String getMetadataVersion() {
        return "0";
    }

    @Override
    public String getServerId() {
        try {
            if ((int) m_instanceOffset < 0)
                return null;
            else
                return message.getString((int) m_instanceOffset, DEFAULT_ENCODING);
        } catch (IOException e) {
            throw new DRCClientRunTimeException(e.getMessage(), e.getCause());
        }
    }

    public List<ByteString> getFirstPKValue() {
        try {
            if ((int) m_pkValOffset < 0)
                return null;
            else
                return message.getByteStringList((int) m_pkValOffset);
        } catch (IOException e) {
            throw new DRCClientRunTimeException(e.getMessage(), e.getCause());
        }
    }

    @Override
    public String getPrimaryKeys() {
        try {
            if ((int) m_pkKeysOffset < 0) {
                return "";
            } else {
                List<Integer> pks = message.getArray((int) m_pkKeysOffset);
                List<ByteString> names = message.getByteStringList(m_colNamesOffset);
                StringBuilder pkKeyName = new StringBuilder();
                if (pks != null) {
                    for (int idx : pks) {
                        if (pkKeyName.length() != 0)
                            pkKeyName.append(",");
                        pkKeyName.append(names.get(idx).toString(DEFAULT_ENCODING));
                    }
                }
                return pkKeyName.toString();
            }
        } catch (IOException e) {
            throw new DRCClientRunTimeException(e.getMessage(), e.getCause());
        }
    }

    public String getUniqueColNames() {
        try {
            if ((int) m_ukColsOffset < 0)
                return "";
            else {
                List<Integer> uks = message.getArray((int) m_ukColsOffset);
                List<ByteString> names = message.getByteStringList(m_colNamesOffset);
                StringBuilder ukKeyName = new StringBuilder();
                if (uks != null) {
                    for (int idx : uks) {
                        if (ukKeyName.length() != 0)
                            ukKeyName.append(",");
                        ukKeyName.append(names.get(idx).toString(DEFAULT_ENCODING));
                    }
                }
                return ukKeyName.toString();
            }
        } catch (IOException e) {
            throw new DRCClientRunTimeException(e.getMessage(), e.getCause());
        }
    }

    public int getFieldCount() {
        return getFieldList().size();
    }

    public List<Field> getFieldList() {
        try {
            if (fields == null) {
                if (m_colNamesOffset < 0 || m_colTypesOffset < 0 || m_oldColsOffset < 0 || m_newColsOffset < 0) {
                    return fields;
                }
                //global encoding
                String encodingStr = BinaryMessageUtils.getString(getRawData(), (int) m_encoding, DEFAULT_ENCODING);
                //pk info
                List<Integer> pks = null;
                if ((int) m_pkKeysOffset > 0) {
                    pks = BinaryMessageUtils.getArray(getRawData(), (int) m_pkKeysOffset);
                }
                //get column count
                ByteBuf wrapByteBuf = Unpooled.wrappedBuffer(getRawData()).order(ByteOrder.LITTLE_ENDIAN);
                wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_colNamesOffset + BYTE_SIZE));
                int count = wrapByteBuf.readInt();
                fields = new ArrayList<Field>(count);
                //op type array
                wrapByteBuf.readerIndex(PREFIX_LENGTH + (int) m_colTypesOffset);
                byte t = wrapByteBuf.readByte();
                int elementSize = elementArray[t & DataType.DT_MASK];
                //encoding
                int colEncodingsCount=0;
                int currentEncodingOffset=0;
                if(m_colsEncsOffset>0){
                    wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_colsEncsOffset + BYTE_SIZE));
                    colEncodingsCount = wrapByteBuf.readInt();
                    currentEncodingOffset = (int) wrapByteBuf.readUnsignedInt();
                }
                //column name
                wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_colNamesOffset + BYTE_SIZE + INT_SIZE));
                int currentColNameOffset = (int) wrapByteBuf.readUnsignedInt();
                //old col value
                wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_oldColsOffset + BYTE_SIZE));
                int oldColCount = wrapByteBuf.readInt();
                int currentOldColOffset = -1;
                //Bug fix: if newCol count or old Count is 0, then the following offset should not be read;
                if (0 != oldColCount) {
                    currentOldColOffset = (int) wrapByteBuf.readUnsignedInt();
                }
                //new col value
                wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_newColsOffset + BYTE_SIZE));
                int newColCount = wrapByteBuf.readInt();
                int currentNewColOffset = -1;
                if (0 != newColCount) {
                    currentNewColOffset = (int) wrapByteBuf.readUnsignedInt();
                }

                //start loop
                for (int i = 0; i < count; i++) {
                    //get pk boolean
                    boolean isPk = false;
                    if (pks != null && pks.contains(i)) {
                        isPk = true;
                    }
                    //get real op type
                    int type = 0;
                    wrapByteBuf.readerIndex(PREFIX_LENGTH + (int) m_colTypesOffset + BYTE_SIZE + INT_SIZE + i * elementSize);
                    switch (elementSize) {
                        case 1:
                            type = wrapByteBuf.readUnsignedByte();
                            break;
                        case 2:
                            type = wrapByteBuf.readUnsignedShort();
                            break;
                        case 4:
                            type = (int) wrapByteBuf.readUnsignedInt();
                            break;
                        case 8:
                            type = (int) wrapByteBuf.readLong();
                            break;
                    }
                    //get real encoding
                    String realEncoding = encodingStr;
                    wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_colsEncsOffset + BYTE_SIZE + INT_SIZE + (i + 1) * INT_SIZE));
                    if (colEncodingsCount > 0) {
                        int nextEncodingOffset = (int) wrapByteBuf.readUnsignedInt();
                        ByteString encodingByteString = new ByteString(wrapByteBuf.array(), PREFIX_LENGTH +
                                currentEncodingOffset + BYTE_SIZE + INT_SIZE + (count + 1) * INT_SIZE + (int) m_colsEncsOffset, nextEncodingOffset - currentEncodingOffset - 1);
                        realEncoding = encodingByteString.toString();
                        if (realEncoding.isEmpty()) {
                            if ((type == 253 || type == 254) && Field.MYSQL_TYPES[type] == Field.Type.STRING) {
                                realEncoding = "binary";
                            } else if (Field.MYSQL_TYPES[type] == Field.Type.BLOB) {
                                realEncoding = "";
                            } else if (type == 245) {
                                realEncoding= UTF8MB4_ENCODING;
                            } else {
                                realEncoding = DEFAULT_ENCODING;
                            }
                        }
                        currentEncodingOffset = nextEncodingOffset;
                    }
                    //type change from blob to string if encoding is not empty
                    //Bug: This line should like following : Field.MYSQL_TYPES[type] == Field.Type.BLOB
                    //But the active of java2object will changed for text, medium text and long text
                    //And consumers using hard cord and treat source column  have text type as byte array
                    //which should be string actually.
                    if (!realEncoding.isEmpty() && type == Field.Type.BLOB.ordinal()) {
                        type = 15;
                    }
                    //colName
                    wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_colNamesOffset + BYTE_SIZE + INT_SIZE + (i + 1) * INT_SIZE));
                    int nextColNameOffset = (int) wrapByteBuf.readUnsignedInt();
                    ByteString ColNameByteString = new ByteString(wrapByteBuf.array(), PREFIX_LENGTH +
                            currentColNameOffset + BYTE_SIZE + INT_SIZE + (count + 1) * INT_SIZE + (int) m_colNamesOffset, nextColNameOffset - currentColNameOffset - 1);
                    String columnName=ColNameByteString.toString();
                    currentColNameOffset = nextColNameOffset;
                    //old col
                    if (oldColCount != 0) {
                        wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_oldColsOffset + BYTE_SIZE + INT_SIZE + (i + 1) * INT_SIZE));
                        int nextOldColOffset = (int) wrapByteBuf.readUnsignedInt();
                        ByteString value = null;
                        if (nextOldColOffset != currentOldColOffset) {
                            value = new ByteString(wrapByteBuf.array(), PREFIX_LENGTH +
                                    currentOldColOffset + BYTE_SIZE + INT_SIZE + (count + 1) * INT_SIZE + (int) m_oldColsOffset, nextOldColOffset - currentOldColOffset - 1);
                        }
                        fields.add(new Field(columnName, type, realEncoding, value, isPk));
                        currentOldColOffset = nextOldColOffset;
                    }
                    //new col
                    if (newColCount != 0) {
                        wrapByteBuf.readerIndex((int) (PREFIX_LENGTH + m_newColsOffset + BYTE_SIZE + INT_SIZE + (i + 1) * INT_SIZE));
                        int nextNewColOffset = (int) wrapByteBuf.readUnsignedInt();
                        ByteString value = null;
                        if (currentNewColOffset != nextNewColOffset) {
                            value = new ByteString(wrapByteBuf.array(), PREFIX_LENGTH +
                                    currentNewColOffset + BYTE_SIZE + INT_SIZE + (count + 1) * INT_SIZE + (int) m_newColsOffset, nextNewColOffset - currentNewColOffset - 1);
                        }
                        fields.add(new Field(columnName, type, realEncoding, value, isPk));
                        currentNewColOffset = nextNewColOffset;
                    }
                }
            }

        } catch (Exception e) {
            fields = null;
            throw new DRCClientRunTimeException(e.getMessage(), e.getCause());
        }
        return fields;
    }


    public List<Long> getTimemarks() throws IOException {
//		System.out.println("offset is " + m_timemarkOffset);
        if (m_timemarkOffset != -1)
            return message.getArray((int) m_timemarkOffset);
        else
            return null;
    }

    public String getTraceId() throws IOException {
        List<ByteString> list = message.getByteStringList(m_filterRuleValOffset);
        if (list == null || list.size() == 0 || list.size() < 3) {
            return null;
        }
        ByteString traceId = list.get(2);
        return traceId == null ? null : traceId.toString();
    }

    private long getCRCValue() throws IOException {
        long crcValue = 0l;
        if (m_tailOffset == -1) {
            return 0l;
        }
        List<Integer> list = message.getArray((int) m_tailOffset);
        if (list == null || list.size() != 12) {
            return 0l;
        }
        crcValue += (long) list.get(8);
        crcValue += ((long) list.get(9)) << 8;
        crcValue += ((long) list.get(10)) << 16;
        crcValue += ((long) list.get(11)) << 24;
        return crcValue;
    }

    public String getThreadId() throws IOException {
        long threadId = 0l;
        if (m_tailOffset == -1) {
            return null;
        }
        List<Integer> list = message.getArray((int) m_tailOffset);
        if (list == null || list.size() == 0) {
            return null;
        }
        threadId += (long) list.get(0);
        threadId += ((long) list.get(1)) << 8;
        threadId += ((long) list.get(2)) << 16;
        threadId += ((long) list.get(3)) << 24;
        return String.valueOf(threadId);
    }

    private void writeObject(ObjectOutputStream out) throws IOException {

    }


    private void readObject(ObjectInputStream in) throws IOException,
            ClassNotFoundException {
        this.parse(new DataInputStream(in));
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();

        builder.append("type:" + getOpt()).append(SEP);
        builder.append("record_id:" + getId()).append(SEP);
        builder.append("db:" + getDbname()).append(SEP);
        builder.append("tb:" + getTablename()).append(SEP);
        builder.append("serverId:" + getServerId()).append(SEP);
        builder.append("checkpoint:" + getCheckpoint()).append(SEP);
        builder.append("primary_value:" + getPrimaryKeys()).append(SEP);
        builder.append("unique_keys:" + getUniqueColNames()).append(SEP);
        builder.append(SEP);
        getFieldList();
        if (null != fields) {
            for (Field field : fields) {
                builder.append(field.toString());
            }
        }
        builder.append(SEP);
        return builder.toString();
    }
}
