package com.aliyun.drc.client.message.drcmessage;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

import com.aliyun.drc.client.impl.DRCClientRunTimeException;
import com.aliyun.drc.client.message.ByteString;

public class VarAreaMessage {
	
	private boolean parsedOK;
	private VarAreaHeader header;
	private byte[] data;
	private int previousBytesLength;
	private int len;
	
	private static final long MAX_LONG = 0xFFFFFFFFFFFFFFFFL;

    private static final int MESSAGE_LENGTH_THRESHOLD=1024*1024*100;
	
	
	public VarAreaMessage() {
		parsedOK = false;
		header = null;
		data = null;
		previousBytesLength = 0;
		len = 0;
	}

	public byte[] getRawData() {
		return data;
	}

	public void parse(DataInputStream is) throws IOException {
		header = new VarAreaHeader();
		header.parseFrom(is);
		len = (int)header.getMsgHeader().getMessageSize();
		if (len <= 4||len>MESSAGE_LENGTH_THRESHOLD)
			throw new IOException("Header shows message size "  + len + " less than 4 or bigger then 100m");
		len -= 4;
		previousBytesLength = header.getRawData().length;
		data = new byte[len + previousBytesLength];
		for (int i = 0; i < previousBytesLength; i ++) {
			data[i] = header.getRawData()[i];
		}
		is.readFully(data, previousBytesLength, len);
		
		parsedOK = true;
	}
	
	public DataInputStream getDataInputStream() throws IOException {
		if (data == null)
			throw new DRCClientRunTimeException("data[] in VarAreaMessage is null");
	
		return new DataInputStream(new ByteArrayInputStream(data, previousBytesLength, len));
	}

	public String getString(int offset, final String encoding) throws IOException {

		if (!parsedOK)
			throw new DRCClientRunTimeException("Message not parsed");
		
		if (offset < 0)
			throw new DRCClientRunTimeException("String offset is negative " + offset);
		
		DataInputStream is = new DataInputStream(new ByteArrayInputStream(data, offset + previousBytesLength, data.length - offset - previousBytesLength));
		
		byte t = is.readByte();
		if ((t & DataType.DC_ARRAY) != 0) {
			throw new DRCClientRunTimeException("Get string failed, message data category array");
		}

		if ((t & DataType.DC_NULL) != 0) {
			return null;
		}
		
		return CIOUtil.readString(is, encoding);
	}

	public ByteString getByteString(int offset) throws IOException {

		if (!parsedOK)
			throw new DRCClientRunTimeException("Message not parsed");
		
		DataInputStream is = new DataInputStream(new ByteArrayInputStream(data, offset + previousBytesLength, data.length - offset - previousBytesLength));
		
		byte t = is.readByte();
		if ((t & DataType.DC_ARRAY) != 0) {
			throw new DRCClientRunTimeException("Get string failed, message data category array");
		}

		if ((t & DataType.DC_NULL) != 0) {
			return null;
		}
		
		return CIOUtil.readByteString(is);
	}

	public List<ByteString> getByteStringList(long offset) throws IOException {
		if (!parsedOK)
			throw new DRCClientRunTimeException("Message not parsed");
		
		if (offset == MAX_LONG)
			return null;
		
		if (offset < 0 || offset > data.length - previousBytesLength)
			throw new DRCClientRunTimeException("Binary message error: " + "data length: " + data.length +
					" previous length: " + previousBytesLength + " offset: " + offset);

		int readBytes = 0;
		ByteArrayInputStream bis = new ByteArrayInputStream
				(data, (int)offset + previousBytesLength, data.length - (int)offset - previousBytesLength);
		DataInputStream is = new DataInputStream(bis);
		
		byte t = is.readByte();
		readBytes ++;
		
		if ((t & DataType.DC_ARRAY) == 0 || (t & DataType.DT_MASK) != DataType.DT_STRING) {
			throw new DRCClientRunTimeException("Data type not array or not string");
		}
		
		long count = CIOUtil.readUnsignedInt(is);
		if (count == 0) {
			return null;
		}
		readBytes += 4;

		long [] offsets = new long[(int)count + 1];
		for (int i = 0; i < (int)count + 1; i ++) {
			offsets[i] = CIOUtil.readUnsignedInt(is);
			readBytes += 4;
		}

		List<ByteString> lists = new ArrayList<ByteString>();
		for (int i = 0; i < (int)count; i ++) {
			if ((int)offsets[i+1] == (int)offsets[i])
				lists.add(null);
			else
				lists.add(new ByteString(data, previousBytesLength +
					(int)offsets[i] + readBytes + (int)offset, (int)offsets[i+1] - (int)offsets[i] - 1 /* \0 */));
		}
		
		return lists;
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	List getArray(int offset) throws IOException {

		if (!parsedOK)
			throw new DRCClientRunTimeException("Message not parsed");
		
		DataInputStream is = new DataInputStream(new ByteArrayInputStream(data, previousBytesLength + offset, data.length - offset - previousBytesLength));
		
		byte t = is.readByte();
		
		if ((t & DataType.DC_ARRAY) == 0) {
			throw new DRCClientRunTimeException("Data type not array or not string");
		}
		
		long count = CIOUtil.readUnsignedInt(is);
		if (count == 0) {
			return null;
		}

	
		List lists = null;
		int elementSize = 0;
		Class typeClass = null;
		switch (t & DataType.DT_MASK) {
		case DataType.DT_INT8:
			elementSize = 1;
			typeClass = Byte.class;
			lists = new ArrayList<Byte>();
			break;
		case DataType.DT_UINT8:
			elementSize = 1;
			typeClass = Integer.class;
			lists = new ArrayList<Integer>();
			break;
		case DataType.DT_INT16:
			elementSize = 2;
			typeClass = Short.class;
			lists = new ArrayList<Short>();
			break;
		case DataType.DT_UINT16:
			elementSize = 2;
			typeClass = Integer.class;
			lists = new ArrayList<Integer>();
			break;
		case DataType.DT_INT32:
			elementSize = 4;
			typeClass = Integer.class;
			lists = new ArrayList<Integer>();
			break;
		case DataType.DT_UINT32:
			elementSize = 4;
			typeClass = Long.class;
			lists = new ArrayList<Long>();
			break;
		case DataType.DT_INT64:
			elementSize = 8;
			typeClass = Long.class;
			lists = new ArrayList<Long>();
			break;
		case DataType.DT_UINT64:
			elementSize = 8;
			typeClass = BigInteger.class;
			lists = new ArrayList<BigInteger>();
			break;
		default:
			throw new DRCClientRunTimeException("Unkown type " + t);
		}

		for (int i = 0; i < (int)count; i ++) {
			switch (elementSize) {
			case 1:
				if (typeClass == Byte.class) {
					lists.add(is.readByte());
				} else if (typeClass == Integer.class) {
					lists.add(is.readUnsignedByte());
				}
				break;
			case 2:
				if (typeClass == Short.class) {
					lists.add(CIOUtil.readShort(is));
				} else if (typeClass == Integer.class) {
					lists.add(CIOUtil.readUnsignedShort(is));
				}
				break;
			case 4:
				if (typeClass == Integer.class) {
					lists.add(CIOUtil.readInt(is));
				} else if (typeClass == Long.class) {
					lists.add(CIOUtil.readUnsignedInt(is));
				}
				break;
			case 8:
				if (typeClass == Long.class) {
					lists.add(CIOUtil.readLong(is));
				} else if (typeClass == BigInteger.class) {
					throw new DRCClientRunTimeException("Unsupported unsigned long");
				}
				break;
			}
		}
		return lists;
	}
}
