package com.aliyun.drc.client.message.transfer;

import com.aliyun.drc.client.message.DataMessage;
import com.aliyun.drc.client.message.ByteString;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

public class String2JavaObject {

    private static final String MAX_INTEGER = Integer.toString(Integer.MAX_VALUE);
    private static final String MAX_LONG = Long.toString(Long.MAX_VALUE);
    private static Timestamp ZEROTIME = Timestamp.valueOf("1970-01-01 00:00:00");

    /* Used to transfer from drc field type to java object type. */
    private static final Map<DataMessage.Record.Field.Type, Integer> jdbcTypes =
            new HashMap<DataMessage.Record.Field.Type, Integer>() {
                private static final long serialVersionUID = 1L;

                {
                    put(DataMessage.Record.Field.Type.INT8, java.sql.Types.TINYINT);
                    put(DataMessage.Record.Field.Type.INT16, java.sql.Types.SMALLINT);
                    put(DataMessage.Record.Field.Type.INT24, java.sql.Types.INTEGER);
                    put(DataMessage.Record.Field.Type.INT32, java.sql.Types.INTEGER);
                    put(DataMessage.Record.Field.Type.INT64, java.sql.Types.BIGINT);
                    put(DataMessage.Record.Field.Type.DECIMAL, java.sql.Types.DECIMAL);
                    put(DataMessage.Record.Field.Type.FLOAT, java.sql.Types.FLOAT);
                    put(DataMessage.Record.Field.Type.DOUBLE, java.sql.Types.DOUBLE);
                    put(DataMessage.Record.Field.Type.NULL, java.sql.Types.NULL);
                    put(DataMessage.Record.Field.Type.TIMESTAMP, java.sql.Types.TIMESTAMP);
                    put(DataMessage.Record.Field.Type.DATE, java.sql.Types.DATE);
                    put(DataMessage.Record.Field.Type.TIME, java.sql.Types.TIME);
                    put(DataMessage.Record.Field.Type.DATETIME, java.sql.Types.TIMESTAMP);
                    put(DataMessage.Record.Field.Type.YEAR, java.sql.Types.INTEGER);
                    put(DataMessage.Record.Field.Type.BIT, java.sql.Types.BIT);
                    put(DataMessage.Record.Field.Type.ENUM, java.sql.Types.CHAR);
                    put(DataMessage.Record.Field.Type.SET, java.sql.Types.CHAR);
                    put(DataMessage.Record.Field.Type.BLOB, java.sql.Types.BLOB);
                    put(DataMessage.Record.Field.Type.STRING, java.sql.Types.VARCHAR);
                    put(DataMessage.Record.Field.Type.GEOMETRY, java.sql.Types.BLOB);
                    put(DataMessage.Record.Field.Type.JSON, java.sql.Types.VARCHAR);
                    put(DataMessage.Record.Field.Type.UNKOWN, java.sql.Types.NULL);
                }
            };

    synchronized public static void setZeroTime(Timestamp ts) {
        ZEROTIME = ts;
    }

    synchronized public static final Timestamp getZeroTime() {
        return ZEROTIME;
    }

    /**
     * Transfer the drc field type to java object type
     *
     * @param type is the drc field type
     * @return the value of the java object type
     */
    public static int fieldType2Java(DataMessage.Record.Field.Type type) {
        return jdbcTypes.get(type);
    }

    /**
     * Transfer from string with field type to the java object
     *
     * @param svalue is the string value
     * @param type   is the field type
     * @return the java object
     * @throws NumberFormatException
     * @throws UnsupportedEncodingException
     */
    public static JavaObject field2Java(final DataMessage.Record.Field field)
            throws NumberFormatException, UnsupportedEncodingException {

        JavaObject ob = new JavaObject();
        ob.setType(jdbcTypes.get(field.getType()));

        /* Get field value rvalue */
        ByteString value = field.getValue();
        if (value == null)
            return ob;

        String rvalue = null;
        if (ob.getType() == java.sql.Types.CHAR || ob.getType() == java.sql.Types.VARCHAR) {
        	if(!field.getEncoding().equals("binary")) {
        		rvalue = value.toString(field.getEncoding());
        	} else {
        		ob.setType(java.sql.Types.BLOB);
        	}
        } else if (ob.getType() != java.sql.Types.BLOB) {
            rvalue = value.toString();
        }

        /* Get java object */

        switch (ob.getType()) {
            case java.sql.Types.BIT:
                bitstr2Object(rvalue, ob);
                break;
            case java.sql.Types.TINYINT:
            case java.sql.Types.SMALLINT:
                ob.setObject(Integer.valueOf(rvalue));
                break;
            case java.sql.Types.INTEGER:
                intstr2Object(rvalue, ob);
                break;
            case java.sql.Types.BIGINT:
                bigintstr2object(rvalue, ob);
                break;
            case java.sql.Types.DECIMAL:
                ob.setObject(new BigDecimal(rvalue));
                break;
            case java.sql.Types.FLOAT:
                ob.setObject(Float.valueOf(rvalue));
                break;
            case java.sql.Types.DOUBLE:
                ob.setObject(Double.valueOf(rvalue));
                break;
            case java.sql.Types.TIMESTAMP:
                timestamp2object(rvalue, ob);
                break;
            case java.sql.Types.DATE:
                date2object(rvalue,ob);
                break;
            case java.sql.Types.TIME:
                time2object(rvalue, ob);
                break;
            case java.sql.Types.BLOB:
                ob.setObject(value.getBytes());
                break;
            case java.sql.Types.CHAR:
            case java.sql.Types.VARCHAR:
                ob.setObject(rvalue);
                break;
            case java.sql.Types.NULL:
            default:
                ob.setObject(null);
        }

        return ob;
    }


    private static Object bitstr2Object(final String rvalue, final JavaObject ob) {
        int ret = rvalue.compareTo("1");
        if (ret > 0) {
            BigInteger bi = new BigInteger(rvalue);
            byte[] bytes = bi.toByteArray();
            int realLength = bytes.length;
            int i = 0;
            for (; i < bytes.length; i++) {
                if (bytes[i] != 0) {
                    break;
                } else {
                    realLength--;
                }
            }
            byte[] realbyte = new byte[realLength];
            for (int j = i; j < bytes.length; j++) {
                realbyte[j - i] = bytes[j];
            }
            ob.setObject(realbyte);
        } else {
            ob.setObject(((ret == 0) ? true : false));
        }
        return ob;
    }

    private static Object intstr2Object(final String rvalue, final JavaObject ob) {
        int rlength = rvalue.length();
        int maxLength = MAX_INTEGER.length();
        if (rvalue.contains("-") || rlength < maxLength ||
                (rlength == maxLength && rvalue.compareTo(MAX_INTEGER) < 0)) {
            ob.setObject(Integer.valueOf(rvalue));
        } else {
            ob.setUnsigned(true);
            ob.setObject(Long.valueOf(rvalue));
        }
        return ob;
    }

    private static Object bigintstr2object(final String rvalue, final JavaObject ob) {
        int rlength = rvalue.length();
        int maxLength = MAX_LONG.length();
        if (rvalue.contains("-") || rlength < maxLength ||
                (rlength == maxLength && rvalue.compareTo(MAX_LONG) < 0)) {
            ob.setObject(Long.valueOf(rvalue));
        } else {
            ob.setUnsigned(true);
            ob.setObject(new BigInteger(rvalue));
        }
        return ob;
    }

    private static Object timestamp2object(final String rvalue, final JavaObject ob) {
        Object objectValue;
        try {
        	 /* assume the format is ssssssssss[.f...] */
        	long secondsRepresentedByMilli;
        	secondsRepresentedByMilli = Long.parseLong(StringUtils.substringBefore(rvalue, ".")) * 1000;     	
            String fractionalRepresentedByNano = StringUtils.substringAfter(rvalue, ".");
            if (fractionalRepresentedByNano.contains(".") || fractionalRepresentedByNano.length() > 9) {
                objectValue=rvalue;
                ob.setOutOfRange(true);
            } else {
                fractionalRepresentedByNano = String.format("%-9s", fractionalRepresentedByNano).replace(" ", "0");
                objectValue = new java.sql.Timestamp(secondsRepresentedByMilli);
                try {  
                	((java.sql.Timestamp)objectValue).setNanos(Integer.parseInt(fractionalRepresentedByNano));
                } catch (NumberFormatException ne) {
                    objectValue=rvalue;
                    ob.setOutOfRange(true);
                }
            }        
        } catch (IllegalArgumentException e) {
        	/* assume the format is yyyy-mm-dd hh:mm:ss[.f...] */
        	try {
        		objectValue = java.sql.Timestamp.valueOf(rvalue);
        	} catch(Exception err) {
        		objectValue=rvalue;
                ob.setOutOfRange(true);
            }
        }
        ob.setObject(objectValue);
        return ob;
    } 


    private static Object date2object(final String rvalue,final JavaObject ob){
        Object objectValue;
        try {
            objectValue = java.sql.Date.valueOf(rvalue.replaceAll("[^0-9]","-"));
        }catch (Exception e){
            objectValue=rvalue;
            ob.setOutOfRange(true);
        }
        ob.setObject(objectValue);
        return ob;
    }

    private static Object time2object(final String rvalue, final JavaObject ob) {
        Object objectValue = java.sql.Time.valueOf(rvalue);
        if (!((java.sql.Time) objectValue).toString().equals(rvalue)) {
            ob.setOutOfRange(true);
        }
        ob.setObject(objectValue);
        return ob;
    }
}

