package com.aliyun.drc.clusterclient.impl;

import com.aliyun.drc.client.Listener;
import com.aliyun.drc.client.message.DataMessage;
import com.aliyun.drc.client.message.DataMessage.Record;
import com.aliyun.drc.clusterclient.message.ClusterMessage;
import com.aliyun.drc.clusterclient.partition.Checkpoint;
import com.aliyun.drc.clusterclient.partition.CloudPartitionImpl;
import com.taobao.drc.clusterclient.MessageNotifier;
import com.taobao.drc.clusterclient.NotifyController;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

public class CloudClientListenerAdapter implements Listener {
    private static Logger logger = LoggerFactory.getLogger(CloudClientListenerAdapter.class);

    private final CloudPartitionImpl partition;
    private final NotifyController<ClusterMessage> notifyController;
    private final MessageNotifier<ClusterMessage> notifier;
    private final DataTypeEnum dataTypeEnum;

    public CloudClientListenerAdapter(CloudPartitionImpl partition, NotifyController<ClusterMessage> notifyController, MessageNotifier<ClusterMessage> notifier, String dataType) {
        this.partition = partition;
        this.notifyController = notifyController;
        this.notifier = notifier;
        if (dataType.equalsIgnoreCase("dml")) {
            dataTypeEnum = DataTypeEnum.DML;
        } else {
            dataTypeEnum = DataTypeEnum.ALL;
        }
    }

    @Override
    public void notify(DataMessage message) throws Exception {
        if (notifyController.isClosed()) {
            return;
        }
        notifyController.setNotifying(true);
        try {
            while (!notifyController.isValid()) {
                if (notifyController.isClosed()) {
                    logger.warn("Client for partition [{}] was stopped", partition.getName());
                    return;
                }
                Thread.sleep(1);
            }

            List<ClusterMessage> messages = new ArrayList<ClusterMessage>();
            for (Record record : message.getRecordList()) {
                // record pushed in memory and msgQueue size log tagging in every 60s
                ClusterMessage msg = new ClusterMessage(record, partition);
                switch (record.getOpt()) {
                    case HEARTBEAT:
                        partition.bookCheckpoint(msg.getCheckpoint());
                        partition.ackAsConsumed(msg.getCheckpoint());
                        messages.add(msg);
                        break;
                    case DDL:
                        if (dataTypeEnum == DataTypeEnum.DML) {
                            break;
                        }
                    default:
                        messages.add(msg);
                        partition.bookCheckpoint(msg.getCheckpoint());
                        notifyController.onNotified(msg);
                }
            }
            if (messages.size() == 0) {
                return;
            }
            notifier.onMessages(messages);
        } finally {
            notifyController.setNotifying(false);
        }
    }

    @Override
    public void notifyRuntimeLog(String level, String log) throws Exception {
        if (level.equalsIgnoreCase("INFO")) {
            logger.info(log);
        } else if (level.equalsIgnoreCase("ERROR")) {
            logger.error(log);
        } else if (level.equalsIgnoreCase("WARN")) {
            logger.warn(log);
        } else {
            logger.warn(log);
        }
    }

    @Override
    public void handleException(Exception e) {
        logger.warn("Caught exception in listener adapter for partition [{}], stop partition...", partition.getName(), e);
        notifyController.close();
    }
}
