package com.aliyun.drc.clusterclient.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.aliyun.drc.client.DRCClient;
import com.aliyun.drc.client.DRCClientFactory;
import com.aliyun.drc.client.DataFilter;
import com.aliyun.drc.client.DataFilterBase;
import com.aliyun.drc.clusterclient.ClusterContext;
import com.aliyun.drc.clusterclient.message.ClusterMessage;
import com.aliyun.drc.clusterclient.partition.Checkpoint;
import com.aliyun.drc.clusterclient.partition.CloudPartitionImpl;
import com.taobao.drc.clusterclient.MessageNotifier;
import com.taobao.drc.clusterclient.NotifyController;
import com.taobao.drc.clusterclient.PartitionClient;
import com.taobao.drc.clusterclient.clustermanager.PartitionInfo;
import com.taobao.drc.clusterclient.partition.IPartition;
import com.taobao.drc.clusterclient.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

/**
 * @author yangyang
 * @since 2017/7/4
 */
public class CloudPartitionClient implements PartitionClient<Checkpoint> {
	private static final Logger logger = LoggerFactory.getLogger(CloudPartitionClient.class);

	private final ClusterContext context;
	private final PartitionInfo partitionInfo;
	private final MessageNotifier<ClusterMessage> notifier;
	private final String initOffset;

	private final NotifyController<ClusterMessage> notifyController;
	private final CloudPartitionImpl partition;
	private final DRCClient client;
	private Thread clientThread;

	public CloudPartitionClient(ClusterContext context, PartitionInfo partitionInfo,
			MessageNotifier<ClusterMessage> notifier, String initOffset) throws Exception {
		this.context = context;
		this.partitionInfo = partitionInfo;
		this.notifier = notifier;
		this.initOffset = initOffset;

		this.notifyController = new NotifyController<ClusterMessage>(Time.SYSTEM);
		this.partition = new CloudPartitionImpl(context, partitionInfo);

		Properties properties = new Properties();
		properties.putAll(context.getProperties());
		if (partitionInfo.getTables() != null && !partitionInfo.getTables().isEmpty()) {
			properties.setProperty(ClusterContext.TAG_META_MAPPING,
					JSON.toJSONString(partitionInfo.getTables(), SerializerFeature.SortField));
		}
		this.client = DRCClientFactory.create(DRCClientFactory.Type.MYSQL, properties);
		initClient();
	}

	private void initClient() throws Exception {
		if (context.isUsePublicIp()) {
			client.usePublicIp();
		}
		logger.info("CloudPartitionClient before parseDataFilter");
		client.addDataFilter(parseDataFilter());
		logger.info("CloudPartitionClient after parseDataFilter");
		client.addListener(
				new CloudClientListenerAdapter(partition, notifyController, notifier, context.getDataType()));
		client.initService(context.getAppGroupUserName(), partitionInfo.getTopic(), context.getAppGroupPassword(),
				parseCheckpoint(initOffset), null);
	}

	private com.aliyun.drc.client.impl.Checkpoint parseCheckpoint(String offset) {
		Checkpoint checkpoint = new Checkpoint(offset, partition);
		com.aliyun.drc.client.impl.Checkpoint drcClientCheckpoint = new com.aliyun.drc.client.impl.Checkpoint();
		if (checkpoint.getInstance() != null) {
			drcClientCheckpoint.setServerId(checkpoint.getInstance());
		}
		if (checkpoint.getFilePosition() != null) {
			drcClientCheckpoint.setPosition(checkpoint.getFilePosition());
		}
		if (checkpoint.getTimestamp() != null) {
			drcClientCheckpoint.setTimestamp(checkpoint.getTimestamp());
		}
		if (checkpoint.getId() != null) {
			drcClientCheckpoint.setRecordId(checkpoint.getId());
		}
		return drcClientCheckpoint;
	}

	private DataFilterBase parseDataFilter() {
		logger.info("CloudPartitionClient begin parseDataFilter");
		String filterStr = partitionInfo.makeFilterString();
		logger.info("CloudPartitionClient Filter for partition [{}][{}][{}]: [{}]", context.getAppGuid(),
				context.getAppGroup(), partitionInfo.getPartition(), filterStr);
		return new DataFilter(filterStr);
	}

	@Override
	public void start() throws IOException {
		if (clientThread != null) {
			throw new IllegalStateException("Client for partition [" + partitionInfo + "] has already started");
		}
		try {
			clientThread = client.startService();
			clientThread.setName("DTS-DRCClient-" + partition.getName() + "-Thread");
			clientThread.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
				@Override
				public void uncaughtException(Thread arg0, Throwable arg1) {
					logger.error(arg0.getName() + ", " + arg1.toString());
				}
			});
		} catch (IOException e) {
			throw e;
		} catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public boolean isActive() {
		return clientThread != null && clientThread.isAlive();
	}

	@Override
	public String offset() {
		if (client != null) {
			Checkpoint checkpoint = partition.getCheckpoint();
			if (checkpoint != null) {
				return checkpoint.toString();
			}
		}
		return null;
	}

	@Override
	public NotifyController getNotifyController() {
		return notifyController;
	}

	@Override
	public IPartition<Checkpoint> getPartition() {
		return partition;
	}

	@Override
	public void close() throws IOException {
		if (clientThread == null) {
			return;
		}
		try {
			client.stopService();
			clientThread.join();
		} catch (IOException e) {
			logger.error("Failed to stop client", e);
			throw e;
		} catch (Exception e) {
			logger.error("Failed to stop client", e);
			throw new IllegalStateException(e);
		}
	}

	@Override
	public Map<String, Object> getMetrics() {
		Map<String, Object> map = new TreeMap<String, Object>();
		map.putAll(partition.getMetrics());
		map.putAll(notifyController.getMetrics());
		return map;
	}
}
