package com.tydic.base.mock;

import com.tydic.base.mock.annotation.Autoinject;
import com.tydic.base.mock.annotation.InjectType;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.context.ApplicationContext;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;

/** <br>
 * 标题: @AutoMock 注解处理器<br>
 * 描述: 实现@AutoMock注解的功能<br>
 * 公司: www.tydic.com<br>
 *
 * @author zhangcheng
 * @date 2018/03/05
 */
public class AutoinjectHandle implements AnnotationHandle {

    protected final Log logger = LogFactory.getLog(this.getClass());

    @Override
    public boolean hasExist(Field field) {
        if(field == null){
            return false;
        }
        for(Annotation annotation : field.getDeclaredAnnotations()){
            if(annotation instanceof Autoinject){
                return true;
            }
        }
        return false;
    }

    @Override
    public Object invok(Object bean, String beanName,Field field,ApplicationContext applicationContext) throws BeansException {

        Autoinject autoMock = field.getDeclaredAnnotation(Autoinject.class);

        // 如果AutoType.MOCK，且mock类存在时，将mock类注入
        if (InjectType.MOCK == autoMock.value() && this.hasBeanMock(field)){
            this.invokLog(beanName, field, autoMock, InjectType.MOCK);
            return this.getBeanByMock(bean,field);
        }

        // 如果AutoType.WIRED，且impl类存在时，将impl类注入
        if (InjectType.WIRED == autoMock.value() && this.hasBeanImpl(field,applicationContext)){
            this.invokLog(beanName, field, autoMock, InjectType.WIRED);
            return this.getBeanByImpl(bean,field,applicationContext);
        }

        // 如果AutoType.WIRED，impl类不存在，且mock类存在时，将mock类注入
        if (InjectType.WIRED == autoMock.value() && this.hasBeanMock(field)){
            this.invokLog(beanName, field, autoMock, InjectType.MOCK);
            return this.getBeanByMock(bean,field);
        }

        if (InjectType.MOCK == autoMock.value()){
            throw new BeanInitializationException("Bean初始化异常，名称为 '"+beanName+"' 的字段 '"+field.getName()+"' mock包下没有impl类");
        }

        throw new BeanInitializationException("Bean初始化异常，名称为 '"+beanName+"' 的字段 '"+field.getName()+"' mock包和impl包下都没有impl类");
    }

    private void invokLog(String beanName, Field field, Autoinject autoMock, InjectType injectType) {
        if (this.logger.isInfoEnabled()) {
            this.logger.info("完成重新装配Bean名称为 '" + beanName + "' 的字段名称为 '" + field.getName() + "' 装配类型为 '" + autoMock.value() + "' 实际装配类型为 '"+ injectType +"'");
        }
    }

    /** 获取Bean的impl实例从IOC容器中 */
    private Object getBeanByImpl(Object bean, Field field, ApplicationContext applicationContext)throws BeansException{
        try {
            field.setAccessible(true);
            field.set(bean,applicationContext.getBean(field.getName()+"Impl"));
            return bean;
        }catch (Exception e){
            throw new BeanInitializationException("Bean初始化异常，" + e.getMessage());
        }
    }

    /** 获取Bean的mock实例使用反射创建 */
    private Object getBeanByMock(Object bean, Field field){
        try {
            String mockPakcage = field.getType().getPackage().getName()+".mock";
            String mockName = field.getType().getSimpleName()+"Impl";
            String clazzName = mockPakcage + "."+ mockName;
            Class<? extends Object> clazz = Class.forName(clazzName);
            field.setAccessible(true);
            field.set(bean,clazz.newInstance());
            return bean;
        }catch (Exception e){
            throw new BeanInitializationException("Bean初始化异常，" + e.getMessage());
        }
    }

    /** 判断Bean对应的impl包下是否有impl类在IOC容器中 */
    private boolean hasBeanImpl(Field field, ApplicationContext applicationContext){
        try {
            applicationContext.getBean(field.getName()+"Impl");
            return true;
        }catch (BeansException e){
            return false;
        }
    }

    /**
     * 判断Bean对应的mock包下是否有impl类在API工程中
     * @param field
     * @return
     */
    private boolean hasBeanMock(Field field){
        try {
            String mockPakcage = field.getType().getPackage().getName()+".mock";
            String mockName = field.getType().getSimpleName()+"Impl";
            String clazzName = mockPakcage + "."+ mockName;
            Class<? extends Object> clazz = Class.forName(clazzName);
            clazz.newInstance();
            return true;
        }catch (Exception e){
            return false;
        }
    }
}
