package com.tydic.base.mock;

import com.alibaba.fastjson.JSON;
import com.tydic.base.mock.exception.CaseNotExistException;
import com.tydic.base.mock.exception.MockException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

/**
 * <br>
 * 标题: Mock核心API<br>
 * 描述: 提供获取服务响应报文服务。主要功能包括：加载报文配置文件、解析业务场景、实例化对象并封装响应报文<br>
 * 公司: www.tydic.com<br>
 *
 * @author zhangcheng
 * @date 2018/03/05
 */
public class Mock {

    protected static final Log logger = LogFactory.getLog(Mock.class);
    /** 全局服务 */
    private static Map<String, Map<String, String>> globalService = new HashMap<>();
    /** 当前服务 */
    private static Map<String, String> serviceCase = new HashMap<>();

    /**
     * 获取响应报文实例
     *
     * @author zhangcheng
     * @date 2018/03/05
     */
    public static <T> T getResponse(Class<T> type, String serviceName, String caseName) {
        try {

            if (logger.isDebugEnabled()) {
                logger.debug("Mock ServiceName is '" + serviceName + "' RspBO is '" + type + "' CaseName is '" + caseName + "'");
            }

            loadServiceCase(serviceName);

            if (!serviceCase.containsKey(caseName)) {
                throw new CaseNotExistException("Mock CaseName '" + caseName + "' not exist, Please check this file '" + getServiceResourcesPath(serviceName) + "'");
            }

            String jsonStr = String.valueOf(serviceCase.get(caseName));

            if (logger.isDebugEnabled()) {
                logger.debug("Mock CaseName is '" + caseName + "' the response is '" + jsonStr + "'");
            }

            return JSON.parseObject(jsonStr, type);
        } catch (Exception e) {
            e.printStackTrace();
            throw new MockException("获取响应报文实例异常: " + e.getMessage());
        }
    }

    /** serviceName 为类名，如：OrderService */
    private static void loadServiceCase(String serviceName) throws Exception {

        // 从缓存中查找是否已加载该服务的业务场景
        if (globalService.containsKey(serviceName)) {
            serviceCase = globalService.get(serviceName);
            return;
        }

        // 从资源文件中加载该服务的业务场景
        InputStream is = Mock.class.getClass().getResourceAsStream(getServiceResourcesPath(serviceName));

        if (null == is) {
            throw new FileNotFoundException("缺失资源文件: '" + getServiceResourcesPath(serviceName) + "'");
        }

        byte[] buf = new byte[1024];
        ByteArrayOutputStream baos = new ByteArrayOutputStream();

        for (int i; (i = is.read(buf)) != -1; ) {
            baos.write(buf, 0, i);
        }
        String data = baos.toString("UTF-8");
        serviceCase = JSON.parseObject(data, Map.class);
        is.close();

        // 缓存该服务的业务场景
        globalService.put(serviceName, serviceCase);
    }

    /**
     * 获取服务的json资源路径
     */
    private static String getServiceResourcesPath(String serviceName) {
        return "/mock/" + serviceName + ".json";
    }
}
