package com.tydic.base.mock;

import com.tydic.base.mock.exception.FieldsNotFoundException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.ApplicationContext;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

/** <br>
 * 标题: Bean装配器<br>
 * 描述: 提供注解解析入口，实现Bean装配，基于spring的BeanPostProcessor接口做扩展<br>
 * 公司: www.tydic.com<br>
 *
 * @author zhangcheng
 * @date 2018/03/05
 */
public class MockProcessor implements BeanPostProcessor{

    private static List<AnnotationHandle> globalHandles = new ArrayList<>();

    @Autowired
    private ApplicationContext applicationContext;

    static {
        globalHandles.add(new AutoinjectHandle());
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        return this.invokeBean(bean,beanName);
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return this.invokeBean(bean,beanName);
    }

    /**
     * 重新解析Bean依赖
     */
    private Object invokeBean(Object bean, String beanName){

        Field[] fields = getDeclaredFields(bean);
        for (Field field : fields){
            for (AnnotationHandle handle : globalHandles){
                if (handle.hasExist(field)){
                    bean = handle.invok(bean,beanName,field,applicationContext);
                }
            }
        }

        return bean;
    }

    /**
     * 获取Bean的所有字段
     * @param bean
     * @return
     */
    private Field[] getDeclaredFields(Object bean) {
        if (AopUtils.isAopProxy(bean)){
            try {
                return bean.getClass().getSuperclass().getDeclaredFields();
            } catch (SecurityException e) {
                e.printStackTrace();
                throw new FieldsNotFoundException("Mock get bean '"+bean.getClass().getSuperclass().getSimpleName()+"' declared fields not found ");
            }
        }
        return bean.getClass().getDeclaredFields();
    }
}
