package com.ohaotian.base.cache;

import com.ohaotian.base.constant.SessionConstant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;

/**
 * CacheTemplate
 *
 * @author liubinqi(liubq@tydic.com)
 */
public class CacheTemplate {

    private final static Logger log = LoggerFactory.getLogger(CacheTemplate.class);


    private static volatile CacheTemplate instance;

    private static CaffeineCache caffeineCache;

    private CacheService redisCache;


    private static final RedisPubSubClusterPolicy policy = new RedisPubSubClusterPolicy();

    private CacheTemplate() {

    }

    public static CacheTemplate getIstance() {
        // 对象实例化时与否判断（不使用同步代码块，instance不等于null时，直接返回对象，提高运行效率）
        if (instance == null) {
            //同步代码块（对象未初始化时，使用同步代码块，保证多线程访问时对象在第一次创建后，不再重复被创建）
            synchronized (CacheTemplate.class) {
                //未初始化，则初始instance变量
                if (instance == null) {
                    instance = new CacheTemplate();
                }
            }
        }
        return instance;
    }

    protected void init(CaffeineCache caffeineCache, CacheService redisCache) {
        this.caffeineCache = caffeineCache;
        this.redisCache = redisCache;
        policy.connect(redisCache.getJedisPool());
    }


    /**
     * 设置缓存
     */
    public void set(String key, Object value) {
        try {
            caffeineCache.put(key, value);
            redisCache.put(key, value, SessionConstant.defaultMaxInactiveIntervalReids);
        } finally {
            policy.sendClearCmd(key);
        }

    }


    /**
     * 获取缓存值
     */
    public Object get(String key) {

        Object level1Value = caffeineCache.get(key);
        if (level1Value != null) {
            return level1Value;
        }
        Object level2Value = redisCache.get(key);
        if (level2Value != null) {
            caffeineCache.put(key, level2Value);
            Jedis jedis = null;
            try {
                jedis = redisCache.getJedisPool().getResource();
                jedis.expire(key, 600);
            } finally {
                if (jedis != null) {
                    jedis.close();
                }

            }
            return level2Value;
        }
        return null;
    }


    /**
     * 删除集群缓存
     */
    public void del(String key) {
        caffeineCache.evict(key);
        redisCache.delete(key);
        policy.sendClearCmd(key);
    }

    /**
     * 清除本地缓存
     */
    public void evict(String key) {
        caffeineCache.evict(key);
    }


}
