package com.ohaotian.plugin.cache;

import com.ohaotian.plugin.common.annotation.MarkdownDoc;
import redis.clients.jedis.JedisPool;

import java.util.Set;

/**
 * 封装多种缓存的调用。
 */
@MarkdownDoc
public interface CacheClient {

    /**
     * 往缓存中存放值（永不过期）<br>
     * 适用场景: 需要常驻缓存中的数据<br>
     */
    void set(String key, Object value);

    /**
     * 往缓存中存放值（一段时间内有效）<br>
     * 适用场景: 需要常驻缓存中的数据<br>
     */
    void set(String key, Object value, int expire);

    /**
     * 从缓存中获取值
     */
    Object get(String key);

    /**
     * 从缓存中获取指定类型的结果
     */
    <T> T get(String key, Class<T> requiredType);

    /**
     * 从缓存中删除指定的key
     */
    void delete(String key);

    /**
     * 做穿透查询，缓存中查询不到的数据通过本地查询执行器回调改执行器得到数据，最后放入缓存
     */
    <T> T getFinal(CacheExecuterService<T> executer, String... key);

    /**
     * 根据表达式获取前缀一样的Key集合
     */
    Set<String> getkeys(String pattern);

    /**
     * 获取key的失效时间
     */
    Long getExpireTimeByKey(String key);

    /**
     * redis计数器，没有失效时间的
     */
    Long incr(String key);

    /**
     * redis计数器，有失效时间
     */
    Long incrExpireTime(String key, int expire);

    /**
     * redis计数器，没有失效时间的
     */
    Long incrBy(String key, long num);

    /**
     * redis计数器，有失效时间
     */
    Long incrExpireTimeBy(String key, long num, int expire);

    /**
     * <p>获取计数器的值</p>
     */
    Long getIncr(String key);

    /**
     * <p>通过key向list尾部添加字符串</P>
     *
     * @param key key
     * @param objs 可以使一个string 也可以使string数组
     * @return 返回list的value个数
     */
    Long rpush(String key, String... objs);

    /**
     * <p>通过key从list的头部删除一个value,并返回该value</p>
     *
     * @param key
     * @return
     */
    String lpop(String key);


    /**
     * <p>通过key返回list的长度. LLEN命令将返回存储在key列表的长度。如果key不存在，它被解释为一个空列表，则返回0。当存储在关key的值不是一个列表，则会返回错误。</p>
     *
     * @param key key
     * @return 返回队列的长度
     */
    Long llen(String key);


    /**
     * <p>
     * 将 key 的值设为 value ，当且仅当 key 不存在。
     * 若给定的 key 已经存在，则 SETNX 不做任何动作。
     * SETNX 是『SET if Not eXists』(如果不存在，则 SET)的简写.
     * </p>
     *
     * @param key
     * @return 返回值：设置成功，返回 1 。设置失败，返回 0 。
     */
    Long setnx(String key, Object value);

    /**
     * 该方法废弃，建议使用set(String key, Object value)
     *
     * 往缓存中存放值（永不过期）<br>
     * 适用场景: 需要常驻缓存中的数据<br>
     */
    @Deprecated
    void put(String key, Object value);

    /**
     * 该方法废弃，建议使用 set(String key, Object value, int expire)
     *
     * 往缓存中存放值（一段时间内有效）<br>
     * 适用场景: 需要常驻缓存中的数据<br>
     */
    @Deprecated
    void put(String key, Object value, int expire);

    /**
     * <p>
     * 语法：
     * GETSET key value
     * 功能：
     * 将给定 key 的值设为 value ，并返回 key 的旧值 (old value)，当 key 存在但不是字符串类型时，返回一个错误，当key不存在时，返回nil。
     * </p>
     */
    Object getSet(String key, Object value);


    /**
     * 获取Redis链接池
     *
     * @return Redis链接池
     */
    JedisPool getJedisPool();

    /**
     * 设置CacheManager.
     *
     * @param cacheManager cacheManager
     */
    void setCacheManager(CacheManager cacheManager);
}