package com.ohaotian.plugin.cache.config;

import com.ohaotian.plugin.cache.CacheClient;
import com.ohaotian.plugin.cache.RedisClusterManager;
import com.ohaotian.plugin.cache.impl.CacheClientImpl;
import com.ohaotian.plugin.cache.RedisManager;
import com.ohaotian.plugin.common.util.RSAUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

import java.util.HashSet;
import java.util.Set;

@Configuration
public class PluginCacheConfig {

    private static final Logger logger = LoggerFactory.getLogger(PluginCacheConfig.class);

    @Value("${redis.host}")
    private String host;

    @Value("${redis.port}")
    private int port;

    @Value("${redis.timeout}")
    private int timeout;

    @Value("${redis.pwd}")
    private String pwd;

    @Value("${redis.pwd.publicKey:}")
    private String publicKey;

    @Value("${redis.database:0}")
    private int database;

    @Value("${redis.soTimeOut:2000}")
    private int soTimeOut; //@Version 2.0.7
    @Value("${redis.maxAttemts:5}")
    private int maxAttemts; //@Version 2.0.7

    @Value("${redis.pool.maxTotal}")
    private int poolMaxTotal;

    @Value("${redis.pool.maxIdle}")
    private int poolMaxIdle;

    @Value("${redis.pool.maxWaitMillis}")
    private int poolMaxWaitMillis;

    @Value("${redis.pool.testOnBorrow:false}")
    private boolean poolTestOnBorrow;

    @Value("${redis.pool.testOnReturn:false}")
    private boolean poolTestOnReturn;



    /**
     * 如果有使用加密则执行解密
     */
    public void encrypt() {
        if (StringUtils.isNotEmpty(publicKey)) {
            try {
                logger.info("redis密码启用加密！");
                pwd = RSAUtil.decrypt(publicKey, pwd);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public static void main(String[] args) {
        try {
            RSAUtil.genKeyPair("123abcd");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Bean
    @Primary
    public JedisPoolConfig jedisPoolConfig() {
        printRedisConfig();
        encrypt();
        JedisPoolConfig config = new JedisPoolConfig();
        config.setMaxTotal(poolMaxTotal);
        config.setMaxIdle(poolMaxIdle);
        config.setMaxWaitMillis(poolMaxWaitMillis);
        config.setTestOnBorrow(poolTestOnBorrow);
        config.setTestOnReturn(poolTestOnReturn);
        return config;
    }

    private void printRedisConfig() {
        logger.info("plugin-cache-config:redis:host:{}, port:{}, timeout:{}, pwd:{}, database:{}", host, port, timeout, pwd, database);
        logger.info("plugin-cache-config:reids:poolMaxTotal:{}, poolMaxIdle:{}, poolMaxWaitMillis:{}, poolTestOnBorrow:{}, poolTestOnReturn:{}", poolMaxTotal, poolMaxIdle, poolMaxWaitMillis, poolTestOnBorrow, poolTestOnReturn);
    }

    /**
     * 单机
     * @return
     */
    @Bean
    @Primary
    @Conditional(SignleOrNot.class)
    public JedisPool jedisPool() {
        return new JedisPool(jedisPoolConfig(), host, port, timeout, pwd, database);
    }

    /**
     * 集群
     * @return
     */
    @Bean
    @Primary
    @Conditional(ClusterOrNot.class)
    public JedisCluster jedisCluster() {
        String clusterNodes = host;
        // 集群redis
        String[] cNodes = clusterNodes.split(",");
        Set<HostAndPort> nodes =new HashSet<>();
        //分割出集群节点
        for(String node : cNodes) {
            String[] hp = node.split(":");
            nodes.add(new HostAndPort(hp[0], Integer.parseInt(hp[1])));
        }
        JedisCluster jedisCluster = new JedisCluster(nodes, timeout, soTimeOut, maxAttemts, pwd, jedisPoolConfig());
        return jedisCluster;
    }

    @Bean
    @Primary
    @Conditional(ClusterOrNot.class)
    public RedisClusterManager redisClusterManager() {
        RedisClusterManager manager = new RedisClusterManager();
        manager.setJedisClusterPool(jedisCluster());
        manager.setExpire(0);
        return manager;
    }

    @Bean
    @Primary
    @Conditional(SignleOrNot.class)
    public  RedisManager redisManager() {
        RedisManager manager = new RedisManager();
        manager.setJedisPool(jedisPool());
        return manager;
    }

    @Bean
    @Primary
    public CacheClient cacheClient() {
        CacheClient cacheClient = new CacheClientImpl();
        if(host.indexOf(",") > -1){
            logger.info("redis-cluster 启动");
            cacheClient.setCacheManager(redisClusterManager());
        }else{
            logger.info("redis 单机启动");
            cacheClient.setCacheManager(redisManager());
        }
        return cacheClient;
    }
}
