package com.ohaotian.plugin.cache.impl;


import com.ohaotian.license.verify.service.LicenseVerifyService;
import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.ohaotian.plugin.cache.CacheClient;
import com.ohaotian.plugin.cache.CacheExecuterService;
import com.ohaotian.plugin.cache.CacheManager;
import com.ohaotian.plugin.cache.RedisClusterManager;
import com.ohaotian.plugin.util.ConvertUtil;
import com.ohaotian.plugin.common.util.SerializationUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;

import java.util.HashSet;
import java.util.Set;


public class CacheClientImpl implements CacheClient {

    private final static Logger log = LoggerFactory.getLogger(CacheClientImpl.class);
    private boolean isCluster = false; //是否集群
    private LicenseVerifyService licenseVerifyService;

    /**
     * 缓存底层接口
     */
    private CacheManager cacheManager;

    /**
     * 是否集群
     * @return
     */
    @Override
    public boolean isCluster() {
        return isCluster;
    }
    /**
     * 返回 缓存底层接口
     *
     * @return 缓存底层接口
     */
    public CacheManager getCacheManager() {
        return cacheManager;
    }

    /**
     * 设置 缓存底层接口
     *
     * @param cacheManager 缓存底层接口
     */
    @Override
    public void setCacheManager(CacheManager cacheManager) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        this.cacheManager = cacheManager;
        if(cacheManager instanceof RedisClusterManager){
            this.isCluster = true;
        }
    }

    @Override
    public void set(String key, Object value) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        cacheManager.set(key.getBytes(), SerializationUtils.serializeSlightly(value));

    }

    @Override
    public void put(String key, Object value) {
        cacheManager.set(key.getBytes(), SerializationUtils.serializeSlightly(value));

    }

    @Override
    public void set(String key, Object value, int expire) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        cacheManager.set(key.getBytes(), SerializationUtils.serializeSlightly(value), expire);
    }

    @Override
    public void put(String key, Object value, int expire) {
        cacheManager.set(key.getBytes(), SerializationUtils.serializeSlightly(value), expire);
    }

    @Override
    public Object get(String key) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        byte[] value = cacheManager.get(key.getBytes());

        if (value == null) {
            return null;
        }
        return SerializationUtils.deserializeSlightly(value);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T get(String key, Class<T> requiredType) {
        byte[] value = cacheManager.get(key.getBytes());

        if (value == null) {
            return null;
        }
        return (T) SerializationUtils.deserializeSlightly(value);
    }

    @Override
    public void delete(String key) {
        cacheManager.del(key.getBytes());
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T getFinal(CacheExecuterService<T> executer, String... key) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        String finalKey = this.getFinalKey(key);
        log.debug("finalKey=" + finalKey);
        byte[] value = cacheManager.get(finalKey.getBytes());

        if (value == null || value.length == 0) {
            log.debug("数据在缓存中不存在！执行本地数据查询器查询数据");
            Object obj = executer.getParamFromDb(key);
            if (obj != null && !"".equals(obj)) {
                log.debug("value=" + obj.toString());
                this.put(finalKey, obj);
                return (T) obj;
            } else {
                log.debug("数据在数据库中不存在！");
                return null;
            }
        }
        Object finalValue = SerializationUtils.deserializeSlightly(value);
        log.debug("value=" + finalValue);
        return (T) finalValue;
    }

    /**
     * 获取完整的key<br>
     * 适用场景: <br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param key
     * @return
     * @autho liuce
     * @time 2016年8月15日 下午1:36:36
     */
    private String getFinalKey(String... key) {
        StringBuffer fkey = new StringBuffer();
        for (String s : key) {
            fkey.append(s);
            fkey.append("_");
        }
        fkey.setLength(fkey.length() - 1);
        return fkey.toString();
    }


    @Override
    public Set<String> getkeys(String pattern) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        Set<String> keys = new HashSet<String>();

        Set<byte[]> keybs = cacheManager.keys(pattern);

        for (byte[] bs : keybs) {
            keys.add(new String(bs));
        }

        return keys;
    }

    @Override
    public Long getExpireTimeByKey(String key) {
        return cacheManager.getExpireTimeByKey(key.getBytes());
    }

    @Override
    public Long incr(String key) {
        return cacheManager.incr(key.getBytes());
    }

    @Override
    public Long incrExpireTime(String key, int expire) {
        return cacheManager.incrExpireTime(key.getBytes(), expire);
    }

    @Override
    public Long incrBy(String key, long num) {
        return cacheManager.incrBy(key.getBytes(), num);
    }

    @Override
    public Long incrExpireTimeBy(String key, long num, int expire) {
        return cacheManager.incrExpireTimeBy(key.getBytes(), num, expire);
    }

    @Override
    public Long getIncr(String key) {
        return cacheManager.getIncr(key.getBytes());
    }

    //队列操作 begin

    /**
     * <p>通过key向list尾部添加字符串</p>
     *
     * @param key
     * @param obj
     * @return 返回list的value个数
     */
    @Override
    public Long rpush(String key, Object obj) {
        return cacheManager.rpush(key.getBytes(), ConvertUtil.serialize(obj));
    }


    @Override
    public String lpop(String key) {
        return cacheManager.lpop(key);
    }

    /**
     * <p>通过key返回list的长度.
     * LLEN命令将返回存储在key列表的长度。如果key不存在，它被解释为一个空列表，则返回0。当存储在关key的值不是一个列表，则会返回错误。</p>
     *
     * @param key
     * @return
     */
    @Override
    public Long llen(String key) {
        return cacheManager.llen(key);
    }
    //队列操作 end

    //锁定操作 begin

    /**
     * 将 key 的值设为 value ，当且仅当 key 不存在。
     * 若给定的 key 已经存在，则 SETNX 不做任何动作。
     * SETNX 是『SET if Not eXists』(如果不存在，则 SET)的简写
     *
     * @param key
     * @return 返回值：设置成功，返回 1 。设置失败，返回 0 。
     */
    @Override
    public Long setnx(String key, Object value) {
        return cacheManager.setnx(key.getBytes(), SerializationUtils.serializeSlightly(value));
    }

    /**
     * 语法：
     * GETSET key value
     * 功能：
     * 将给定 key 的值设为 value ，并返回 key 的旧值 (old value)，当 key 存在但不是字符串类型时，返回一个错误，当key不存在时，返回nil。
     */
    @Override
    public Object getSet(String key, Object value) {
        byte[] bytes = cacheManager.getSet(key.getBytes(), SerializationUtils.serializeSlightly(value));
        return bytes == null ? null : SerializationUtils.deserializeSlightly(bytes);
    }
    //锁定操作 end

    /**
     * 获取Redis链接池
     *
     * @return redis pool
     */
    @Override
    public JedisPool getJedisPool() {
        return cacheManager.getJedisPool();
    }

    @Override
    public JedisCluster getJedisCluster() {
        return cacheManager.getJedisCluster();
    }

    public <T> T blpop(String key, int waitSeconds, Class<T> clazz) {
        return cacheManager.blpop( key,  waitSeconds, clazz);
    }

    @Override
    public Long expire(String key, int secound) {
        return cacheManager.expire(key, secound);
    }
}

