package com.ohaotian.plugin.common.interceptor;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ohaotian.plugin.base.constant.BaseRspConstants;
import com.ohaotian.plugin.base.exception.strategy.ExceptionStrategy;
import com.ohaotian.plugin.common.util.RegexUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.support.DefaultMessageSourceResolvable;
import org.springframework.core.annotation.Order;
import org.springframework.http.MediaType;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author liubq
 * @Description: 统一异常处理器
 */
@Order(-1000)
public class ExceptionResolver implements HandlerExceptionResolver {
    /**
     * 异常日志
     */
    private Logger logger = LoggerFactory.getLogger(ExceptionResolver.class);

    @Value("${rpc.type}")
    private String rpcType;

    private static final String OAP_AUTH="/oap/api/**";

    @Override
    public ModelAndView resolveException(HttpServletRequest request, HttpServletResponse response, Object handler, Exception e) {

        Map<String, Object> result = new HashMap<>();
        String url = request.getRequestURI();
         boolean respCodeFlag = false;
        if(RegexUtils.wildcardEquals("**/**"+OAP_AUTH, url)){
            respCodeFlag = true;
        }
        if(RegexUtils.wildcardEquals("**"+ OAP_AUTH, url)){
            respCodeFlag = true;
        }

        if (e instanceof ExceptionStrategy) {
            ExceptionStrategy exceptionStrategy = (ExceptionStrategy) e;
            String msg = exceptionStrategy.resolverException();
            addResult(result, msg,respCodeFlag);
        } else if (e instanceof ConstraintViolationException) {
            StringBuilder sb =  resolverBindException(e, result);
            addResult(result, sb.toString(),respCodeFlag);
        } else if (e instanceof BindException) {
            BindException be = (BindException) e;
            BindingResult bindingResult = be.getBindingResult();
            if (bindingResult.hasErrors()) {
                addResult(result, bindingResult.getAllErrors().get(0).getDefaultMessage(), respCodeFlag);
            }
        } else if (e instanceof MethodArgumentNotValidException) {
            MethodArgumentNotValidException me = (MethodArgumentNotValidException) e;
            String message = me.getBindingResult().getAllErrors().stream().map(DefaultMessageSourceResolvable::getDefaultMessage).collect(Collectors.joining());
            addResult(result, message,respCodeFlag);
        } else {
            //处理其他异常
            addResult(result, BaseRspConstants.RSP_DESC_ERROR,respCodeFlag);
        }
        logger.error("程序出现异常：" + JSON.toJSONString(result), e);

        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        response.setCharacterEncoding("UTF-8");
        response.setHeader("Cache-Control", "no-cache, must-revalidate");

        if ("SpringCloud".equals(rpcType)) {
            if (e instanceof ExceptionStrategy) {
                ExceptionStrategy exceptionStrategy = (ExceptionStrategy) e;
                try {
                    response.setHeader("errorMsg", URLEncoder.encode(exceptionStrategy.resolverException(), "utf-8"));
                } catch (UnsupportedEncodingException e1) {
                    e1.printStackTrace();
                }
            }
        }

        try {
            response.getWriter().write(JSON.toJSONString(result, SerializerFeature.WriteDateUseDateFormat, SerializerFeature.WriteNullNumberAsZero, SerializerFeature.WriteMapNullValue, SerializerFeature.WriteNullListAsEmpty, SerializerFeature.WriteNullStringAsEmpty, SerializerFeature.WriteNullBooleanAsFalse));
        } catch (IOException ex) {
            logger.error("与客户端通讯异常：" + ex.getMessage(), ex);
        }
        //return new ModelAndView();
//        if ("SpringCloud".equals(rpcType)){
//            return  new ModelAndView();
//        }
//        else{
        return new ModelAndView();
//        }
    }

    /**
     * 处理参数绑定异常
     */
    private StringBuilder resolverBindException(Throwable e, Map<String, Object> result) {
        // 里面嵌了一个ConstraintViolationException
        ConstraintViolationException ve = (ConstraintViolationException) e;
        // 可以拿到一个验证错误详细信息的集合
        Set<ConstraintViolation<?>> violations = ve.getConstraintViolations();
        StringBuilder sb = new StringBuilder();

        int size = violations.size();
        final int[] num = {1};
        violations.forEach(entry -> {
            if (size == num[0]++) {
                sb.append(entry.getMessage());
            } else {
                sb.append(entry.getMessage()).append("并且");
            }

        });
      return sb;
    }


    /**
     * 封装code和msg
     */
    private final void addResult(Map<String, Object> result, final String message,boolean respCodeFlag) {
        if(respCodeFlag){
            result.put("respCode", "8888");
            result.put("respDesc", message);
        }else {
            result.put("code", BaseRspConstants.CODE_FAILUR);
            result.put("message", message);
        }
    }
}
