package com.ohaotian.plugin.file.oss;

import com.aliyun.oss.*;
import com.aliyun.oss.model.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * <br>
 * 标题: Oss工具类<br>
 * 描述: <br>
 * 公司: www.tydic.com<br>
 *
 * @autho zhenweishan
 * @time 2016年10月26日 下午8:54:07
 */
public class OssUtil {

    private static final Logger log = LoggerFactory.getLogger(OssUtil.class);

    private static OSS ossClient;

    /**
     * 获取oss操作客户端
     *
     * @return oss客户端
     */
    public static OSS getOssclient(OssConfig ossConfig) {
        if (ossClient == null) {
            ClientBuilderConfiguration conf = new ClientBuilderConfiguration();
            conf.setSupportCname(false);
             ossClient = new OSSClientBuilder().build(ossConfig.getEndpoint(), ossConfig.getAccessKeyId(), ossConfig.getAccessKeySecret());
        }
        return ossClient;
    }

    /**
     * <br>
     * 适用场景: oss文件上传<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param file
     * @param ossConfig
     * @param uploadPath key
     * @return
     * @autho wesley
     * @time 2016年10月14日 下午6:38:46
     */
    public static void uploadFile(File file, OssConfig ossConfig, String uploadPath) {

        String fileName = file.getName();
        InputStream content = null;
        try {
            OSS ossClient = getOssclient(ossConfig);
            content = new FileInputStream(file);
            // 创建上传Object的Metadata
            ObjectMetadata meta = new ObjectMetadata();
            // 必须设置ContentLength
            meta.setContentLength(file.length());
            // 上传Object.
            ossClient.putObject(ossConfig.getBucketName(), uploadPath + fileName, content, meta);
        } catch (OSSException e) {
            log.error("连接远程oss服务器出错！", e);
        } catch (FileNotFoundException e) {
            log.error("检查文件【" + fileName + "是否存在】", e);
        } catch (ClientException e) {
            log.error("连接oss客户端出错！", e);
        } finally {
            try {
                content.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 通过流式上传<br>
     * 适用场景: <br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param fileName
     * @param ossConfig
     * @param inputStream
     * @param uploadPath
     * @autho liuce
     * @time 2016年11月6日 下午7:25:49
     */
    public static void uploadFileByInputStream(String fileName, OssConfig ossConfig, InputStream inputStream, String uploadPath) {
        try {
            OSS ossClient = getOssclient(ossConfig);
            // 上传Object.
            ossClient.putObject(ossConfig.getBucketName(), uploadPath + fileName, inputStream);
        } catch (OSSException e) {
            log.error("连接远程oss服务器出错！", e);
        } catch (ClientException e) {
            log.error("连接oss客户端出错！", e);
        }
    }


    /**
     * 通过流式上传<br>
     * 适用场景: <br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param filePath
     * @param ossConfig
     * @param inputStream
     * @autho liuce
     * @time 2016年11月6日 下午7:25:49
     */
    public static void uploadFileToOssByInputStream(String filePath, OssConfig ossConfig, InputStream inputStream) {
        try {
            OSS ossClient = getOssclient(ossConfig);
            // 上传Object.
            ossClient.putObject(ossConfig.getBucketName(), filePath, inputStream);
        } catch (OSSException e) {
            log.error("连接远程oss服务器出错！", e);
        } catch (ClientException e) {
            log.error("连接oss客户端出错！", e);
        }
    }

    /**
     * <br>
     * 适用场景: oss下载文件<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param filePath
     * @param ossConfig
     * @return
     * @autho wesley
     * @time 2016年10月17日 下午5:48:07
     */
    public static File downloadFile(String filePath, OssConfig ossConfig) {
        OSS ossClient = getOssclient(ossConfig);
        String fileName = UUID.randomUUID() + filePath.substring(filePath.lastIndexOf("."), filePath.length());
        String dir = System.getProperty("java.io.tmpdir");
        fileName = dir + File.separator + fileName;
        File dirFile = new File(dir);
        if (!dirFile.exists()) {
            dirFile.mkdirs();
        }
        dirFile = new File(fileName);
        try {
            // 新建GetObjectRequest
            GetObjectRequest getObjectRequest = new GetObjectRequest(ossConfig.getBucketName(), filePath);
            // 下载Object到文件
            ossClient.getObject(getObjectRequest, dirFile);
        } catch (Throwable e) {
            log.error("从oss上下载文件,bucketName：" + ossConfig.getBucketName() + ";osskey：" + filePath + ";下载的文件路径:" + fileName, e);
            if (e instanceof RuntimeException) {
                throw (RuntimeException) e;
            } else {
                throw new RuntimeException(e);
            }
        }
        return dirFile;
    }

    public static InputStream downloadByInputStream(String filePath, OssConfig ossConfig) {
        InputStream in = null;
        OSS ossClient = getOssclient(ossConfig);
        try {
            // 下载Object到文件
            OSSObject ossObject = ossClient.getObject(ossConfig.getBucketName(), filePath);
            in = ossObject.getObjectContent();
        } catch (Throwable e) {
            log.error("从oss上下载文件,bucketName：" + ossConfig.getBucketName() + ";osskey：" + filePath, e);
            if (e instanceof RuntimeException) {
                throw (RuntimeException) e;
            } else {
                throw new RuntimeException(e);
            }
        }

        // 外面一点要关流
        return in;
    }

    /**
     * <br>
     * 适用场景: oss删除文件<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param filePath
     * @param ossConfig
     * @autho wesley
     * @time 2016年10月18日 上午10:00:26
     */
    public static void deleteFile(String filePath, OssConfig ossConfig) {
        OSS ossClient = getOssclient(ossConfig);
        try {
            ossClient.deleteObject(ossConfig.getBucketName(), filePath);
        } catch (OSSException e) {
            log.error("Caught an OSSException, which means your request made it to OSS, " + "but was rejected with an error response for some reason.");
        } catch (ClientException e) {
            log.error("Caught an ClientException, which means the client encountered " + "a serious internal problem while trying to communicate with OSS, "
                    + "such as not being able to access the network.");
        } finally {
            ossClient.shutdown();
        }
    }

    public static List<String> listObjects(OssConfig ossConfig, String prefix) {
        OSS ossClient = getOssclient(ossConfig);
        List<String> list;
        try {
            ObjectListing objectListing = ossClient.listObjects(ossConfig.getBucketName(), prefix);
            List<OSSObjectSummary> sums = objectListing.getObjectSummaries();
            list = sums.stream().map(OSSObjectSummary::getKey).collect(Collectors.toList());
        } finally {
            ossClient.shutdown();
        }
        return list;
    }
}
