package com.ohaotian.plugin.file.fastdfs;

import com.github.tobato.fastdfs.domain.fdfs.StorePath;
import com.github.tobato.fastdfs.exception.FdfsUnsupportStorePathException;
import com.github.tobato.fastdfs.service.FastFileStorageClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.io.*;

/**
 * 标题：FastDFS工具类
 * <p>
 * 说明：基于tobato客户端的FastDFS工具类
 * <br>
 * 时间：2018/07/09<br>
 * 版权：copyright © 2018 www.tydic.com Inc. All rights reserved.
 * </p>
 *
 * @author ZhangCheng
 */
public class FastdfsUtil {

    private static final Logger logger = LoggerFactory.getLogger(FastdfsUtil.class);
    private static FastFileStorageClient storageClient;

    // 初始化客户端（建议通过Spring注入）
    public static void setStorageClient(FastFileStorageClient client) {
        storageClient = client;
    }

    public static FastdfsFileInfo uploadFile(String fileName, FastdfsConfig config, InputStream fis, String uploadPath) {
        logger.info("plugin-file: Upload file to FastDFS File Name [{}]", fileName);
        long startTime = System.currentTimeMillis();

        try {
            // 获取文件扩展名
            String fileExtName = getExtByFileName(fileName);
            // 上传文件
            StorePath storePath = storageClient.uploadFile(
                    fis,
                    fis.available(),
                    fileExtName,
                    null);

            logger.info("plugin-file: upload_file time used:" + (System.currentTimeMillis() - startTime) + " ms");
            logger.info("plugin-file: upload file successfully!!!" +
                    "group_name:" + storePath.getGroup() +
                    ", remoteFileName:" + storePath.getPath());

            return new FastdfsFileInfo(storePath.getGroup(), storePath.getPath());
        } catch (IOException e) {
            logger.error("plugin-file: IO Exception when uploading the file:" + fileName, e);
        } catch (Exception e) {
            logger.error("plugin-file: Exception when uploading the file:" + fileName, e);
        }

        return new FastdfsFileInfo();
    }

    public static File downloadFile(FastdfsFileInfo fastdfsFileInfo, FastdfsConfig config) {
        String dir = System.getProperty("java.io.tmpdir");
        File dirFile = new File(dir);
        if (!dirFile.exists()) {
            dirFile.mkdirs();
        }
        String fileName = dir + "/" + fastdfsFileInfo.getFileName();
        logger.debug("plugin-file: Download FastDFS File To Tmpdir File Name [{}]", fileName);

        InputStream inputStream = downloadByInputStream(fastdfsFileInfo, config);
        if (inputStream == null) {
            return null;
        }

        File file = new File(fileName);
        File parentDir = new File(file.getParent());
        if (!parentDir.exists()) {
            parentDir.mkdirs();
        }

        try (FileOutputStream fos = new FileOutputStream(file)) {
            byte[] b = new byte[1024];
            int len;
            while ((len = inputStream.read(b)) != -1) {
                fos.write(b, 0, len);
            }
        } catch (Exception e) {
            logger.error("plugin-file: Download FastDFS File Error: " + e.getMessage(), e);
            return null;
        } finally {
            try {
                inputStream.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return file;
    }

    public static InputStream downloadByInputStream(FastdfsFileInfo fastdfsFileInfo, FastdfsConfig config) {
        try {
            StorePath storePath = parseStorePath(fastdfsFileInfo.getGroupName(), fastdfsFileInfo.getFileName());
            return storageClient.downloadFile(
                    storePath.getGroup(),
                    storePath.getPath(),
                    inputStream -> {
                        // 复制输入流
                        ByteArrayOutputStream baos = new ByteArrayOutputStream();
                        byte[] buffer = new byte[1024];
                        int len;
                        while ((len = inputStream.read(buffer)) != -1) {
                            baos.write(buffer, 0, len);
                        }
                        return new ByteArrayInputStream(baos.toByteArray());
                    });
        } catch (Exception e) {
            logger.error("plugin-file: Get File from Fast DFS failed", e);
        }
        return null;
    }

    // 解析存储路径
    private static StorePath parseStorePath(String groupName, String filePath) {
        if (StringUtils.isEmpty(groupName) || StringUtils.isEmpty(filePath)) {
            throw new FdfsUnsupportStorePathException("解析文件路径失败");
        }
        return new StorePath(groupName, filePath);
    }

    public static String getExtByFileName(String fileName) {
        try {
            return fileName.substring(fileName.lastIndexOf(".") + 1);
        } catch (Exception e) {
            throw new RuntimeException("plugin-file Upload File To FastDFS Error:Get File Ext By File Name Error.Please Check File Name [" + fileName + "]", e);
        }
    }
}