package com.ohaotian.plugin.mock;

import com.ohaotian.plugin.mock.annotation.Autoinject;
import com.ohaotian.plugin.mock.annotation.InjectType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.context.ApplicationContext;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;

/** <br>
 * 标题: @AutoMock 注解处理器<br>
 * 描述: 实现@AutoMock注解的功能<br>
 * 公司: www.tydic.com<br>
 *
 * @author zhangcheng
 * @date 2018/03/05
 */
public class AutoinjectHandle implements AnnotationHandle {

    private static Logger logger = LoggerFactory.getLogger(AutoinjectHandle.class);

    @Override
    public boolean hasExist(Field field) {
        if(field == null){
            return false;
        }
        for(Annotation annotation : field.getDeclaredAnnotations()){
            if(annotation instanceof Autoinject){
                return true;
            }
        }
        return false;
    }

    @Override
    public Object invok(Object bean, String beanName,Field field,ApplicationContext applicationContext) throws BeansException {

        Autoinject autoMock = field.getDeclaredAnnotation(Autoinject.class);

        //如果AutoType.MOCK，且mock类存在时，将mock类注入
        if (InjectType.MOCK == autoMock.value() && this.hasBeanMock(field)){
            this.invokLog(beanName, field, autoMock, InjectType.MOCK);
            return this.getBeanByMock(bean,field);
        }

        //如果AutoType.WIRED，
        if (InjectType.WIRED == autoMock.value()){
            //If exists the implement of the bean, inject the instance of this implement.
            Object beanImpl = this.findBeanImpl(field, applicationContext);
            if (null != beanImpl) {
                this.invokLog(beanName, field, autoMock, InjectType.WIRED);
                return this.getBeanByImpl(bean, field, beanImpl);
            }
            //mock类存在时，将impl类注入
            if(this.hasBeanMock(field)) {
                this.invokLog(beanName, field, autoMock, InjectType.MOCK);
                return this.getBeanByMock(bean,field);
            }
        }

        if (InjectType.MOCK == autoMock.value()){
            throw new BeanInitializationException("Mock bean exception: 名称为 '"+beanName+"' 的字段 '"+field.getName()+"' mock包下没有impl类");
        }

        throw new BeanInitializationException("Mock bean exception: 名称为 '"+beanName+"' 的字段 '"+field.getName()+"' mock包和impl包下都没有impl类");
    }

    private void invokLog(String beanName, Field field, Autoinject autoMock, InjectType injectType) {
        if (this.logger.isInfoEnabled()) {
            this.logger.info("完成重新装配Bean名称为 '" + beanName + "' 的字段名称为 '" + field.getName() + "' 装配类型为 '" + autoMock.value() + "' 实际装配类型为 '"+ injectType +"'");
        }
    }

    /** 获取Bean的impl实例从IOC容器中 */
    private Object getBeanByImpl(Object bean, Field field, Object beanImpl) throws BeansException {
        try {
            field.setAccessible(true);
            field.set(bean, beanImpl);
            return bean;
        } catch (Exception e) {
            throw new BeanInitializationException("Bean初始化异常，" + e.getMessage());
        }
    }

    /** 获取Bean的mock实例使用反射创建 */
    private Object getBeanByMock(Object bean, Field field){
        try {
            String mockPakcage = field.getType().getPackage().getName()+".mock";
            String mockName = field.getType().getSimpleName()+"Impl";
            String clazzName = mockPakcage + "."+ mockName;
            Class<? extends Object> clazz = Class.forName(clazzName);
            field.setAccessible(true);
            field.set(bean,clazz.newInstance());
            return bean;
        }catch (Exception e){
            throw new BeanInitializationException("Bean初始化异常，" + e.getMessage());
        }
    }

    /**
     * 判断Bean对应的impl包下是否有impl类在IOC容器中
     */
    private Object findBeanImpl(Field field, ApplicationContext applicationContext) {
        Object bean = null;
        try {
            bean = applicationContext.getBean(field.getName() + "Impl");
        } catch (BeansException e) {
            try {
                bean = applicationContext.getParent().getBean(field.getName() + "Impl");
            } catch (BeansException e1) {
                try {
                    bean = applicationContext.getParent().getBean(field.getName());
                } catch (BeansException e2) {
                    logger.error("Can't find the implement of {}", field.getName());
                }
            }
        }
        return bean;
    }

    /**
     * 判断Bean对应的mock包下是否有impl类在API工程中
     * @param field
     * @return
     */
    private boolean hasBeanMock(Field field){
        try {
            String mockPakcage = field.getType().getPackage().getName()+".mock";
            String mockName = field.getType().getSimpleName()+"Impl";
            String clazzName = mockPakcage + "."+ mockName;
            Class<? extends Object> clazz = Class.forName(clazzName);
            clazz.newInstance();
            return true;
        }catch (Exception e){
            return false;
        }
    }
}
