package com.ohaotian.plugin.mq.proxy.ext.redismq;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.ohaotian.plugin.mq.proxy.CacheStore;
import com.ohaotian.plugin.mq.proxy.ProxyMessage;
import com.ohaotian.plugin.mq.proxy.ext.ProxyMqTransactionChecker;
import com.ohaotian.plugin.mq.proxy.status.ProxyTransactionStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.Timer;
import java.util.TimerTask;

public class RedisMqTransactionCheckListener implements RedisTransactionListener {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private ProxyMqTransactionChecker mqTransactionChecker = new ProxyMqTransactionChecker();
    private final JedisPool jedisPool;
    private String defaultQueue;
    private Timer timer = new Timer();
    private final RedisMqMessageSender messageSender;
    private final ThreadLocal<ObjectMapper> objectMapperThreadLocal = new ThreadLocal<ObjectMapper>() {
        @Override
        protected ObjectMapper initialValue() {
            return new ObjectMapper();
        }
    };

    class QueueCheckTask extends TimerTask {
        private final JedisPool jedisPool;
        private final String checkQueue;
        private final String nextQueue;
        QueueCheckTask(JedisPool jedisPool, String checkQueue, String nextQueue) {
            this.jedisPool = jedisPool;
            this.checkQueue = checkQueue;
            this.nextQueue = nextQueue;
        }

        @Override
        public void run() {
            for (int i = 0; i < 1000; i++) {
                ProxyMessage message = popCheckList(checkQueue);
                if (message == null) {
                    break;
                }
                try {
                    ProxyTransactionStatus transactionStatus = checkLocalTransactionState(message);
                    if (logger.isDebugEnabled()) {
                        logger.debug("Check[" + message + "] status[" + transactionStatus + "]");
                    }
                    if (transactionStatus == ProxyTransactionStatus.UNKNOW) {
                        if (nextQueue != null) {
                            pushCheckList(nextQueue, message);
                        } else {
                            logger.error("max check reached for message[" + message + "]");
                        }
                    } else if (transactionStatus == ProxyTransactionStatus.COMMIT) {
                        messageSender.send(message);
                    }
                } catch (Throwable e) {
                    if (nextQueue != null) {
                        pushCheckList(nextQueue, message);
                    } else {
                        logger.error("max check reached for message[" + message + "]");
                    }
                }
            }

        }
    }

    public void startup() {
        int period = 10000;
        int maxQueue = 16;
        for (int i = 1; i <= maxQueue; i++) {
            String checkQueue = "QUEUE-" + i;
            if (i == 1) {
                defaultQueue = checkQueue;
            }
            timer.schedule(new QueueCheckTask(jedisPool, checkQueue, i == maxQueue ? null : "QUEUE-" + (i + 1)), period, period * i);
        }
    }

    public void shtudown() {
        timer.cancel();
    }

    public RedisMqTransactionCheckListener(JedisPool jedisPool, RedisMqMessageSender messageSender) {
        this.jedisPool = jedisPool;
        this.messageSender = messageSender;
    }

    public void addCheckList(ProxyMessage message) {
        this.pushCheckList(defaultQueue, message);
    }

    public void pushCheckList(String checkQueue, ProxyMessage message) {
        Jedis jedis = null;
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("push [" + message + "] to QUEUE[" + checkQueue + "]");
            }
            jedis = jedisPool.getResource();
            jedis.lpush(checkQueue, objectMapperThreadLocal.get().writeValueAsString(message));
        } catch (Exception e) {
            throw new IllegalArgumentException("add check queue[" + checkQueue + "] error", e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    public ProxyMessage popCheckList(String checkQueue) {
        Jedis jedis = null;
        ProxyMessage message = null;
        try {
            jedis = jedisPool.getResource();
            String value = jedis.lpop(checkQueue);
            if (value != null) {
                message = objectMapperThreadLocal.get().readValue(value, ProxyMessage.class);
            }
        } catch (Exception e) {
            throw new IllegalArgumentException("add check queue[" + checkQueue + "] error", e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return message;
    }

    public ProxyTransactionStatus checkLocalTransactionState(ProxyMessage msg) {
        return mqTransactionChecker.check(msg.getMessageId(), null /*TODO: add key*/, msg.getSubject(), msg.getTag());
    }

    public void setCacheStore(CacheStore cacheStore) {
        mqTransactionChecker.setCacheStore(cacheStore);
    }
}
