package com.ohaotian.plugin.mq.proxy.ext.rocketmq;

import com.ohaotian.plugin.mq.proxy.*;
import com.ohaotian.plugin.mq.proxy.callback.ProxyLocalTransactionExecuter;
import com.ohaotian.plugin.mq.proxy.callback.ProxySendCallback;
import com.ohaotian.plugin.mq.proxy.config.ApolloConfigVO;
import com.ohaotian.plugin.mq.proxy.ext.ProxyMessageProducerEx;
import com.ohaotian.plugin.mq.proxy.ext.ProxyMqTransactionChecker;
import com.ohaotian.plugin.mq.proxy.internal.ProxyMessageConfig;
import com.ohaotian.plugin.mq.proxy.internal.ProxyMessageException;
import com.ohaotian.plugin.mq.proxy.status.ProxyTransactionStatus;
import com.ohaotian.plugin.mq.proxy.*;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.*;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.remoting.common.RemotingHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.util.CollectionUtils;

import java.io.UnsupportedEncodingException;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;

public class RocketMqMessageSender implements ProxyMessageProducerEx, ApplicationContextAware {
    private final Logger logger = LoggerFactory.getLogger(this.getClass().getName());
    private DefaultMQProducer producer;
    private TransactionMQProducer transProductor;
    private final String subject;
    private final Map<ProxyMessageType, Set<String>> typeTagsMapping;
    private RocketMqTransactionCheckListener transactionCheckerListener;
    private CacheStore cacheStore;

    private Properties mergedProps = new Properties();
    @Autowired
    private static ApolloConfigVO apolloConfigVO;
    private final ProxyMessageConfig messageConfig;

    public RocketMqMessageSender(ProxyMessageConfig messageConfig, Map<ProxyMessageType, Set<String>> typeTagsMapping) {
        this.subject = messageConfig.getSubject();
        this.messageConfig = messageConfig;
        this.typeTagsMapping = typeTagsMapping;
    }

    public ProxySendResult send(ProxyMessage message) {
        checkStatus(message, ProxyMessageType.SYNCHRONIZATION);
        ProxySendResult proxySendResult;
        try {
            SendResult sendResult = producer.send(getMessage(message));
            proxySendResult = getProxySendResult(sendResult);
        } catch (Exception e) {
            throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
        }
        return  proxySendResult;
    }

    public ProxySendResult sendInTransaction(final ProxyMessage proxyMessage, final ProxyLocalTransactionExecuter localTransactionExecuter, final Object arg) {
        checkStatus(proxyMessage, ProxyMessageType.TRANSACTION);
        ProxySendResult proxySendResult;
        Message message = getMessage(proxyMessage);
        try {
            SendResult sendResult = transProductor.sendMessageInTransaction(message, new LocalTransactionExecuter() {
                public LocalTransactionState executeLocalTransactionBranch(Message message, Object arg) {
                    ProxyTransactionStatus transactionStatus = localTransactionExecuter.exec(proxyMessage, arg);
                    if (transactionStatus == ProxyTransactionStatus.COMMIT) {
                        return LocalTransactionState.COMMIT_MESSAGE;
                    } else if (transactionStatus == ProxyTransactionStatus.ROLLBACK) {
                        return LocalTransactionState.ROLLBACK_MESSAGE;
                    } else if (transactionStatus == ProxyTransactionStatus.UNKNOW) {
                        return LocalTransactionState.UNKNOW;
                    }
                    return LocalTransactionState.UNKNOW;
                }
            }, arg);
            proxySendResult = getProxySendResult(sendResult);
            cacheStore.set(sendResult.getMsgId(), ProxyTransactionStatus.COMMIT.toString(), ProxyMqTransactionChecker.MQ_TRAN_CHECK_EXPIRE);
            if (logger.isDebugEnabled()) {
                logger.debug("send tran msg msgId={" + proxySendResult.getMsgId() + "}  topic=" + proxyMessage.getSubject() + " tag=" + proxyMessage.getTag() + "  body={" + proxyMessage.getContent() + "}  SendResult={" + proxySendResult.getStatus() + "}");
            }
        } catch (Exception e) {
            // TODO: Can't get msgId
//            cacheStore.set(message.getMsgID(), ProxyTransactionStatus.ROLLBACK.toString(), ProxyMqTransactionChecker.MQ_TRAN_CHECK_EXPIRE);
            throw new ProxyMessageException("send subject[" + proxyMessage.getSubject() + "] tag[" + proxyMessage.getTag() + "] error", e);
        }
        return  proxySendResult;
    }

    private ProxySendResult getProxySendResult(SendResult sendResult) {
        ProxySendResult proxySendResult;
        proxySendResult = new ProxySendResult();
        proxySendResult.setMsgId(sendResult.getMsgId());
        proxySendResult.setStatus(sendResult.getSendStatus().name());
        return proxySendResult;
    }

    private Message getMessage(ProxyMessage message) {
        Message sendMessage;
        try {
            sendMessage = new Message(message.getSubject(), message.getTag(), message.getContent().getBytes(RemotingHelper.DEFAULT_CHARSET));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Convert rocketmq message error", e);
        }
        return sendMessage;
    }

    public void send(ProxyMessage message, final ProxySendCallback sendCallback) {
        checkStatus(message, ProxyMessageType.ASYNCHRONOUS);
        try {
            producer.send(getMessage(message), new SendCallback() {
                public void onSuccess(SendResult sendResult) {
                    if (sendCallback != null) {
                        sendCallback.onSuccess(getProxySendResult(sendResult));
                    }
                }
                public void onException(Throwable e) {
                    if (sendCallback != null) {
                        sendCallback.onException(new ProxyExceptionContext(e));
                    }
                }
            });
        } catch (Throwable e) {
            if (sendCallback != null) {
                sendCallback.onException(new ProxyExceptionContext(e));
            } else {
                throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
            }
        }
    }

    private void checkStatus(ProxyMessage message, ProxyMessageType proxyMessageType) {
        //TODO: checkStatus
        if(true){
            return;
        }
        if (!this.subject.equals(message.getSubject())) {
            throw new IllegalArgumentException("Unsupported subject[" + message.getSubject() + "].Supported subject[" + this.subject + "].");
        }
        Set<String> allowedTags = typeTagsMapping.get(proxyMessageType);
        if (allowedTags == null || (!allowedTags.contains(message.getTag()) && !allowedTags.contains("*"))) {
            throw new IllegalStateException("Subject[" + message.getSubject() + "]unsupported tag[" + message.getTag() + "].Supported tags " + (allowedTags == null ? "[]" : allowedTags) + " for messageType[" + proxyMessageType + "].");
        }
    }

    public void sendOneway(ProxyMessage message) {
        checkStatus(message, ProxyMessageType.ONEWAY);
        try {
            producer.sendOneway(getMessage(message));
        } catch (Throwable e) {
            throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
        }
    }

    public void startup() {
        if (typeTagsMapping.containsKey(ProxyMessageType.TRANSACTION)) {
            transProductor = new TransactionMQProducer(((DefaultProxyMessageConfig) messageConfig).getId());
            this.transactionCheckerListener = new RocketMqTransactionCheckListener();
            this.transactionCheckerListener.setCacheStore(this.cacheStore);
        } else {
            transProductor = null;
        }
        boolean createCommonProductor = false;
        if (typeTagsMapping.containsKey(ProxyMessageType.SYNCHRONIZATION)) {
            createCommonProductor = true;
        }
        if (!createCommonProductor && typeTagsMapping.containsKey(ProxyMessageType.ASYNCHRONOUS)) {
            createCommonProductor = true;
        }
        if (!createCommonProductor && typeTagsMapping.containsKey(ProxyMessageType.ONEWAY)) {
            createCommonProductor = true;
        }
        if (!createCommonProductor && typeTagsMapping.containsKey(ProxyMessageType.ORDERED)) {
            createCommonProductor = true;
        }
        if (createCommonProductor) {
            producer = new DefaultMQProducer(((DefaultProxyMessageConfig) messageConfig).getId());
        } else {
            producer = null;
        }
        try {
            if (this.producer != null) {
                this.producer.setNamesrvAddr(messageConfig.getProperties().getProperty("mq.rocket.namesrvaddr", apolloConfigVO.getRocketNamesrvaddr()));
                this.producer.setInstanceName(UUID.randomUUID().toString());
                this.producer.setRetryTimesWhenSendFailed(0);
                this.producer.setRetryTimesWhenSendAsyncFailed(0);
                this.producer.start();
            }
            if (this.transProductor != null) {
                this.transProductor.setNamesrvAddr(messageConfig.getProperties().getProperty("mq.rocket.namesrvaddr", apolloConfigVO.getRocketNamesrvaddr()));
                this.transProductor.setInstanceName(UUID.randomUUID().toString());
                this.transProductor.setTransactionCheckListener(this.transactionCheckerListener);
                this.transProductor.setCheckThreadPoolMinSize(5);
                this.transProductor.setCheckThreadPoolMaxSize(50);
                this.transProductor.setCheckRequestHoldMax(2000);
                this.transProductor.start();
            }
        } catch (MQClientException e) {
            throw new ProxyMessageException("start product[" + this.subject + "] error", e);
        }
    }

    public void shutdown() {
        if (this.producer != null) {
            this.producer.shutdown();
        }
        if (this.transProductor != null) {
            this.transProductor.shutdown();
        }
    }

    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.cacheStore = applicationContext.getBean(CacheStore.class);
        Map<String, Properties> propsMapping = applicationContext.getBeansOfType(Properties.class);
        if (propsMapping != null) {
            for (Properties props : propsMapping.values()) {
                CollectionUtils.mergePropertiesIntoMap(props, mergedProps);
            }
        }
    }
}
