package com.ohaotian.plugin.mq.proxy.support;

import com.ohaotian.plugin.mq.proxy.DefaultProxyMessageConfig;
import com.ohaotian.plugin.mq.proxy.RedisCacheStore;
import com.ohaotian.plugin.mq.proxy.constants.LogUtils;
import com.ohaotian.plugin.mq.proxy.constants.MessageConfigUtils;
import com.ohaotian.plugin.mq.proxy.impl.ProxyMessageRegister;
import com.ohaotian.plugin.mq.proxy.impl.ProxyProducerFactoryBean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.annotation.AnnotationAttributes;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternUtils;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.core.type.classreading.MetadataReader;
import org.springframework.core.type.classreading.MetadataReaderFactory;
import org.springframework.core.type.classreading.SimpleMetadataReaderFactory;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static org.springframework.core.io.support.ResourcePatternResolver.CLASSPATH_ALL_URL_PREFIX;

/**
 * 标题：启用MQ插件配置
 * <p>
 * 说明：自动扫描消费者并注册为Bean，自动注册生产者
 * <br>
 * 时间：2018/07/25<br>
 * 版权：copyright © 2018 www.tydic.com Inc. All rights reserved.
 * </p>
 *
 * @author ZhangCheng
 */
@Configuration
public class PluginMqProxyConfiguration implements ImportBeanDefinitionRegistrar{

    private static final Logger logger = LoggerFactory.getLogger(PluginMqProxyConfiguration.class);

    private static final String RESOURCE_PATTERN = "**/*.class";

    public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
        AnnotationAttributes attributes = AnnotationAttributes.fromMap(importingClassMetadata.getAnnotationAttributes(EnablePluginMqProxy.class.getName()));
        String[] basePackages  = attributes.getStringArray("value");
        if (ObjectUtils.isEmpty(basePackages)){
            basePackages  = attributes.getStringArray("basePackages");
        }
        if (ObjectUtils.isEmpty(basePackages)) {
            basePackages = new String[] {ClassUtils.getPackageName(importingClassMetadata.getClassName())};
        }

        RootBeanDefinition redisCacheStore = new RootBeanDefinition(RedisCacheStore.class);
        registry.registerBeanDefinition(MessageConfigUtils.BeanNames.REDIS_CACHE_STORE,redisCacheStore);

        // 加载代理消息注册机
        this.registerRegister(registry);
        // 加载代理消息生产者
        this.registerProducer(registry);
        // 加载代理消息消费者
        this.registerConsumer(registry,basePackages);
    }

    private void registerRegister(BeanDefinitionRegistry registry) {
        RootBeanDefinition proxyMessageRegister = new RootBeanDefinition(ProxyMessageRegister.class);
        MutablePropertyValues propertyValues = new MutablePropertyValues();
        propertyValues.add(MessageConfigUtils.Fields.STRATEGY,MessageConfigUtils.getElKeyByKey(MessageConfigUtils.Keys.Mq.STRATEGY));
        propertyValues.add(MessageConfigUtils.Fields.ENABLE,MessageConfigUtils.getElKeyByKey(MessageConfigUtils.Keys.Mq.ENABLE));
        proxyMessageRegister.setPropertyValues(propertyValues);
        proxyMessageRegister.setInitMethodName("startup");
        proxyMessageRegister.setDestroyMethodName("shutdown");
        registry.registerBeanDefinition(MessageConfigUtils.BeanNames.PROXY_MESSAGE_REGISTER,proxyMessageRegister);

        LogUtils.debug(logger,"Loaded bean definitions from class ["+ProxyMessageRegister.class.getName()+']');
    }

    private void registerConsumer(BeanDefinitionRegistry registry, String[] basePackages) {
        LogUtils.debug(logger,"Started scan mq proxy consumer",null);
        List<Class<?>> consumerList = new ArrayList<Class<?>>(16);
        for (String basePackage : basePackages) {
            try {
                LogUtils.debug(logger,"Scan mq proxy consumer on this package ["+basePackage+"]");
                consumerList.addAll(this.findConsumerClass(basePackage));
            } catch (IOException e) {
                LogUtils.warn(logger,"Scan this package error",basePackage);
            }
        }

        if (CollectionUtils.isEmpty(consumerList)){
            LogUtils.debug(logger,"Ended scan mq proxy consumer no consumer");
            return;
        }else {
            LogUtils.debug(logger,"Ended scan mq proxy consumer on consumer count ["+consumerList.size()+"]");
        }

        RootBeanDefinition consumerDefinition;
        for (Class<?> consumerClass : consumerList) {
            consumerDefinition = new RootBeanDefinition(consumerClass);
            MutablePropertyValues propertyValues = new MutablePropertyValues();
            propertyValues.add(MessageConfigUtils.Fields.ID,this.getConsumerConfig(consumerClass, MessageConfigUtils.Fields.ID));
            propertyValues.add(MessageConfigUtils.Fields.SUBJECT,this.getConsumerConfig(consumerClass, MessageConfigUtils.Fields.SUBJECT));
            propertyValues.add(MessageConfigUtils.Fields.TAGS,this.getConsumerConfig(consumerClass, MessageConfigUtils.Fields.TAGS));
            consumerDefinition.setPropertyValues(propertyValues);
            registry.registerBeanDefinition(this.getBeanNameByClass(consumerClass),consumerDefinition);
            LogUtils.debug(logger,"Loaded bean definitions from class ["+consumerClass.getName()+']');
        }
    }

    private Object getConsumerConfig(Class<?> consumerClass, String key) {
        for (Annotation annotation : consumerClass.getAnnotations()) {
            if (!(annotation instanceof MqProxyConsumer)){
                continue;
            }
            MqProxyConsumer consumer = (MqProxyConsumer)annotation;
            if (MessageConfigUtils.Fields.ID.equals(key)){
                String value = consumer.id();
                if (StringUtils.isEmpty(value)){
                    throw new IllegalArgumentException(LogUtils.PLUGIN_NAME+"消费者类["+consumerClass.getName()+"]的@MqProxyConsumer注解的id未配置");
                }
                return value;
            }
            if (MessageConfigUtils.Fields.SUBJECT.equals(key)){
                String value = consumer.subject();
                if (StringUtils.isEmpty(value)){
                    throw new IllegalArgumentException(LogUtils.PLUGIN_NAME+"消费者类["+consumerClass.getName()+"]的@MqProxyConsumer注解的subject未配置");
                }
                return value;
            }
            if (MessageConfigUtils.Fields.TAGS.equals(key)){
                String[] value = consumer.tags();
                if (value.length == 0){
                    throw new IllegalArgumentException(LogUtils.PLUGIN_NAME+"消费者类["+consumerClass.getName()+"]的@MqProxyConsumer注解的tags未配置");
                }
                return Arrays.asList(value);
            }
        }
        return null;
    }

    private List<Class<?>> findConsumerClass(String basePackage)throws IOException{
        List<Class<?>> consumerList = new ArrayList<Class<?>>(16);
        String packageSearchPath = CLASSPATH_ALL_URL_PREFIX + this.replaceDotByDelimiter(basePackage) + "/" + RESOURCE_PATTERN;
        ResourceLoader resourceLoader = new DefaultResourceLoader();
        MetadataReaderFactory readerFactory = new SimpleMetadataReaderFactory(resourceLoader);
        Resource[] resources = ResourcePatternUtils.getResourcePatternResolver(resourceLoader).getResources(packageSearchPath);
        for (Resource resource : resources) {
            MetadataReader reader = readerFactory.getMetadataReader(resource);
                Class<?> targetClass = this.getClassTypeByClassName(reader.getClassMetadata().getClassName());
            if (targetClass == null) {
                continue;
            }
            if (this.isMqProxyConsumer(targetClass)){
                consumerList.add(targetClass);
            }
        }
        return consumerList;
    }

    private void registerProducer(BeanDefinitionRegistry registry) {
        RootBeanDefinition producer = new RootBeanDefinition(ProxyProducerFactoryBean.class);
        DefaultProxyMessageConfig config = new DefaultProxyMessageConfig();
        config.setId(MessageConfigUtils.DEFAULT_PRODUCER_ID);

        MutablePropertyValues propertyValues = new MutablePropertyValues();
        propertyValues.add(MessageConfigUtils.Fields.MESSAGE_CONFIG,config);
        producer.setPropertyValues(propertyValues);
        producer.setInitMethodName("startup");
        producer.setDestroyMethodName("shutdown");
        registry.registerBeanDefinition(MessageConfigUtils.BeanNames.PRODUCER,producer);
        LogUtils.debug(logger,"Loaded bean definitions from class ["+ProxyProducerFactoryBean.class.getName()+']');
    }

    private String replaceDotByDelimiter(String path) {
        return StringUtils.replace(path, ".", "/");
    }

    private Class<?> getClassTypeByClassName(String className) {
        Class<?> clazz = null;
        try {
            clazz = ClassUtils.forName(className, this.getClass().getClassLoader());
        } catch (ClassNotFoundException e) {
            LogUtils.debug(logger,"未找到指定消费者", className);
        }
        return clazz;
    }

    private String getBeanNameByClass(Class<?> consumerClass) {
        String name = consumerClass.getSimpleName();
        if(Character.isLowerCase(name.charAt(0))){
            return name;
        }
        else{
            return (new StringBuilder()).append(Character.toLowerCase(name.charAt(0))).append(name.substring(1)).toString();
        }
    }

    private boolean isMqProxyConsumer(Class<?> targetClass) {
        for (Annotation annotation : targetClass.getAnnotations()) {
            if (annotation instanceof MqProxyConsumer){
                LogUtils.debug(logger,"Scan mq proxy consumer find consumer ["+targetClass.getName()+"]");
                return true;
            }
        }
        return false;
    }
}
