package com.ohaotian.plugin.mq.proxy.ext.redismq;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.ohaotian.plugin.mq.proxy.*;
import com.ohaotian.plugin.mq.proxy.callback.ProxyLocalTransactionExecuter;
import com.ohaotian.plugin.mq.proxy.callback.ProxySendCallback;
import com.ohaotian.plugin.mq.proxy.ext.ProxyMessageProducerEx;
import com.ohaotian.plugin.mq.proxy.ext.ProxyMqTransactionChecker;
import com.ohaotian.plugin.mq.proxy.internal.ProxyMessageConfig;
import com.ohaotian.plugin.mq.proxy.internal.ProxyMessageException;
import com.ohaotian.plugin.mq.proxy.status.ProxyTransactionStatus;
import com.ohaotian.plugin.mq.proxy.*;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.remoting.common.RemotingHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.util.CollectionUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Protocol;
import redis.clients.jedis.Transaction;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

public class RedisMqMessageSender implements ProxyMessageProducerEx, ApplicationContextAware {
    private final Logger logger = LoggerFactory.getLogger(this.getClass().getName());
    private final String subject;
    private final Map<ProxyMessageType, Set<String>> typeTagsMapping;
    private CacheStore cacheStore;

    private final ProxyMessageConfig messageConfig;
    private final Properties mergedProps = new Properties();
    private JedisPool jedisPool;
    private RedisMqTransactionCheckListener redisMqTransactionCheckListener;

    private final ThreadLocal<ObjectMapper> objectMapperThreadLocal = new ThreadLocal<ObjectMapper>() {
        @Override
        protected ObjectMapper initialValue() {
            return new ObjectMapper();
        }
    };

    public RedisMqMessageSender(ProxyMessageConfig messageConfig, Map<ProxyMessageType, Set<String>> typeTagsMapping) {
        this.subject = messageConfig.getSubject();
        this.typeTagsMapping = typeTagsMapping;
        this.messageConfig = messageConfig;
    }

    private void setOneConsumer(Jedis jedis, ProxyMessage message) {
        Transaction tx = jedis.multi();
        tx.set(message.getMessageId(), "1");
        tx.expire(message.getMessageId(), 10);
        tx.exec();
    }

    public ProxySendResult send(ProxyMessage message) {
        this.setMessageId(message);
        checkStatus(message, ProxyMessageType.SYNCHRONIZATION);
        ProxySendResult proxySendResult;
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            setOneConsumer(jedis, message);
            jedis.publish(getChannel(message), objectMapperThreadLocal.get().writeValueAsString(message));
            proxySendResult = makeSendSuccessResult(message);
        } catch (Exception e) {
            throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return proxySendResult;
    }

    public ProxySendResult sendInTransaction(final ProxyMessage message, final ProxyLocalTransactionExecuter localTransactionExecuter, final Object arg) {
        this.setMessageId(message);
        checkStatus(message, ProxyMessageType.TRANSACTION);
        ProxySendResult proxySendResult = makeSendSuccessResult(message);
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            setOneConsumer(jedis, message);
            ProxyTransactionStatus transactionStatus = localTransactionExecuter.exec(message, arg);
            switch (transactionStatus) {
                case COMMIT:
                    jedis.publish(getChannel(message), objectMapperThreadLocal.get().writeValueAsString(message));
                    break;
                case ROLLBACK:
                    break;
                case UNKNOW:
                    this.redisMqTransactionCheckListener.addCheckList(message);
                    break;
                default:
                    break;
            }
            cacheStore.set(proxySendResult.getMsgId(), ProxyTransactionStatus.COMMIT.toString(), ProxyMqTransactionChecker.MQ_TRAN_CHECK_EXPIRE);
            if (logger.isDebugEnabled()) {
                logger.debug("send tran msg msgId={" + proxySendResult.getMsgId() + "}  topic=" + message.getSubject() + " tag=" + message.getTag() + "  body={" + message.getContent() + "}  SendResult={" + proxySendResult.getStatus() + "}");
            }
        } catch (Exception e) {
            cacheStore.set(message.getMessageId(), ProxyTransactionStatus.ROLLBACK.toString(), ProxyMqTransactionChecker.MQ_TRAN_CHECK_EXPIRE);
            throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return proxySendResult;
    }

    private Message getMessage(ProxyMessage message) {
        Message sendMessage;
        try {
            sendMessage = new Message(message.getSubject(), message.getTag(), message.getContent().getBytes(RemotingHelper.DEFAULT_CHARSET));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Convert rocketmq message error", e);
        }
        return sendMessage;
    }

    public void send(ProxyMessage message, final ProxySendCallback sendCallback) {
        this.setMessageId(message);
        checkStatus(message, ProxyMessageType.ASYNCHRONOUS);
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            setOneConsumer(jedis, message);
            jedis.publish(getChannel(message), objectMapperThreadLocal.get().writeValueAsString(message));
            sendCallback.onSuccess(makeSendSuccessResult(message));
        } catch (Throwable e) {
            if (sendCallback != null) {
                sendCallback.onException(new ProxyExceptionContext(e));
            } else {
                throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
            }
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    private ProxySendResult makeSendSuccessResult(ProxyMessage message) {
        ProxySendResult proxySendResult = new ProxySendResult();
        proxySendResult.setStatus(ProxySendResult.SEND_OK);
        proxySendResult.setMsgId(message.getMessageId());
        return proxySendResult;
    }

    public static String getChannel(ProxyMessage message) {
        return message.getSubject() + "[" + message.getTag() + "]";
    }

    private void setMessageId(ProxyMessage message) {
        if (message.getMessageId() == null || "".equals(message.getMessageId())) {
            message.setMessageId(Long.toString(System.nanoTime()));
        }
    }

    private void checkStatus(ProxyMessage message, ProxyMessageType proxyMessageType) {
        //todo: checkStatus
        if(true) {
            return;
        }
        if (!this.subject.equals(message.getSubject())) {
            throw new IllegalArgumentException("Unsupported subject[" + message.getSubject() + "].Supported subject[" + this.subject + "].");
        }
        Set<String> allowedTags = typeTagsMapping.get(proxyMessageType);
        if (allowedTags == null || (!allowedTags.contains(message.getTag()) && !allowedTags.contains("*"))) {
            throw new IllegalStateException("Subject[" + message.getSubject() + "]unsupported tag[" + message.getTag() + "].Supported tags " + (allowedTags == null ? "[]" : allowedTags) + " for messageType[" + proxyMessageType + "].");
        }
    }

    public void sendOneway(ProxyMessage message) {
        this.setMessageId(message);
        checkStatus(message, ProxyMessageType.ONEWAY);
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            setOneConsumer(jedis, message);
            jedis.publish(getChannel(message), objectMapperThreadLocal.get().writeValueAsString(message));
        } catch (Throwable e) {
            throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    public void startup() {
        if (this.jedisPool != null) {
            return;
        }
        GenericObjectPoolConfig redisConfig = new GenericObjectPoolConfig();
        URI redisUri = null;
        String redisCfgUrl = messageConfig.getProperties().getProperty("mq.redis.url", mergedProps.getProperty("mq.redis.url"));
        try {
            redisUri = new URI(redisCfgUrl);
        } catch (URISyntaxException e) {
            throw new IllegalStateException("[" + redisCfgUrl + "] parsed error", e);
        }
        jedisPool = new JedisPool(redisConfig, redisUri.getHost(), redisUri.getPort(), 3000, redisUri.getUserInfo(), Protocol.DEFAULT_DATABASE);
        this.redisMqTransactionCheckListener = new RedisMqTransactionCheckListener(jedisPool, this);
        this.redisMqTransactionCheckListener.startup();
    }

    public void shutdown() {
        jedisPool.close();
        jedisPool = null;
        this.redisMqTransactionCheckListener.shtudown();
        this.redisMqTransactionCheckListener = null;
    }

    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Map<String, Properties> propsMapping = applicationContext.getBeansOfType(Properties.class);
        if (propsMapping != null) {
            for (Properties props : propsMapping.values()) {
                CollectionUtils.mergePropertiesIntoMap(props, mergedProps);
            }
        }
        this.cacheStore = applicationContext.getBean(CacheStore.class);
    }
}
