package com.ohaotian.plugin.mq.proxy.ext.aliyunmq;

import com.aliyun.openservices.ons.api.*;
import com.aliyun.openservices.ons.api.transaction.LocalTransactionExecuter;
import com.aliyun.openservices.ons.api.transaction.TransactionProducer;
import com.aliyun.openservices.ons.api.transaction.TransactionStatus;
import com.ctrip.framework.apollo.Config;
import com.ctrip.framework.apollo.ConfigService;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfig;
import com.ohaotian.license.verify.service.LicenseVerifyService;
import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.ohaotian.plugin.mq.proxy.*;
import com.ohaotian.plugin.mq.proxy.callback.ProxyLocalTransactionExecuter;
import com.ohaotian.plugin.mq.proxy.callback.ProxySendCallback;
import com.ohaotian.plugin.mq.proxy.config.ApolloConfigVO;
import com.ohaotian.plugin.mq.proxy.ext.ProxyMessageProducerEx;
import com.ohaotian.plugin.mq.proxy.ext.ProxyMqTransactionChecker;
import com.ohaotian.plugin.mq.proxy.internal.ProxyMessageConfig;
import com.ohaotian.plugin.mq.proxy.internal.ProxyMessageException;
import com.ohaotian.plugin.mq.proxy.status.ProxyTransactionStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.UnsupportedEncodingException;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

public class AliyunMqMessageSender implements ProxyMessageProducerEx, ApplicationContextAware {

    private final Logger logger = LoggerFactory.getLogger(this.getClass().getName());
    private boolean started;
    private Properties mergedProps = new Properties();
   // @ApolloConfig
    //private Config apolloConfig= ConfigService.getAppConfig();
    private String defaultValue = "defaultValue";
    private Producer producer;
    private TransactionProducer transProductor;
    private ProxyMessageConfig messageConfig;
    private final String subject;
    private final Map<ProxyMessageType, Set<String>> typeTagsMapping;
    private AliyunMqTransactionCheckerListener transactionCheckerListener;
    private CacheStore cacheStore;
    private ApolloConfigVO apolloConfigVO;
    private LicenseVerifyService licenseVerifyService;

    public AliyunMqMessageSender(ProxyMessageConfig messageConfig, Map<ProxyMessageType, Set<String>> typeTagsMapping) {
        logger.info("[" + this.getClass().getName() + "] required properties[mq.producerId|mq.accessKey|mq.secretKey|mq.onsAddr|mq.sendMsgTimeoutMillis]");
        this.subject = messageConfig.getSubject();
        this.messageConfig = messageConfig;
        this.typeTagsMapping = typeTagsMapping;
    }

    @Override
    public ProxySendResult send(ProxyMessage message) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        checkStatus(message, ProxyMessageType.SYNCHRONIZATION);
        ProxySendResult proxySendResult;
        try {
            SendResult sendResult = producer.send(getMessage(message));
            proxySendResult = getProxySendResult(sendResult);
        } catch (Exception e) {
            throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
        }
        return proxySendResult;
    }

    @Override
    public ProxySendResult sendInTransaction(final ProxyMessage proxyMessage, final ProxyLocalTransactionExecuter localTransactionExecuter, final Object arg) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        checkStatus(proxyMessage, ProxyMessageType.TRANSACTION);
        ProxySendResult proxySendResult;
        Message message = getMessage(proxyMessage);
        try {
            SendResult sendResult = transProductor.send(message, new LocalTransactionExecuter() {
                @Override
                public TransactionStatus execute(Message msg, Object arg) {
                    ProxyTransactionStatus transactionStatus = localTransactionExecuter.exec(proxyMessage, arg);
                    if (transactionStatus == ProxyTransactionStatus.COMMIT) {
                        return TransactionStatus.CommitTransaction;
                    } else if (transactionStatus == ProxyTransactionStatus.ROLLBACK) {
                        return TransactionStatus.RollbackTransaction;
                    } else if (transactionStatus == ProxyTransactionStatus.UNKNOW) {
                        return TransactionStatus.Unknow;
                    }
                    return TransactionStatus.Unknow;
                }
            }, arg);
            proxySendResult = getProxySendResult(sendResult);
            cacheStore.set(message.getMsgID(), ProxyTransactionStatus.COMMIT.toString(), ProxyMqTransactionChecker.MQ_TRAN_CHECK_EXPIRE);
            if (logger.isDebugEnabled()) {
                logger.debug("send tran msg msgId={" + message.getMsgID() + "}  topic=" + proxyMessage.getSubject() + " tag=" + proxyMessage.getTag() + "  body={" + proxyMessage.getContent() + "}  SendResult={" + proxySendResult.getStatus() + "}");
            }
        } catch (Exception e) {
            cacheStore.set(message.getMsgID(), ProxyTransactionStatus.ROLLBACK.toString(), ProxyMqTransactionChecker.MQ_TRAN_CHECK_EXPIRE);
            throw new ProxyMessageException("send subject[" + proxyMessage.getSubject() + "] tag[" + proxyMessage.getTag() + "] error", e);
        }
        return proxySendResult;
    }

    private ProxySendResult getProxySendResult(SendResult sendResult) {
        ProxySendResult proxySendResult;
        proxySendResult = new ProxySendResult();
        proxySendResult.setMsgId(sendResult.getMessageId());
        proxySendResult.setStatus(ProxySendResult.SEND_OK);
        return proxySendResult;
    }

    private Message getMessage(ProxyMessage message) {
        Message sendMessage;
        try {
            sendMessage = new Message(message.getSubject(), message.getTag(), message.getContent().getBytes("UTF-8"));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Convert rocketmq message error", e);
        }
        return sendMessage;
    }

    @Override
    public void send(ProxyMessage message, final ProxySendCallback sendCallback) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        checkStatus(message, ProxyMessageType.ASYNCHRONOUS);
        try {
            producer.sendAsync(getMessage(message), new SendCallback() {
                @Override
                public void onSuccess(SendResult sendResult) {
                    if (sendCallback != null) {
                        sendCallback.onSuccess(getProxySendResult(sendResult));
                    }
                }

                @Override
                public void onException(OnExceptionContext context) {
                    if (sendCallback != null) {
                        sendCallback.onException(new ProxyExceptionContext(context.getException()));
                    }
                }
            });
        } catch (Throwable e) {
            if (sendCallback != null) {
                sendCallback.onException(new ProxyExceptionContext(e));
            } else {
                throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
            }
        }
    }

    private void checkStatus(ProxyMessage message, ProxyMessageType proxyMessageType) {
        //todo: checkStatus
        if(true) {
            return;
        }
        if (!this.subject.equals(message.getSubject())) {
            throw new IllegalArgumentException("Unsupported subject[" + message.getSubject() + "].Supported subject[" + this.subject + "].");
        }
        Set<String> allowedTags = typeTagsMapping.get(proxyMessageType);
        if (allowedTags == null || (!allowedTags.contains(message.getTag()) && !allowedTags.contains("*"))) {
            throw new IllegalStateException("Subject[" + message.getSubject() + "]unsupported tag[" + message.getTag() + "].Supported tags " + (allowedTags == null ? "[]" : allowedTags) + " for messageType[" + proxyMessageType + "].");
        }
    }

    @Override
    public void sendOneway(ProxyMessage message) {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        checkStatus(message, ProxyMessageType.ONEWAY);
        try {
            producer.sendOneway(getMessage(message));
        } catch (Throwable e) {
            throw new ProxyMessageException("send subject[" + message.getSubject() + "] tag[" + message.getTag() + "] error", e);
        }
    }

    @Override
    public void startup() {
        if (!licenseVerifyService.verify()){
            throw new ZTBusinessException("证书已过期或已损坏，请联系管理员");
        }

        Properties config = new Properties();
        Properties consumerProps = messageConfig.getProperties();
        config.put(PropertyKeyConst.ProducerId, ((DefaultProxyMessageConfig) messageConfig).getId());
        config.put(PropertyKeyConst.AccessKey, consumerProps.getProperty("mq.accessKey", apolloConfigVO.getAccessKey()));
        config.put(PropertyKeyConst.SecretKey, consumerProps.getProperty("mq.secretKey", apolloConfigVO.getSecretKey()));
        config.put(PropertyKeyConst.SendMsgTimeoutMillis, consumerProps.getProperty("mq.sendMsgTimeoutMillis", apolloConfigVO.getSendMsgTimeoutMillis()));
//        config.put(PropertyKeyConst.ONSAddr, consumerProps.getProperty("mq.onsAddr", mergedProps.getProperty("mq.onsAddr")));
//        config.put(PropertyKeyConst.NAMESRV_ADDR, consumerProps.getProperty("mq.onsAddr", mergedProps.getProperty("mq.onsAddr")));

        logger.info("***********是否使用gid***********");
        if(StringUtils.isEmpty(apolloConfigVO.getSupportGID())){
            logger.info("***********不使用gid***********");
            config.put(PropertyKeyConst.ProducerId, ((DefaultProxyMessageConfig) messageConfig).getId());
            config.put(PropertyKeyConst.ONSAddr, consumerProps.getProperty("mq.onsAddr", apolloConfigVO.getOnsAddr()));
        }else{
            logger.info("***********使用gid***********"+apolloConfigVO.getSupportGID());
            config.put(PropertyKeyConst.NAMESRV_ADDR, consumerProps.getProperty("mq.onsAddr", apolloConfigVO.getOnsAddr()));
        }


        if (typeTagsMapping.containsKey(ProxyMessageType.TRANSACTION)) {
            transactionCheckerListener = new AliyunMqTransactionCheckerListener();
            transactionCheckerListener.setCacheStore(cacheStore);
            transProductor = ONSFactory.createTransactionProducer(config, transactionCheckerListener);
        } else {
            transProductor = null;
        }
        boolean createCommonProductor = false;
        if (typeTagsMapping.containsKey(ProxyMessageType.SYNCHRONIZATION)) {
            createCommonProductor = true;
        }
        if (!createCommonProductor && typeTagsMapping.containsKey(ProxyMessageType.ASYNCHRONOUS)) {
            createCommonProductor = true;
        }
        if (!createCommonProductor && typeTagsMapping.containsKey(ProxyMessageType.ONEWAY)) {
            createCommonProductor = true;
        }
        if (!createCommonProductor && typeTagsMapping.containsKey(ProxyMessageType.ORDERED)) {
            createCommonProductor = true;
        }
        if (createCommonProductor) {
            producer = ONSFactory.createProducer(config);
        } else {
            producer = null;
        }

        if (this.producer != null) {
            this.producer.start();
        }
        if (this.transProductor != null) {
            this.transProductor.start();
        }
    }

    @Override
    public void shutdown() {
        if (this.producer != null) {
            this.producer.shutdown();
        }
        if (this.transProductor != null) {
            this.transProductor.shutdown();
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        ApolloConfigVO bean = applicationContext.getBean(ApolloConfigVO.class);
        apolloConfigVO=bean;
        Map<String, Properties> propsMapping = applicationContext.getBeansOfType(Properties.class);
        if (propsMapping != null) {
            for (Properties props : propsMapping.values()) {
                CollectionUtils.mergePropertiesIntoMap(props, mergedProps);
            }
        }
        this.cacheStore = applicationContext.getBean(CacheStore.class);
    }
}
