package com.ohaotian.plugin.mq.proxy.ext.redismq;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.ohaotian.plugin.mq.proxy.ProxyMessage;
import com.ohaotian.plugin.mq.proxy.ProxyMessageType;
import com.ohaotian.plugin.mq.proxy.config.ApolloConfigVO;
import com.ohaotian.plugin.mq.proxy.constants.Strategy;
import com.ohaotian.plugin.mq.proxy.impl.ConsumerRegisterInfo;
import com.ohaotian.plugin.mq.proxy.impl.MQRegister;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.apache.rocketmq.client.consumer.MQPushConsumer;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.util.CollectionUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Protocol;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.*;

public class RedisMQMessageConsumerRegister implements MQRegister, ApplicationContextAware {
    private Map<String, MQPushConsumer> consumerMapping = new HashMap<String, MQPushConsumer>();
    private boolean started;
    private Properties mergedProps = new Properties();

    private JedisPool jedisPool;
    private List<Jedis> jedisList = new ArrayList<Jedis>();
    private ConsumerRegisterInfo consumerInfo;
    private final ThreadLocal<ObjectMapper> objectMapperThreadLocal = new ThreadLocal<ObjectMapper>() {
        @Override
        protected ObjectMapper initialValue() {
            return new ObjectMapper();
        }
    };

    public void register(ConsumerRegisterInfo consumerInfo, ApolloConfigVO apolloConfigVO) {
        this.consumerInfo = consumerInfo;
        ProxyMessageType[] messageTypes = consumerInfo.getMessageTypes();
        for (ProxyMessageType messageType : messageTypes) {
            if (supportConcurrently(messageType)) {
                continue;
            }
            throw new UnsupportedOperationException("Unsuppoorted messageType[" + messageType + "] for subject[" + consumerInfo.getSubject() + "]");
        }
    }

    private boolean supportConcurrently(ProxyMessageType messageType) {
        return messageType == ProxyMessageType.ASYNCHRONOUS || messageType == ProxyMessageType.SYNCHRONIZATION || messageType == ProxyMessageType.ONEWAY || messageType == ProxyMessageType.TRANSACTION;
    }

    public boolean support(String strategy) {
        return RedisMQMessageConsumerRegister.strategySupported(strategy);
    }

    public static boolean strategySupported(String strategy) {
        return Strategy.isRedisMQ(strategy);
    }

    public void startup() {
        if (started) {
            return;
        }
        /*redis://password@localhost:6379/0*/
        GenericObjectPoolConfig redisConfig = new GenericObjectPoolConfig();
        URI redisUri = null;
        String redisCfgUrl = consumerInfo.getProperties().getProperty("mq.redis.url", mergedProps.getProperty("mq.redis.url"));
        try {
            redisUri = new URI(redisCfgUrl);
        } catch (URISyntaxException e) {
            throw new IllegalStateException("[" + redisCfgUrl + "] parsed error", e);
        }
        jedisPool = new JedisPool(redisConfig, redisUri.getHost(), redisUri.getPort(), 3000, redisUri.getUserInfo(), Protocol.DEFAULT_DATABASE);
        List<String> channelList = new ArrayList<String>();
        for (String tag : consumerInfo.getTags()) {
            channelList.add(RedisMqMessageSender.getChannel(new ProxyMessage(consumerInfo.getSubject(), tag, null)));
        }
        final String[] channels = new String[channelList.size()];
        channelList.toArray(channels);
        final Jedis jedis = jedisPool.getResource();
        new Thread(new Runnable() {
            public void run() {
                jedis.subscribe(new RedisMQMessageListener(jedisPool, consumerInfo.getConsumerWrappers()), channels);
                jedisList.add(jedis);
            }
        }).start();

        started = true;
    }

    public void shutdown() {
        if (!started) {
            return;
        }
        for (Jedis jedis: jedisList) {
            jedis.close();
        }
        jedisList.clear();
        jedisPool.close();
        jedisPool = null;
        started = false;
    }

    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Map<String, Properties> propsMapping = applicationContext.getBeansOfType(Properties.class);
        if (propsMapping != null) {
            for (Properties props : propsMapping.values()) {
                CollectionUtils.mergePropertiesIntoMap(props, mergedProps);
            }
        }
    }
}
