package com.ohaotian.plugin.es.elasticsearch;

import com.ohaotian.plugin.es.core.Client;
import com.ohaotian.plugin.es.core.ClientException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.LinkedList;
import java.util.List;
import org.elasticsearch.action.admin.indices.create.CreateIndexAction;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequestBuilder;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexAction;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequestBuilder;
import org.elasticsearch.action.delete.DeleteAction;
import org.elasticsearch.action.delete.DeleteRequestBuilder;
import org.elasticsearch.action.index.IndexAction;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.search.SearchAction;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.client.IndicesAdminClient;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.InetSocketTransportAddress;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.transport.client.PreBuiltTransportClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class ElasticSearchClient implements Client {

    private final Logger logger = LogManager.getLogger(ElasticSearchClient.class);

    private TransportClient client;

    @Autowired
    private ElasticSearchConfig config;

    @Override public void initialize() throws ClientException {
        logger.debug("cluster name: {}", config.getClusterName());
        Settings settings = Settings.builder()
            .put("cluster.name", config.getClusterName())
            .build();

        client = new PreBuiltTransportClient(settings);

        List<AddressPairs> pairsList = parseClusterNodes(config.getClusterNodes());
        for (AddressPairs pairs : pairsList) {
            try {
                ((TransportClient)client).addTransportAddress(new InetSocketTransportAddress(InetAddress.getByName(pairs.host), pairs.port));
            } catch (UnknownHostException e) {
                throw new ElasticSearchClientException(e.getMessage(), e);
            }
        }
    }

    private List<AddressPairs> parseClusterNodes(String nodes) {
        List<AddressPairs> pairsList = new LinkedList<>();
        logger.info("elasticsearch cluster nodes: {}", nodes);
        String[] nodesSplit = nodes.split(",");
        for (int i = 0; i < nodesSplit.length; i++) {
            String node = nodesSplit[i];
            String host = node.split(":")[0];
            String port = node.split(":")[1];
            pairsList.add(new AddressPairs(host, Integer.valueOf(port)));
        }

        return pairsList;
    }

    class AddressPairs {
        private String host;
        private Integer port;

        public AddressPairs(String host, Integer port) {
            this.host = host;
            this.port = port;
        }
    }

    public SearchRequestBuilder searchRequestBuilder() {
        return SearchAction.INSTANCE.newRequestBuilder(client);
    }

    public CreateIndexRequestBuilder createIndexRequestBuilder() {
        IndicesAdminClient adminClient = client.admin().indices();
        return CreateIndexAction.INSTANCE.newRequestBuilder(adminClient);
    }

    public DeleteIndexRequestBuilder deleteIndexRequestBuilder() {
        IndicesAdminClient adminClient = client.admin().indices();
        return DeleteIndexAction.INSTANCE.newRequestBuilder(adminClient);
    }

    public IndexRequestBuilder indexRequestBuilder() {
        IndexRequestBuilder indexRequestBuilder = IndexAction.INSTANCE.newRequestBuilder(client);
        return indexRequestBuilder;
    }

    public DeleteRequestBuilder deleteRequestBuilder() {
        return DeleteAction.INSTANCE.newRequestBuilder(client);
    }
}
