package com.ohaotian.plugin.security.config;


import com.ohaotian.plugin.security.constants.SercurityConstants;
import com.ohaotian.plugin.security.filter.Ajax401Filter;
import com.ohaotian.plugin.security.interceptor.CasFilterSecurityInterceptor;
import com.ohaotian.plugin.security.property.FilterStaticConfig;
import org.jasig.cas.client.session.SingleSignOutFilter;
import org.jasig.cas.client.session.SingleSignOutHttpSessionListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.servlet.ServletListenerRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.cas.ServiceProperties;
import org.springframework.security.cas.web.CasAuthenticationEntryPoint;
import org.springframework.security.cas.web.CasAuthenticationFilter;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.web.access.intercept.FilterSecurityInterceptor;
import org.springframework.security.web.authentication.logout.LogoutFilter;
import org.springframework.security.web.authentication.switchuser.SwitchUserFilter;


@Configuration
@EnableWebSecurity
public class WebSecurityConfig extends WebSecurityConfigurerAdapter {

    private static final Logger logger = LoggerFactory.getLogger(WebSecurityConfig.class);


    @Autowired
    private CasFilterSecurityInterceptor myFilterSecurityInterceptor;

    @Autowired
    private FilterStaticConfig filterStaticConfig;


    /**
     * 自定义过滤规则及其安全配置
     */
    @Override
    protected void configure(HttpSecurity http) throws Exception {
        // 解决不允许显示在iframe的问题
        http.headers().frameOptions().disable();

        // CsrfConfigurer
        http.csrf().disable();

        http.authorizeRequests()//配置安全策略
                .antMatchers(SercurityConstants.SIGNUO + "/**").permitAll();//定义/请求不需要验证

        filterStaticConfig.getResources().forEach(resource -> {
            try {
                http.authorizeRequests()//配置安全策略
                        .antMatchers(resource).permitAll();//定义/请求不需要验证
            } catch (Exception e) {
                logger.error("白名单配置异常：", e);
            }
        });

        http.authorizeRequests().anyRequest().authenticated();//其余的所有请求都需要验证

        //定义logout不需要验证
        http.logout().permitAll();

        // 入口
        CasAuthenticationEntryPoint entryPoint = getApplicationContext().getBean(CasAuthenticationEntryPoint.class);

        CasAuthenticationFilter casAuthenticationFilter = getApplicationContext()
                .getBean(CasAuthenticationFilter.class);
        SingleSignOutFilter singleSignOutFilter = getApplicationContext().getBean(SingleSignOutFilter.class);
        LogoutFilter logoutFilter = getApplicationContext().getBean(LogoutFilter.class);
        Ajax401Filter ajax401Filter = getApplicationContext().getBean(Ajax401Filter.class);
        SwitchUserFilter switchUserFilter = getApplicationContext().getBean(SwitchUserFilter.class);
        /**
         * 执行顺序为
         * LogoutFilter-->SingleSignOutFilter-->CasAuthenticationFilter-->
         * ExceptionTranslationFilter
         */
        http.addFilterAfter(switchUserFilter, FilterSecurityInterceptor.class).exceptionHandling().authenticationEntryPoint(entryPoint)
                .and()
                .addFilter(casAuthenticationFilter)
                .addFilterBefore(logoutFilter, LogoutFilter.class)
                .addFilterBefore(singleSignOutFilter, CasAuthenticationFilter.class)
                .addFilterBefore(ajax401Filter, CasAuthenticationFilter.class)
//                .addFilterBefore(switchUserFilter,CasAuthenticationFilter.class)
                .addFilterBefore(myFilterSecurityInterceptor, FilterSecurityInterceptor.class);
        http.antMatcher("/**");

    }


    @Autowired
    public void configureGlobal(AuthenticationManagerBuilder auth) throws Exception {
        //放入cas凭证校验器
        AuthenticationProvider authenticationProvider = (AuthenticationProvider) getApplicationContext()
                .getBean("casProvider");
        auth.authenticationProvider(authenticationProvider);

    }

    @Bean
    public ServletListenerRegistrationBean<SingleSignOutHttpSessionListener> singleSignOutHttpSessionListener() {
        ServletListenerRegistrationBean<SingleSignOutHttpSessionListener> servletListenerRegistrationBean =
                new ServletListenerRegistrationBean<>();
        servletListenerRegistrationBean.setListener(new SingleSignOutHttpSessionListener());
        return servletListenerRegistrationBean;
    }


    /**
     * cas filter类
     * <p>
     * 针对/login请求的校验
     *
     * @return
     */
    @Bean
    public CasAuthenticationFilter casAuthenticationFilter(ServiceProperties properties) throws Exception {
        CasAuthenticationFilter casAuthenticationFilter = new CasAuthenticationFilter();
        casAuthenticationFilter.setServiceProperties(properties);
        casAuthenticationFilter.setFilterProcessesUrl("/login/cas");
        casAuthenticationFilter.setAuthenticationManager(authenticationManager());
        casAuthenticationFilter.setContinueChainBeforeSuccessfulAuthentication(false);
        return casAuthenticationFilter;
    }


}
