package com.ohaotian.plugin.security.config;

import com.ohaotian.cas.SingleSignOutFilter;
import com.ohaotian.plugin.security.filter.Ajax401Filter;
import com.ohaotian.plugin.security.filter.CSRFilter;
import com.ohaotian.plugin.security.filter.TokenAuthenticationFilter;
import com.ohaotian.plugin.security.property.AppServerConfig;
import com.ohaotian.plugin.security.property.CasServerConfig;
import com.ohaotian.plugin.security.service.DefaultUserDetailsService;
import org.jasig.cas.client.session.SingleSignOutHttpSessionListener;
import org.jasig.cas.client.validation.Cas20ServiceTicketValidator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.web.servlet.ServletListenerRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.cas.ServiceProperties;
import org.springframework.security.cas.authentication.CasAuthenticationProvider;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.web.authentication.logout.LogoutFilter;
import org.springframework.security.web.authentication.logout.SecurityContextLogoutHandler;
import org.springframework.security.web.authentication.switchuser.SwitchUserFilter;


@Configuration
public class CasConfiguration {


    @Autowired
    private CasServerConfig casServerConfig;

    @Autowired
    private AppServerConfig appServerConfig;

    @Autowired
    private UserDetailsService userDetailsService;


    /**
     * 设置客户端service的属性
     * <p>
     * 主要设置请求cas服务端后的回调路径,一般为主页地址，不可为登录地址
     * <p>
     * </p>
     *
     * @return
     */
    @Bean
    public ServiceProperties serviceProperties() {
        ServiceProperties serviceProperties = new ServiceProperties();
        // 设置回调的service路径，此为主页路径
        serviceProperties.setService(appServerConfig.getUrl() + "/login/cas");
        // 对所有的未拥有ticket的访问均需要验证
        serviceProperties.setAuthenticateAllArtifacts(true);
        return serviceProperties;
    }

    /**
     * 配置ticket校验器
     *
     * @return
     */
    @Bean
    public Cas20ServiceTicketValidator cas20ServiceTicketValidator() {
        // 配置上服务端的校验ticket地址
        return new Cas20ServiceTicketValidator(casServerConfig.getIntranet());
    }

    /**
     * 单点注销，接受cas服务端发出的注销session请求
     *
     * @return
     */
    @Bean
    public SingleSignOutFilter singleSignOutFilter() {
        SingleSignOutFilter outFilter = new SingleSignOutFilter();
        // 设置cas服务端路径前缀，应用于front channel的注销请求
        outFilter.setCasServerUrlPrefix(casServerConfig.getOuter());
        outFilter.setIgnoreInitConfiguration(true);
        outFilter.setClusterNodeUrls(casServerConfig.getClusterIp());
        return outFilter;
    }

    /**
     * 单点请求cas客户端退出Filter类
     * <p>
     * 请求/logout，转发至cas服务端进行注销
     */
    @Bean
    public LogoutFilter logoutFilter() {
        LogoutFilter logoutFilter = new LogoutFilter(casServerConfig.getLogout(), new SecurityContextLogoutHandler());

        logoutFilter.setFilterProcessesUrl("/logout");
        return logoutFilter;
    }

    /**
     * ajax请求过滤器
     *
     * @return
     */
    @Bean
    public Ajax401Filter ajax401Filter() {
        return new Ajax401Filter();
    }

    /**
     * 跨站点请求伪造 CSRF攻击
     *
     * @return
     */
    @Bean
    public CSRFilter csrFilter() {
        return new CSRFilter();
    }

    @Bean
    public TokenAuthenticationFilter tokenAuthenticationFilter() {
        return new TokenAuthenticationFilter();
    }


    /**
     * 创建cas校验类
     * <p>
     * <p>
     * <b>Notes:</b> TicketValidator、AuthenticationUserDetailService属性必须设置;
     * serviceProperties属性主要应用于ticketValidator用于去cas服务端检验ticket
     * </p>
     *
     * @return
     */
    @Bean("casProvider")
    public CasAuthenticationProvider casAuthenticationProvider() {
        CasAuthenticationProvider provider = new CasAuthenticationProvider();
        provider.setKey("casProvider");
        provider.setServiceProperties(serviceProperties());
        provider.setTicketValidator(cas20ServiceTicketValidator());
        provider.setUserDetailsService(userDetailsService);
        return provider;
    }

    /**
     * 用户自定义的AuthenticationUserDetailsService
     */
    @ConditionalOnMissingBean(
            name = {"userDetailsService"}
    )
    @Bean
    public UserDetailsService userDetailsService() {
        return new DefaultUserDetailsService();
    }


    /**
     * 认证的入口，即跳转至服务端的cas地址
     * <p>
     * <p>
     * <b>Note:</b>浏览器访问不可直接填客户端的login请求,若如此则会返回Error页面，无法被此入口拦截
     * </p>
     */
    @Bean
    public ZTCasAuthenticationEntryPoint casAuthenticationEntryPoint() {
        ZTCasAuthenticationEntryPoint entryPoint = new ZTCasAuthenticationEntryPoint();
        entryPoint.setServiceProperties(serviceProperties());
        entryPoint.setLoginUrl(casServerConfig.getLogin());

        return entryPoint;
    }


    @Bean
    public SwitchUserFilter switchUserFilter(UserDetailsService userDetailsService) throws Exception {
        SwitchUserFilter switchUserFilter = new SwitchUserFilter();
        switchUserFilter.setUserDetailsService(userDetailsService);
        switchUserFilter.setUsernameParameter("userId");
        switchUserFilter.setTargetUrl(appServerConfig.getIndex());
        switchUserFilter.setSwitchUserUrl("/switch");
        return switchUserFilter;
    }


    @Bean
    public ServletListenerRegistrationBean servletListenerRegistrationBean() {
        ServletListenerRegistrationBean servletListenerRegistrationBean = new ServletListenerRegistrationBean();
        servletListenerRegistrationBean.setListener(new SingleSignOutHttpSessionListener());
        return servletListenerRegistrationBean;
    }


}
