package com.ohaotian.plugin.security.filter;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ohaotian.plugin.base.constant.BaseRspConstants;
import com.ohaotian.plugin.security.constants.SercurityConstants;
import com.ohaotian.plugin.security.entity.UserInfo;
import com.ohaotian.plugin.security.jwt.Jwt;
import com.ohaotian.plugin.security.jwt.TokenState;
import com.ohaotian.plugin.security.service.GetUserInfoByUserIdService;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.web.filter.GenericFilterBean;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Map;

/**
 * 使用 token 进行身份验证的过滤器。
 * 如果 request header 中有 auth-token，使用 auth-token 的值查询对应的登陆用户，如果用户有效则放行访问，否则返回 401 错误。
 */
public class TokenAuthenticationFilter extends GenericFilterBean {
    // 是否允许当前请求创建 session
    private static ThreadLocal<Boolean> allowSessionCreation = new ThreadLocal<>();

    private GetUserInfoByUserIdService getUserInfoByTokenService;


    public Authentication attemptAuthentication(HttpServletRequest request, HttpServletResponse response)
            throws AuthenticationException, IOException {
        String token = request.getHeader("auth-token");

        // 模拟 token 无效返回 null
        if (SercurityConstants.GENERAL_TOKEN.equals(token)) {
            UserInfo user = new UserInfo();
            return new UsernamePasswordAuthenticationToken(user, user.getPassword(), user.getAuthorities());
        }


        Map<String, Object> resultMap = Jwt.validToken(token);
        TokenState state = TokenState.getTokenState((String) resultMap.get("state"));
        JSONObject jsonObject = new JSONObject();
        switch (state) {
            case VALID:
                //取出payload中数据,放入到request作用域中
                net.minidev.json.JSONObject data = (net.minidev.json.JSONObject) resultMap.get("data");
                request.setAttribute("data", data);

                String appCode=data.getAsString("appCode");
                //放行
                // 使用 token 信息查找缓存中的登陆用户信息
                UserDetails user;
                if (getUserInfoByTokenService != null) {
                    user = getUserInfoByTokenService.getUserInfoByUserId((Long) data.get("userId"), token,appCode);
                } else {
                    user = (UserDetails) data.get("userInfo");
                }
                return new UsernamePasswordAuthenticationToken(user, user.getPassword(), user.getAuthorities());
            case EXPIRED:
                jsonObject.put("MESSAGE", "Token 过期，请重新申请 token");
                break;
            default:
                jsonObject.put("MESSAGE", "Token 无效，请重新申请 token");
                break;
        }
        response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
        response.setContentType("text/html;charset=UTF-8");
        PrintWriter out = response.getWriter();
        jsonObject.put("code", BaseRspConstants.CODE_FAILUR);
        String jsonString = JSON.toJSONString(jsonObject, SerializerFeature.WriteMapNullValue, SerializerFeature.WriteNullStringAsEmpty);
        out.write(jsonString);
        out.close();
        return null;
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;
        Authentication auth;
        // 默认创建 session
        allowSessionCreation.set(true);
        // 如果 header 里有 auth-token 时，则使用 token 查询用户数据进行登陆验证
        if (request.getHeader("auth-token") != null) {
            // 1. 尝试进行身份认证
            // 2. 如果用户无效，则返回 401
            // 3. 如果用户有效，则保存到 SecurityContext 中，供本次方式后续使用
            auth = attemptAuthentication(request, response);
            if (auth == null) {
                return;
            }
            // 保存认证信息到 SecurityContext，禁止 HttpSessionSecurityContextRepository 创建 session
            allowSessionCreation.set(false);
            SecurityContextHolder.getContext().setAuthentication(auth);
        }
        // 继续调用下一个 filter: UsernamePasswordAuthenticationToken
        chain.doFilter(request, response);
    }

    public static boolean isAllowSessionCreation() {
        return allowSessionCreation.get();
    }


    public GetUserInfoByUserIdService getGetUserInfoByTokenService() {
        return getUserInfoByTokenService;
    }

    public void setGetUserInfoByTokenService(GetUserInfoByUserIdService getUserInfoByTokenService) {
        this.getUserInfoByTokenService = getUserInfoByTokenService;
    }
}
