package com.ohaotian.plugin.cache;

import com.ohaotian.plugin.constant.SessionConstant;
import redis.clients.jedis.Jedis;

/**
 * CacheTemplate
 *
 * @author liubinqi(liubq@tydic.com)
 */
public class CacheTemplate {

    private static volatile CacheTemplate INSTANCE;

    private static CaffeineCache CAFFEINECACHE;

    private CacheClient redisCache;

    private SessionConstant sessionConstant;


    private static final RedisPubSubClusterPolicy POLICY = new RedisPubSubClusterPolicy();

    private CacheTemplate() {

    }

    public static CacheTemplate getIstance() {
        // 对象实例化时与否判断（不使用同步代码块，instance不等于null时，直接返回对象，提高运行效率）
        if (INSTANCE == null) {
            //同步代码块（对象未初始化时，使用同步代码块，保证多线程访问时对象在第一次创建后，不再重复被创建）
            synchronized (CacheTemplate.class) {
                //未初始化，则初始instance变量
                if (INSTANCE == null) {
                    INSTANCE = new CacheTemplate();
                }
            }
        }
        return INSTANCE;
    }

    protected void init(CaffeineCache caffeineCache, CacheClient redisCache, SessionConstant sessionConstant) {
        CAFFEINECACHE = caffeineCache;
        this.redisCache = redisCache;
        this.sessionConstant = sessionConstant;
        POLICY.connect(redisCache.getJedisPool());
    }


    /**
     * 设置缓存
     */
    public void set(String key, Object value) {
        try {
            CAFFEINECACHE.put(key, value);
            redisCache.set(key, value, sessionConstant.getDefaultMaxInactiveIntervalReids());
        } finally {
            POLICY.sendClearCmd(key);
        }

    }


    /**
     * 获取缓存值
     */
    public Object get(String key) {

        Object level1Value = CAFFEINECACHE.get(key);
        if (level1Value != null) {
            return level1Value;
        }
        Object level2Value = redisCache.get(key);
        if (level2Value != null) {
            CAFFEINECACHE.put(key, level2Value);
            Jedis jedis = null;
            try {
                jedis = redisCache.getJedisPool().getResource();
                jedis.expire(key, 600);
            } finally {
                if (jedis != null) {
                    jedis.close();
                }

            }
            return level2Value;
        }
        return null;
    }


    /**
     * 删除集群缓存
     */
    public void del(String key) {
        CAFFEINECACHE.evict(key);
        redisCache.delete(key);
        POLICY.sendClearCmd(key);
    }

    /**
     * 清除本地缓存
     */
    public void evict(String key) {
        CAFFEINECACHE.evict(key);
    }


}
