package com.ohaotian.plugin.cache;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPubSub;
import redis.clients.jedis.exceptions.JedisConnectionException;

public class RedisPubSubClusterPolicy extends JedisPubSub {

    private final static Logger log = LoggerFactory.getLogger(RedisPubSubClusterPolicy.class);

    private String channel = "ohaotian-session";

    private JedisPool client;

    private JedisCluster jedisCluster;

    private boolean isCluster = false;

    public void setCluster(boolean cluster) {
        isCluster = cluster;
    }

    /**
     * 加入 Redis 的发布订阅频道
     */
    public void connect(JedisPool jedisPool) {
        long ct = System.currentTimeMillis();
        this.client = jedisPool;
        try (Jedis jedis = this.client.getResource()) {
            jedis.publish(channel, Command.join().json());   //Join Cluster
        } catch (Exception e) {
            log.error("", e);
        }
        new Thread(() -> {
            //当 Redis 重启会导致订阅线程断开连接，需要进行重连
            while (true) {
                try (Jedis jedis = client.getResource()) {
                    jedis.subscribe(this, channel);
                    log.info("Disconnect to redis channel:" + channel);
                    break;
                } catch (JedisConnectionException e) {
                    log.error("Failed connect to redis, reconnect it.", e);
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException ie) {
                        break;
                    }
                }
            }
        }, "RedisSubscribeThread").start();

        log.info("Connected to redis channel:" + channel + ", time " + (System.currentTimeMillis() - ct) + " ms.");
    }


    /**
     * 加入 Redis 的发布订阅频道(集群)
     */
    public void connect(JedisCluster jedisCluster) {
        this.jedisCluster = jedisCluster;
        long ct = System.currentTimeMillis();
        try{
            jedisCluster.publish(channel, Command.join().json());//Join Cluster
        } catch (Exception e) {
            log.error("", e);
        }
        new Thread(() -> {
            //当 Redis 重启会导致订阅线程断开连接，需要进行重连
            while (true) {
                try{
                    jedisCluster.subscribe(this, channel);
                    log.info("Disconnect to redis channel:" + channel);
                    break;
                } catch (JedisConnectionException e) {
                    log.error("Failed connect to redis, reconnect it.", e);
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException ie) {
                        break;
                    }
                }
            }
        }, "RedisSubscribeThread").start();
        log.info("Connected to redis channel:" + channel + ", time " + (System.currentTimeMillis() - ct) + " ms.");
    }

    /**
     * 退出 Redis 发布订阅频道
     */
    public void disconnect(JedisPool jedisPool) {
        this.client = jedisPool;
        try (Jedis jedis = this.client.getResource()) {
            this.unsubscribe();
            jedis.publish(channel, Command.quit().json()); //Quit Cluster
        }
    }

    /**
     * 退出 Redis 发布订阅频道(集群)
     */
    public void disconnect(JedisCluster jedisCluster) {
        this.jedisCluster = jedisCluster;
        this.unsubscribe();
        jedisCluster.publish(channel, Command.quit().json()); //Quit Cluster
    }

    public void clear(String key) {
        if(isCluster){
            jedisCluster.del(key.getBytes());
        }else{
            Jedis jedis = this.client.getResource();
            try {
                jedis.del(key.getBytes());
            } finally {
                if (jedis != null) {
                    jedis.close();
                }
            }
        }
        CacheTemplate.getIstance().evict(key);
    }


    /**
     * 发送清除缓存的广播命令
     *
     * @param key: Cache key name
     */
    public void sendClearCmd(String key) {
        if(isCluster){
            jedisCluster.publish(channel, new Command(Command.OPT_CLEAR_KEY, key).json());
        }else{
            try (Jedis jedis = client.getResource()) {
                jedis.publish(channel, new Command(Command.OPT_CLEAR_KEY, key).json());
            } catch (Exception e) {
                log.error("发送清除缓存的广播命令 ", e);
            }
        }
    }


    /**
     * 当接收到订阅频道获得的消息时触发此方法
     *
     * @param channel 频道名称
     * @param message 消息体
     */
    @Override
    public void onMessage(String channel, String message) {
        try {
            Command cmd = Command.parse(message);

            if (cmd == null || cmd.isLocal()) {
                return;
            }
            switch (cmd.getOperator()) {
                case Command.OPT_JOIN:
                    log.info("Node-" + cmd.getSrc() + " joined to " + this.channel);
                    break;
                case Command.OPT_CLEAR_KEY:
                    CacheTemplate.getIstance().evict(cmd.getKey());
                    log.debug("Received cache clear message, region=" + cmd.getKey());
                    break;
                case Command.OPT_QUIT:
                    log.info("Node-" + cmd.getSrc() + " quit to " + this.channel);
                    break;
                default:
                    log.warn("Unknown message type = " + cmd.getOperator());
            }
        } catch (Exception e) {
            log.error("Failed to handle received msg", e);
        }
    }
}
