package com.ohaotian.plugin.web.filter;

import com.ohaotian.plugin.web.utils.SensitiveWordUtil;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.*;

/**
 * @author liuzhenhua
 * @description: 请求参数GET、POST参数处理
 * @date 2021/12/21 10:46
 */
public class ParamsRequestWrapper extends HttpServletRequestWrapper {

    private Map<String, String[]> parameterMap = new HashMap<>();

    private static final String ENCODING = "UTF-8";
    private static final String CLASSTYPE = "java.lang.String";

    public ParamsRequestWrapper(HttpServletRequest request) {
        super(request);
        parameterMap = request.getParameterMap();
    }

    /**
     * 将parameter的值去除空格后重写回去
     */
    public void modifyParameters(Map<String, String[]> parameterMap) {
        Set<String> set = parameterMap.keySet();
        Iterator<String> it = set.iterator();
        while (it.hasNext()) {
            String key = (String) it.next();
            String[] values = parameterMap.get(key);
            /**
             * 敏感词过滤
             */
            values[0] = SensitiveWordUtil.replaceSensitiveWord(values[0]);
            parameterMap.put(key, values);
        }
    }

    /**
     * 获取所有参数名
     * @return 返回所有参数名
     */
    @Override
    public Enumeration<String> getParameterNames() {
        Vector<String> vector = new Vector<String>(parameterMap.keySet());
        return vector.elements();
    }

    /**
     * 获取指定参数名的值，如果有重复的参数名，则返回第一个的值 接收一般变量 ，如text类型
     * @param name 指定参数名
     * @return 指定参数名的值
     */
    @Override
    public String getParameter(String name) {
        if(parameterMap.containsKey(name)){
            String[] results = parameterMap.get(name);
            return results[0];
        }
        return null;
    }

    /**
     * 获取指定参数名的所有值的数组，如：checkbox的所有数据
     * 接收数组变量 ，如checkobx类型
     */
    @Override
    public String[] getParameterValues(String name) {
        return parameterMap.get(name);
    }

    @Override
    public Map<String, String[]> getParameterMap() {
        return parameterMap;
    }

    public void setParameterMap(Map<String, String[]> parameterMap) {
        this.parameterMap = parameterMap;
    }

    /**
     * 重写getInputStream方法  post请求参数必须通过流才能获取到值
     */
    @Override
    public ServletInputStream getInputStream() throws IOException {
        ServletInputStream stream = super.getInputStream();

        //非json类型，直接返回
        if (!super.getHeader(HttpHeaders.CONTENT_TYPE).equalsIgnoreCase(MediaType.APPLICATION_JSON_VALUE)) {
            return stream;
        }
        String json = IOUtils.toString(stream, ENCODING);

        if (StringUtils.isBlank(json)) {
            return stream;
        }

        System.out.println("转化前参数：" + json);
        //替换敏感词
        String result = SensitiveWordUtil.replaceSensitiveWord(json,SensitiveWordUtil.MaxMatchType);
        System.out.println("转化前参数：" + result);

        ByteArrayInputStream bis = new ByteArrayInputStream(result.getBytes(ENCODING));

        return new ParamsServletInputStream(bis);
    }
}