package com.ohaotian.plugin.web.utils;

import com.ohaotian.plugin.web.cache.Cache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.util.CollectionUtils;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * @author liuzhenhua
 * @description: 请求参数输入流
 * @date 2021/12/21 10:46
 */
@Slf4j
public class SensitiveWordUtil {

    public final static String WORDS = "WORDS";

    /**
     * 敏感词匹配规则
     */
    public static final int MinMatchTYpe = 1;      //最小匹配规则，如：敏感词库["中国","中国人"]，语句："我是中国人"，匹配结果：我是[中国]人
    public static final int MaxMatchType = 2;      //最大匹配规则，如：敏感词库["中国","中国人"]，语句："我是中国人"，匹配结果：我是[中国人]

    private static final char maskChar = '*'; // 掩码

    /**
     * 敏感词集合
     */
    private static HashMap sensitiveWordMap;

    /**
     * 初始化敏感词库，构建DFA算法模型
     * @param sensitiveWordSet 敏感词库
     */
    public static synchronized void init(List<String> sensitiveWordSet) {
        initSensitiveWordMap(sensitiveWordSet);
    }

    /**
     * 初始化敏感词库，构建DFA算法模型
     * @param sensitiveWordSet 敏感词库
     */
    private static void initSensitiveWordMap(List<String> sensitiveWordSet) {
        //初始化敏感词容器，减少扩容操作
        sensitiveWordMap = new HashMap(sensitiveWordSet.size());
        String key;
        Map nowMap;
        Map<String, String> newWorMap;
        //迭代sensitiveWordSet
        Iterator<String> iterator = sensitiveWordSet.iterator();
        while (iterator.hasNext()) {
            //关键字
            key = iterator.next();
            nowMap = sensitiveWordMap;
            for (int i = 0; i < key.length(); i++) {
                //转换成char型
                char keyChar = key.charAt(i);
                //库中获取关键字
                Object wordMap = nowMap.get(keyChar);
                //如果存在该key，直接赋值，用于下一个循环获取
                if (wordMap != null) {
                    nowMap = (Map) wordMap;
                } else {
                    //不存在则，则构建一个map，同时将isEnd设置为0，因为他不是最后一个
                    newWorMap = new HashMap<>();
                    //不是最后一个
                    newWorMap.put("isEnd", "0");
                    nowMap.put(keyChar, newWorMap);
                    nowMap = newWorMap;
                }

                if (i == key.length() - 1) {
                    //最后一个
                    nowMap.put("isEnd", "1");
                }
            }
        }
    }

    /**
     * 判断文字是否包含敏感字符
     * @param txt       文字
     * @param matchType 匹配规则 1：最小匹配规则，2：最大匹配规则
     * @return 若包含返回true，否则返回false
     */
    public static boolean contains(String txt, int matchType) {
        boolean flag = false;
        for (int i = 0; i < txt.length(); i++) {
            int matchFlag = checkSensitiveWord(txt, i, matchType); //判断是否包含敏感字符
            if (matchFlag > 0) {    //大于0存在，返回true
                flag = true;
            }
        }
        return flag;
    }

    /**
     * 判断文字是否包含敏感字符
     *
     * @param txt 文字
     * @return 若包含返回true，否则返回false
     */
    public static boolean contains(String txt) {
        return contains(txt, MaxMatchType);
    }

    /**
     * 获取文字中的敏感词
     *
     * @param txt       文字
     * @param matchType 匹配规则 1：最小匹配规则，2：最大匹配规则
     */
    public static List<String> getSensitiveWord(String txt, int matchType) {
        List<String> sensitiveWordList = new ArrayList<>();

        for (int i = 0; i < txt.length(); i++) {
            //判断是否包含敏感字符
            int length = checkSensitiveWord(txt, i, matchType);
            if (length > 0) {//存在,加入list中
                sensitiveWordList.add(txt.substring(i, i + length));
                i = i + length - 1;//减1的原因，是因为for会自增
            }
        }

        return sensitiveWordList;
    }

    /**
     * 获取文字中的敏感词
     *
     * @param txt 文字
     */
    public static List<String> getSensitiveWord(String txt) {
        return getSensitiveWord(txt, MaxMatchType);
    }

    /**
     * 替换敏感字字符
     * @param txt         文本
     */
    public static String replaceSensitiveWord(String txt) {
        return replaceSensitiveWord(txt, maskChar, MaxMatchType);
    }

    /**
     * 替换敏感字字符
     * @param txt         文本
     * @param maskChar 替换的字符，匹配的敏感词以字符逐个替换，如 语句：我爱中国人 敏感词：中国人，替换字符：*， 替换结果：我爱***
     */
    public static String replaceSensitiveWord(String txt, char maskChar) {
        return replaceSensitiveWord(txt, maskChar, MaxMatchType);
    }

    /**
     * 替换敏感字字符
     *
     * @param txt       文本
     * @param matchType 敏感词匹配规则
     */
    public static String replaceSensitiveWord(String txt, int matchType) {
        return replaceSensitiveWord(txt,maskChar,matchType);
    }

    /**
     * 替换敏感字字符
     *
     * @param txt         文本
     * @param maskChar 替换的字符，匹配的敏感词以字符逐个替换，如 语句：我爱中国人 敏感词：中国人，替换字符：*， 替换结果：我爱***
     * @param matchType   敏感词匹配规则
     */
    public static String replaceSensitiveWord(String txt, char maskChar, int matchType) {
        //缓存获取敏感词汇原记录
        List<String> words = Cache.get(WORDS);
        if (CollectionUtils.isEmpty(words)) {
            //读取敏感词汇文件，存入缓存
            words = readWordsFile();
            Cache.put(WORDS, words);
        }

        if (CollectionUtils.isEmpty(words)) {
            return txt;
        }

        if (sensitiveWordMap == null) {
            init(words);
        }

        String resultTxt = txt;
        //获取所有的敏感词
        List<String> listWords = getSensitiveWord(txt, matchType);
        Iterator<String> iterator = listWords.iterator();
        String word;
        while (iterator.hasNext()) {
            word = iterator.next();
            resultTxt = resultTxt.replaceAll(word, getReplaceChars(maskChar, word.length()));
        }

        return resultTxt;
    }

    /**
     * 读取敏感词汇文件
     */
    public static List<String> readWordsFile() {
        List<String> listWords = new ArrayList<>();
        InputStream inputStream = null;
        InputStreamReader inputStreamReader = null;
        BufferedReader bufferedReader = null;
        try {
            Resource resource = new DefaultResourceLoader().getResource("classpath:shield_word.txt");
            inputStream = resource.getInputStream();
            inputStreamReader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);
            bufferedReader = new BufferedReader(inputStreamReader);
            String word = "";
            while (StringUtils.isNotBlank(word = bufferedReader.readLine())) {
                listWords.add(word);
            }

            /**
             * 加入到缓存
             */
            if (listWords.size() > 0){
                Cache.put(SensitiveWordUtil.WORDS,listWords);
            }

            bufferedReader.close();
            inputStreamReader.close();
            inputStream.close();
        } catch (Exception e) {
            log.error("读取敏感词汇文件出错", e);
        } finally {
            try {
                if (bufferedReader != null) {
                    bufferedReader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
                if (inputStream != null) {
                    inputStream.close();
                }
            } catch (Exception e) {
                log.error("读取敏感词汇文件出错", e);
            }
        }
        return listWords;
    }

    /**
     * 获取替换字符串
     */
    private static String getReplaceChars(char maskChar, int length) {
        StringBuilder resultReplace = new StringBuilder(String.valueOf(maskChar));
        for (int i = 1; i < length; i++) {
            resultReplace.append(maskChar);
        }

        return resultReplace.toString();
    }

    /**
     * 检查文字中是否包含敏感字符，检查规则如下：<br>
     * @return 如果存在，则返回敏感词字符的长度，不存在返回0
     */
    private static int checkSensitiveWord(String txt, int beginIndex, int matchType) {
        //敏感词结束标识位：用于敏感词只有1位的情况
        boolean flag = false;
        //匹配标识数默认为0
        int matchFlag = 0;
        char word;
        Map nowMap = sensitiveWordMap;
        for (int i = beginIndex; i < txt.length(); i++) {
            word = txt.charAt(i);
            //获取指定key
            nowMap = (Map) nowMap.get(word);
            if (nowMap != null) {//存在，则判断是否为最后一个
                //找到相应key，匹配标识+1
                matchFlag++;
                //如果为最后一个匹配规则,结束循环，返回匹配标识数
                if ("1".equals(nowMap.get("isEnd"))) {
                    //结束标志位为true
                    flag = true;
                    //最小规则，直接返回,最大规则还需继续查找
                    if (MinMatchTYpe == matchType) {
                        break;
                    }
                }
            } else {//不存在，直接返回
                break;
            }
        }
        if (matchFlag < 2 || !flag) {//长度必须大于等于1，为词
            matchFlag = 0;
        }
        return matchFlag;
    }

    public static void main(String[] args) {
        String str = "我们伟大的领袖毛泽东创立新中国";
        System.out.println("--->替换前:" + str);
        System.out.println("--->替换后:" + SensitiveWordUtil.replaceSensitiveWord(str, 2));
    }
}