package com.ohaotian.venus.extension.handler;

import com.ohaotian.venus.extension.Extension;
import com.ohaotian.venus.extension.ExtensionProperties;
import com.ohaotian.venus.extension.processor.ExtensionRepository;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.List;

/**
 * 标题：类名称:ExtensionAnnotationHandler
 * 说明：实现@AutoExtension注解的功能
 * 时间：2023/8/10 11:23 AM
 * 作者 @author hegy2017
 */
@Slf4j
@Component
public class ExtensionAnnotationHandler implements ExtensionAnnotationHandle, ApplicationContextAware {

    private ApplicationContext applicationContext;

    @Autowired
    private ExtensionProperties extensionProperties;

    @Autowired
    private ExtensionRepository repository;

    @Override
    public boolean hasExist(Field field) {
        if (field == null) {
            return false;
        }
        for (Annotation annotation : field.getDeclaredAnnotations()) {
            if (annotation instanceof Extension) {
                return true;
            }
        }
        return false;

    }

    @Override
    public Object invoke(Object bean, Field field) {

        try {
            if (!field.isAccessible()) {
                field.setAccessible(true);
            }
            if (field.isAnnotationPresent(Extension.class)) {
                Object value;
                Extension extensionAnnotation = field.getAnnotation(Extension.class);
                String originalBeanName = extensionAnnotation.name();
                String extensionBeanName = field.getType().getSimpleName();

                log.debug("发现注解@Extension,解析后的信息为：originalBeanName=" + originalBeanName + ",扩展信息extensionBeanName=" + extensionBeanName);
                //判定扩展不存在
                if (!hasExtensionBean(extensionBeanName)) {
                    //判断原始不存在，只能注入空
                    if (!hasOriginalBean(originalBeanName)) {
                        value = field.getType().cast(DynamicProxyFactory.createProxy(field.getType()));
                        log.debug("没有扩展实现，没有产品默认实现,请注意，赋值为空！！！！！！：");
                    } else {
                        //注入原始
                        Object tempValue = repository.getOriginalRepo().get(convertFirstLetterToLowerCase(originalBeanName));
                        value = field.getType().cast(tempValue);
                        log.debug("没有扩展实现，有产品的默认实现,使用产品实现：" + originalBeanName + ",找到对象对象的名称：" + value.getClass().getSimpleName());
                    }

                    field.setAccessible(true);
                    field.set(bean, value);
                } else {
                    //bean不做处理
                    log.debug("找到扩展实现，使用扩展的实现：" + bean.getClass().getSimpleName());
                }

            }

            return bean;

        } catch (Exception e) {
            log.error("Failed to init  at filed " + field.getName() + " in class " + bean.getClass().getName() + ", cause: " + e.getMessage(), e);
        }

        return bean;
    }

    @Override
    public void repository(Object bean, Field field) {
        try {
            if (field.isAnnotationPresent(Extension.class)) {
                Extension extensionAnnotation = field.getAnnotation(Extension.class);

                Class<?> fieldType = field.getType();
                String extBeanName = fieldType.getSimpleName();
                //默认扩展类名称为ExtPt结尾
                String originalBeanName = StringUtils.isEmpty(extensionAnnotation.name()) ? extBeanName.substring(0, extBeanName.length() - 5) : extensionAnnotation.name();
                //存储原始类
                repositoryOriginalBean(originalBeanName);
                //存储扩展类
                repositoryExtensionBean(extBeanName);
            }

        } catch (Exception e) {
            log.error("Failed to repository " + field.getName() + " in class " + bean.getClass().getName() + ", cause: " + e.getMessage(), e);
        }
    }

    /**
     * 判断是否有原始实现
     *
     * @param clazzName
     * @return
     */
    private boolean hasOriginalBean(String clazzName) {
        clazzName = convertFirstLetterToLowerCase(clazzName);
        return repository.getOriginalRepo().containsKey(clazzName);
    }

    /**
     * 首个字母统一为小写
     *
     * @param input
     * @return
     */
    public static String convertFirstLetterToLowerCase(String input) {
        if (input == null || input.isEmpty()) {
            return input;
        }

        char firstChar = input.charAt(0);
        if (Character.isUpperCase(firstChar)) {
            return Character.toLowerCase(firstChar) + input.substring(1);
        } else {
            return input;
        }
    }

    /**
     * 存储原始bean的实现对象
     *
     * @param clazzName
     */
    private void repositoryOriginalBean(String clazzName) {
        clazzName = convertFirstLetterToLowerCase(clazzName);
        if (!repository.getOriginalRepo().containsKey(clazzName)) {
            try {
                final Object bean = applicationContext.getBean(clazzName + "Impl");
                repository.getOriginalRepo().put(clazzName, bean);
            } catch (Exception e) {
                log.debug("没有找到对应的实现:" + clazzName + "Impl");
            }
        }
    }

    /**
     * 存储扩展bean的是否存在判定
     *
     * @param clazzName
     */
    private void repositoryExtensionBean(String clazzName) {
        if (!repository.getExtensionRepo().containsKey(clazzName)) {
            if (!StringUtils.isEmpty(extensionProperties.getExtBeans())) {
                String extBeans = extensionProperties.getExtBeans();
                List<String> extBeanList = Arrays.asList(extBeans.split(","));
                if (!extBeanList.isEmpty() && extBeanList.contains(clazzName)) {
                    repository.getExtensionRepo().put(clazzName, Boolean.TRUE);
                }
            }

        }
    }

    /**
     * 判断是否有扩展实现,通过配置文件，转换为List，进行判定
     *
     * @param clazzName
     * @return
     */
    private boolean hasExtensionBean(String clazzName) {
        return repository.getExtensionRepo().containsKey(clazzName);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
