package com.tydic.datasync.event.advice;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.ohaotian.plugin.cache.CacheClient;
import com.tydic.datasync.event.bo.CacheEvent;
import com.tydic.datasync.event.bo.DataSyncConstant;
import com.tydic.dyc.base.bo.BaseRspBo;
import com.tydic.dyc.base.constants.MdcConstants;
import com.tydic.dyc.base.events.Event;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;

/**
 * 标题：类名称:DataSyncDomainEventByPointCutAdvice
 * 说明：切面拦截器，处理事件
 * 时间：2022/3/2 11:26 AM
 * 作者 @author hegy2017
 */
@Slf4j
public class DataSyncDomainEventByPointCutAdvice implements MethodInterceptor {

    /**
     * 缓存对象
     */
    private CacheClient cacheClient;

    public DataSyncDomainEventByPointCutAdvice(CacheClient cacheClient) {
        this.cacheClient = cacheClient;
    }

    /**
     * 这里要做的事情有，拿取对应的方法名称，聚合名称，组装为key，从缓存获取事件列表
     * 和事件入参字段
     *
     * @param invocation
     * @return
     * @throws Throwable
     */
    @Override
    public Object invoke(MethodInvocation invocation) throws Throwable {

        Map<String, Object> fieldValueMap = new HashMap<>(16);
        BaseRspBo rspObj = (BaseRspBo) invocation.proceed();
        // 入参处理
        dealInArgument(invocation.getArguments(), fieldValueMap);
        //获取当前请求的流水
        String traceId = (String) fieldValueMap.get(MdcConstants.REQUEST_ID);

        //如果为空，给一个，避免下面报空指针，让中心层可以独立测试
        if(ObjectUtil.isEmpty(traceId)){
            traceId = UUID.randomUUID().toString().replace("-", "");
        }

        //通过流水，获取当前流水中已经存储好的事件集合
        final String eventsList = (String) cacheClient.get(traceId);
        List<Event> traceEventList;
        if (eventsList != null) {
            JSONArray eventArray = JSONArray.parseArray(eventsList);
            traceEventList = JSONArray.parseArray(eventArray.toString(), Event.class);
        } else {
            traceEventList = new ArrayList<>();
        }
        /**
         *1.获取输入输出参数和里面对应的值,出参处理,对请求和响应参数key，value封装到map中，同名参数->按照出参数覆盖入参数的原则
         */
        dealObject(rspObj, fieldValueMap);

        /**
         * 2.获取事件配置
         */
        List<CacheEvent> cacheEvents = getMethodCacheEvent(invocation);

        /**
         * 3.设置返回值到具体的事件上
         */
        final List<Event> events = dealMethodEventParam(cacheEvents, fieldValueMap);

        /**
         * 4.缓存当前方法所有事件
         */
        traceEventList.addAll(events);
        cacheClient.set(traceId, JSON.toJSONString(traceEventList));


        return rspObj;
    }


    /**
     * 组装redis的key，获取该方法的事件配置信息,通过配置中心加载到缓存中的映射,
     * redis中的信息：UocSaveOrderServiceImpl-saveOrder 存入对象为List<CacheEvent>
     *
     * @param invocation
     * @return
     */
    private List<CacheEvent> getMethodCacheEvent(MethodInvocation invocation) {
        StringBuilder builder = new StringBuilder();
        Method method = invocation.getMethod();
        String className = method.getDeclaringClass().getName();
        String methodName = method.getName();
        builder.append(className).append(DataSyncConstant.CONNECTOR_SYMBOL).append(methodName);


        String eventJson = (String) cacheClient.get(builder.toString());
        if (ObjectUtil.isNotEmpty(eventJson)) {
            JSONArray eventArray = JSONArray.parseArray(eventJson);
            return JSONArray.parseArray(eventArray.toString(), CacheEvent.class);
        } else {
            return new ArrayList<>();
        }
    }


    /**
     * 处理返回对象信息
     *
     * @param cacheEvents
     * @param fieldValueMap
     * @return
     */
    private List<Event> dealMethodEventParam(List<CacheEvent> cacheEvents, Map<String, Object> fieldValueMap) {
        List<Event> rspEventList = new ArrayList<>();
        cacheEvents.forEach(cacheEvent -> {
            Event event = new Event();
            event.setCode(cacheEvent.getEventCode());
            event.setModel(cacheEvent.getModule());

            Map<String, Object> valueMap = new HashMap<>(16);
            final List<String> params = cacheEvent.getParams();
            params.forEach(filed -> {
                final Object value = fieldValueMap.get(filed);
                if (value == null) {
                    throw new ZTBusinessException("配置的字段:" + filed + ",在输入，输出结果中没有定义!");
                } else {
                    valueMap.put(filed, value);
                }
            });
            event.setParamMap(valueMap);
            rspEventList.add(event);
        });
        return rspEventList;
    }

    /**
     * 获取所有入参的集合
     *
     * @param arguments
     * @return
     */
    private void dealInArgument(Object[] arguments, Map<String, Object> map) throws Throwable {
        if (arguments != null) {
            for (Object arg : arguments) {
                dealObject(arg, map);
            }
        }
    }

    /**
     * 对单个对象进行处理
     *
     * @param dealObject
     * @throws Throwable
     */
    private void dealObject(Object dealObject, Map<String, Object> fieldValueMap) throws Throwable {

        final List<Field> fields = getAllField(dealObject.getClass());
        for (Field f : fields) {
            //设置些属性是可以访问的
            f.setAccessible(true);
            //属性名
            String fieldName = f.getName();
            if (null != f.get(dealObject)) {
                Object fieldValue = f.get(dealObject);
                fieldValueMap.put(fieldName, fieldValue);
            }
        }
    }

    /**
     * 获取当前类的所有字段
     *
     * @param clazz
     * @return
     */
    private static List<Field> getAllField(Class clazz) {
        List<Field> fields = new ArrayList<>();
        while (clazz != null) {
            fields.addAll(new ArrayList<>(Arrays.asList(clazz.getDeclaredFields())));
            clazz = clazz.getSuperclass();
        }
        return fields;
    }

}
