package com.tydic.bcm.personal.common.impl;

import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.ohaotian.plugin.db.Page;
import com.tydic.bcm.personal.common.api.BcmCostCenterSyncTaskService;
import com.tydic.bcm.personal.common.bo.BcmSyncCostCenterReqBO;
import com.tydic.bcm.personal.common.bo.BcmSyncCostCenterRspBO;
import com.tydic.bcm.personal.constants.BcmDealResultEnum;
import com.tydic.bcm.personal.constants.BcmPersonalCommonConstant;
import com.tydic.bcm.personal.dao.BcmCostInfoMapper;
import com.tydic.bcm.personal.dao.BcmFmisCostcenterMapper;
import com.tydic.bcm.personal.dao.BcmOrgOrgsFmisMapper;
import com.tydic.bcm.personal.po.*;
import com.tydic.bcm.personal.utils.BcmIdUtil;
import com.tydic.bcm.personal.utils.BcmRuUtil;
import lombok.extern.ohaotian.HTServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 标题 BcmCostCenterSyncTaskServiceImpl
 * 说明 成本中心同步比对任务
 * 时间 2023/8/15 14:09
 *
 * @author 唐富强
 */
@HTServiceImpl
public class BcmCostCenterSyncTaskServiceImpl implements BcmCostCenterSyncTaskService {

    /**
     * 成本中心;NC基础数据临时表：成本中心Mapper
     */
    @Autowired
    private BcmFmisCostcenterMapper bcmFmisCostcenterMapper;

    /**
     * 成本中心Mapper
     */
    @Autowired
    private BcmCostInfoMapper bcmCostInfoMapper;

    /**
     * 财务组织;NC基础数据临时表：财务组织Mapper
     */
    @Autowired
    private BcmOrgOrgsFmisMapper bcmOrgOrgsFmisMapper;

    @Value("${dealSize:100}")
    private Integer dealSize;



    /**
     * 成本中心同步比对任务
     *
     * @param reqBO
     * @return
     */
    @Override
    public BcmSyncCostCenterRspBO syncCostCenter(BcmSyncCostCenterReqBO reqBO) {
        //数据同步
        sync();
        return BcmRuUtil.success(BcmSyncCostCenterRspBO.class);
    }


    /**
     * 待同步成本中心数据获取
     *
     * @return
     */
    private Page<BcmFmisCosetCenterPO> getFmisCosetCenterList(Integer pageNo, Integer pageSize) {
        BcmQueryFmisCosetCenterPO bcmQueryFmisCosetCenterPO = new BcmQueryFmisCosetCenterPO();
        bcmQueryFmisCosetCenterPO.setDealResult(BcmDealResultEnum.PENDING.getDealResult());
        Page<BcmFmisCosetCenterPO> page = new Page<>(pageNo, pageSize);
        List<BcmFmisCosetCenterPO> list = bcmFmisCostcenterMapper.getPageList(bcmQueryFmisCosetCenterPO, page);
        page.setResult(list);
        return page;
    }

    /**
     * 成本中心数据获取
     *
     * @return
     */
    private Map<String, BcmCostInfoPO> getCostMap(List<BcmFmisCosetCenterPO> fmisCosetCenterList) {
        BcmQueryCostInfoPO bcmQueryCostInfoPO = new BcmQueryCostInfoPO();
        bcmQueryCostInfoPO.setCostCenterCodeList(fmisCosetCenterList.stream().map(BcmFmisCosetCenterPO::getCcCode).collect(Collectors.toList()));
        bcmQueryCostInfoPO.setDelFlag(BcmPersonalCommonConstant.DelFlag.NO);
        List<BcmCostInfoPO> costList = bcmCostInfoMapper.getList(bcmQueryCostInfoPO);
        if (CollectionUtils.isEmpty(costList)) {
            return null;
        }
        Map<String, BcmCostInfoPO> costMap = costList.stream().collect(Collectors.toMap(BcmCostInfoPO::getCostCenterCode, o -> o, (k1, k2) -> k2));
        return costMap;
    }

    /**
     * 数据对比同步
     */
    private void sync() {
        Page<BcmFmisCosetCenterPO> page = getFmisCosetCenterList(1, dealSize);
        List<BcmFmisCosetCenterPO> fmisCosetCenterList = page.getResult();
        //没有待同步数据直接结束
        if (CollectionUtils.isEmpty(fmisCosetCenterList)) {
            return;
        }
        //成本中心Map获取
        Map<String, BcmCostInfoPO> costMap = getCostMap(fmisCosetCenterList);
        //处理成功id集合
        List<Long> successIdList = new ArrayList<>(fmisCosetCenterList.size());
        //无需处理id集合
        List<Long> skipIdList = new ArrayList<>(fmisCosetCenterList.size());

        List<Long> failIdList = new ArrayList<>(fmisCosetCenterList.size());
        /**
         * 组织信息map获取
         */
        Map<String, List<BcmOrgOrgsFmisPO>> orgListMap = getOrgListMap(fmisCosetCenterList);
        //成本中心数据为null所有数据做新增处理
        if (CollectionUtils.isEmpty(costMap)) {
            Map<String, List<Long>> idMap = addCost(fmisCosetCenterList, orgListMap);
            successIdList = idMap.get(BcmDealResultEnum.SUCCESS.getDealResult());
            failIdList = idMap.get(BcmDealResultEnum.FAIL.getDealResult());
        } else {
            Date now = new Date();
            List<BcmFmisCosetCenterPO> addList = new ArrayList<>(fmisCosetCenterList.size());
            List<BcmCostInfoPO> updateList = new ArrayList<>(fmisCosetCenterList.size());
            for (BcmFmisCosetCenterPO fmisCosetCenterPO : fmisCosetCenterList) {
                BcmCostInfoPO costInfoPO = costMap.get(fmisCosetCenterPO.getCcCode());
                if (costInfoPO == null) {
                    //成本中心不存在新增
                    addList.add(fmisCosetCenterPO);
                } else {
                    List<BcmOrgOrgsFmisPO> orgList = orgListMap.get(fmisCosetCenterPO.getOrgCode());
                    //当存在数据差异需要更新
                    if (!costInfoPO.getCostCenterName().equals(fmisCosetCenterPO.getCcName())
                            || !costInfoPO.getFinancialOrgName().equals(fmisCosetCenterPO.getOrgName())
                            || !costInfoPO.getFinancialOrgCode().equals(fmisCosetCenterPO.getOrgCode())) {
                        BcmCostInfoPO updatePO = new BcmCostInfoPO();
                        //未获取到组织信息更新失败
                        if (CollectionUtils.isEmpty(orgList)) {
                            failIdList.add(fmisCosetCenterPO.getId());
                            continue;
                        }
                        //组织信息获取
                        BcmOrgOrgsFmisPO orgOrgsFmisPO = getOrg(orgList);
                        updatePO.setCostCenterCode(fmisCosetCenterPO.getCcCode());
                        updatePO.setCostCenterName(fmisCosetCenterPO.getCcName());
                        updatePO.setUpdateTime(now);
                        updatePO.setFinancialOrgCode(fmisCosetCenterPO.getOrgCode());
                        updatePO.setFinancialOrgName(fmisCosetCenterPO.getOrgName());
                        updatePO.setFinancialOrgId(orgOrgsFmisPO.getPkOrg());
                        updateList.add(updatePO);
                        successIdList.add(fmisCosetCenterPO.getId());
                    } else {
                        //无差异不需要更新
                        skipIdList.add(fmisCosetCenterPO.getId());
                    }
                }

            }
            //成本中心新增
            if (!CollectionUtils.isEmpty(addList)) {
                Map<String, List<Long>> idMap = addCost(fmisCosetCenterList, orgListMap);
                successIdList.addAll(idMap.get(BcmDealResultEnum.SUCCESS.getDealResult()));
                failIdList.addAll(idMap.get(BcmDealResultEnum.FAIL.getDealResult()));
            }

            //更新成本中心
            if (!CollectionUtils.isEmpty(updateList)) {
                bcmCostInfoMapper.updateBatch(updateList);
            }
        }
        //同步成功结果更新
        if (!CollectionUtils.isEmpty(successIdList)) {
            updateFmisCosetCenter(successIdList, BcmDealResultEnum.SUCCESS.getDealResult());
        }
        //无需同步结果更新
        if (!CollectionUtils.isEmpty(skipIdList)) {
            updateFmisCosetCenter(skipIdList, BcmDealResultEnum.SKIP.getDealResult());
        }
        //同步失败结果更新
        if (!CollectionUtils.isEmpty(failIdList)) {
            updateFmisCosetCenter(failIdList, BcmDealResultEnum.FAIL.getDealResult());
        }
        //当页码大于1再次执行同步
        if (page.getTotalPages() > 1) {
            sync();
        }
    }


    /**
     * 新增成本中心数据
     *
     * @param fmisCosetCenterList
     */
    private Map<String, List<Long>> addCost(List<BcmFmisCosetCenterPO> fmisCosetCenterList, Map<String, List<BcmOrgOrgsFmisPO>> orgListMap) {
        List<BcmCostInfoPO> bcmCostInfoPOS = new ArrayList<>(fmisCosetCenterList.size());
        Date now = new Date();
        //成功同步id集合
        List<Long> successIdList = new ArrayList<>(fmisCosetCenterList.size());
        //同步失败id集合
        List<Long> failIdList = new ArrayList<>(fmisCosetCenterList.size());
        Map<String, List<Long>> idMap = new HashMap<>(2);
        for (BcmFmisCosetCenterPO fmisCosetCenterPO : fmisCosetCenterList) {
            List<BcmOrgOrgsFmisPO> orgList = orgListMap.get(fmisCosetCenterPO.getOrgCode());
            //未获取到组织信息更新失败
            if (CollectionUtils.isEmpty(orgList)) {
                failIdList.add(fmisCosetCenterPO.getId());
                continue;
            }
            //组织信息获取
            BcmOrgOrgsFmisPO orgOrgsFmisPO = getOrg(orgList);
            BcmCostInfoPO bcmCostInfoPO = new BcmCostInfoPO();
            bcmCostInfoPO.setId(BcmIdUtil.nextId());
            bcmCostInfoPO.setStatus(BcmPersonalCommonConstant.Status.ENABLE);
            bcmCostInfoPO.setCostCenterCode(fmisCosetCenterPO.getCcCode());
            bcmCostInfoPO.setCostCenterName(fmisCosetCenterPO.getCcName());
            bcmCostInfoPO.setFinancialOrgId(orgOrgsFmisPO.getPkOrg());
            bcmCostInfoPO.setFinancialOrgName(orgOrgsFmisPO.getName());
            bcmCostInfoPO.setFinancialOrgCode(orgOrgsFmisPO.getCode());
            bcmCostInfoPO.setCreateTime(now);
            bcmCostInfoPO.setUpdateTime(now);
            bcmCostInfoPO.setDelFlag(BcmPersonalCommonConstant.DelFlag.NO);
            bcmCostInfoPOS.add(bcmCostInfoPO);
            successIdList.add(fmisCosetCenterPO.getId());
        }
        if (!CollectionUtils.isEmpty(bcmCostInfoPOS)) {
            bcmCostInfoMapper.insertBatch(bcmCostInfoPOS);
        }
        idMap.put(BcmDealResultEnum.SUCCESS.getDealResult(), successIdList);
        idMap.put(BcmDealResultEnum.FAIL.getDealResult(), failIdList);
        return idMap;
    }


    /**
     * 更新NC基础数据临时表：成本中心
     *
     * @param idList
     * @param dealResult
     */
    private void updateFmisCosetCenter(List<Long> idList, String dealResult) {
        BcmQueryFmisCosetCenterPO bcmQueryFmisCosetCenterPO = new BcmQueryFmisCosetCenterPO();
        bcmQueryFmisCosetCenterPO.setIdList(idList);
        bcmQueryFmisCosetCenterPO.setDealResult(dealResult);
        bcmQueryFmisCosetCenterPO.setDealTime(new Date());
        bcmFmisCostcenterMapper.updateBatchById(bcmQueryFmisCosetCenterPO);
    }

    /**
     * 组织信息Map获取
     *
     * @param fmisCosetCenterList
     * @return
     */
    private Map<String, List<BcmOrgOrgsFmisPO>> getOrgListMap(List<BcmFmisCosetCenterPO> fmisCosetCenterList) {
        Set<String> orgCodeSet = fmisCosetCenterList.stream().map(BcmFmisCosetCenterPO::getOrgCode).collect(Collectors.toSet());
        BcmQueryOrgOrgsFmisPO bcmQueryOrgOrgsFmisPO = new BcmQueryOrgOrgsFmisPO();
        bcmQueryOrgOrgsFmisPO.setOrgCodeList(new ArrayList<>(orgCodeSet));
        List<BcmOrgOrgsFmisPO> orgList = bcmOrgOrgsFmisMapper.getList(bcmQueryOrgOrgsFmisPO);
        if (CollectionUtils.isEmpty(orgList)) {
            throw new ZTBusinessException("未获取到组织信息");
        }
        return orgList.stream().collect(Collectors.groupingBy(BcmOrgOrgsFmisPO::getCode));
    }

    /**
     * 获取最新的一条组织信息数据
     *
     * @param orgList
     * @return
     */
    private BcmOrgOrgsFmisPO getOrg(List<BcmOrgOrgsFmisPO> orgList) {
        orgList = orgList.stream().sorted(Comparator.comparing(BcmOrgOrgsFmisPO::getPushTime).reversed()).collect(Collectors.toList());
        return orgList.get(0);
    }
}
