package com.tydic.bcm.personal.task.impl;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.nacos.client.utils.StringUtils;
import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.ohaotian.plugin.db.Page;
import com.tydic.bcm.personal.constants.BcmDealResultEnum;
import com.tydic.bcm.personal.constants.BcmPersonalCommonConstant;
import com.tydic.bcm.personal.dao.BcmFmisPaymentRecordMapper;
import com.tydic.bcm.personal.dao.BcmSyncLogMapper;
import com.tydic.bcm.personal.po.BcmFmisPaymentRecordPO;
import com.tydic.bcm.personal.po.BcmQueryFmisPaymentRecordPO;
import com.tydic.bcm.personal.po.BcmSyncLogPO;
import com.tydic.bcm.personal.task.api.BcmFmisPaymentRecordSyncTaskService;
import com.tydic.bcm.personal.task.bo.BcmFmisPaymentRecordSyncTaskReqBO;
import com.tydic.bcm.personal.task.bo.BcmFmisPaymentRecordSyncTaskRspBO;
import com.tydic.bcm.personal.task.bo.BcmFscPayInfoBackAbilityReqBO;
import com.tydic.bcm.personal.utils.BcmHttpUtil;
import com.tydic.bcm.personal.utils.BcmIdUtil;
import com.tydic.bcm.personal.utils.BcmRuUtil;
import com.tydic.dyc.base.bo.BaseRspBo;
import com.tydic.dyc.base.constants.BaseRspConstant;
import lombok.extern.ohaotian.HTServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 标题 BcmFmisPaymentRecordSyncTaskServiceImpl
 * 说明 付款记录同步对比服务实现
 * 时间 2023/8/18 10:14
 *
 * @author 唐富强
 */
@HTServiceImpl
public class BcmFmisPaymentRecordSyncTaskServiceImpl implements BcmFmisPaymentRecordSyncTaskService {

    /**
     * 付款记录临时表Mapper
     */
    @Autowired
    private BcmFmisPaymentRecordMapper bcmFmisPaymentRecordMapper;

    /**
     * 同步记录Mapper
     */
    @Autowired
    private BcmSyncLogMapper bcmSyncLogMapper;

    /**
     * 每次处理条数
     */
    @Value("${dealSize:100}")
    private Integer dealSize;

    /**
     * 推送数据能力平台地址
     */
    @Value("${SYNC_PAYMENT_RECORD_URL:url}")
    private String syncPaymentRecordUrl;

    /**
     * 付款记录同步对比服务实现
     *
     * @param reqBO
     * @return
     */
    @Override
    public BcmFmisPaymentRecordSyncTaskRspBO syncPaymentRecord(BcmFmisPaymentRecordSyncTaskReqBO reqBO) {
        //数据同步对比
        sync();
        return BcmRuUtil.success(BcmFmisPaymentRecordSyncTaskRspBO.class);
    }

    /**
     * 数据同步对比
     */
    private void sync() {
        Page<BcmFmisPaymentRecordPO> syncPage = getFmisPaymentRecordList();
        List<BcmFmisPaymentRecordPO> syncList = syncPage.getResult();
        List<Long> succssIdList = new ArrayList<Long>(syncList.size());
        List<Long> failIdList = new ArrayList<Long>(syncList.size());
        if (CollectionUtils.isEmpty(syncList)) {
            return;
        }
        BcmSyncLogPO syncLogPO = getSyncLog();
        if (syncLogPO == null) {

            for (BcmFmisPaymentRecordPO po : syncList) {
                boolean check = pushAbility(po);
                if (check) {
                    succssIdList.add(po.getId());
                } else {
                    failIdList.add(po.getId());
                }
            }
            //新增同步记录
            addSyncLog();
        } else {
            //无需同步id集合
            List<Long> skipIdList = new ArrayList<Long>(syncList.size());
            //需要推送到能力平台的数据
            List<BcmFmisPaymentRecordPO> addList = new ArrayList<>(syncList.size());
            //待同步的结算单id集合
            List<String> storeIdList = syncList.stream().map(BcmFmisPaymentRecordPO::getStoreId).collect(Collectors.toList());
            //对比数据
            Map<String, BcmFmisPaymentRecordPO> thanMap = getThanMap(storeIdList, syncLogPO.getLastDate());
            //无对比数据
            if (CollectionUtils.isEmpty(thanMap)) {
                addList.addAll(syncList);
                succssIdList.addAll(syncList.stream().map(BcmFmisPaymentRecordPO::getId).collect(Collectors.toList()));
            } else {
                //存在对比数据
                for (BcmFmisPaymentRecordPO fee : syncList) {
                    BcmFmisPaymentRecordPO than = thanMap.get(fee.getStoreId());
                    if (than == null) {
                        addList.add(fee);
                    } else {
                        Long id = fee.getId();
                        setComparePO(fee);
                        setComparePO(than);
                        if (fee.hashCode() != than.hashCode()) {
                            addList.add(fee);
                        } else {
                            skipIdList.add(id);
                        }
                    }
                }
            }
            //推送能力平台
            if (!CollectionUtils.isEmpty(addList)) {
                for (BcmFmisPaymentRecordPO po : addList) {
                    boolean check = pushAbility(po);
                    if (check) {
                        succssIdList.add(po.getId());
                    } else {
                        failIdList.add(po.getId());
                    }
                }

            }
            //更新付款记录临时表处理结果
            if (!CollectionUtils.isEmpty(syncList)) {
                updateFmisPaymentRecord(skipIdList, BcmDealResultEnum.SKIP.getDealResult());
            }
            //更新同步记录
            updateSyncLog(syncLogPO.getId());
        }
        //更新付款记录临时表处理结果
        if (!CollectionUtils.isEmpty(succssIdList)) {
            updateFmisPaymentRecord(succssIdList, BcmDealResultEnum.SUCCESS.getDealResult());
        }
        if (!CollectionUtils.isEmpty(failIdList)) {
            updateFmisPaymentRecord(failIdList, BcmDealResultEnum.FAIL.getDealResult());
        }
        if (syncPage.getTotalPages() > 1) {
            sync();
        }
    }

    /**
     * 对比PO获取
     *
     * @param po
     * @return
     */
    private void setComparePO(BcmFmisPaymentRecordPO po) {
        po.setId(null);
        po.setDealTime(null);
        po.setDealResult(null);
    }

    /**
     * 对比数据获取
     *
     * @param storeIdList
     * @param pushTime
     * @return
     */
    private Map<String, BcmFmisPaymentRecordPO> getThanMap(List<String> storeIdList, Date pushTime) {
        BcmQueryFmisPaymentRecordPO bcmQueryFmisPaymentRecordPO = new BcmQueryFmisPaymentRecordPO();
        bcmQueryFmisPaymentRecordPO.setPushTime(pushTime);
        bcmQueryFmisPaymentRecordPO.setStoreIdList(storeIdList);
        bcmQueryFmisPaymentRecordPO.setDealResult(BcmDealResultEnum.SUCCESS.getDealResult());
        List<BcmFmisPaymentRecordPO> list = bcmFmisPaymentRecordMapper.getList(bcmQueryFmisPaymentRecordPO);
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        return list.stream().collect(Collectors.toMap(BcmFmisPaymentRecordPO::getStoreId, o -> o, (k1, k2) -> k2));
    }

    /**
     * 数据推送到能力平台
     *
     * @param po
     */
    private Boolean pushAbility(BcmFmisPaymentRecordPO po) {

        BcmFscPayInfoBackAbilityReqBO bcmFscPayInfoReqBO = new BcmFscPayInfoBackAbilityReqBO();
        bcmFscPayInfoReqBO.setBusiAmount(new BigDecimal(po.getPaymentAmount()));
        bcmFscPayInfoReqBO.setBusiOrderNo(po.getStoreCode());
        bcmFscPayInfoReqBO.setBusiOrderId(Long.valueOf(po.getStoreId()));
        try {
            bcmFscPayInfoReqBO.setBusiTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(po.getPaymentTime()));
        } catch (ParseException e) {
            throw new ZTBusinessException("时间格式错误");
        }

        String rspStr = BcmHttpUtil.doPost(syncPaymentRecordUrl, JSONObject.toJSONString(bcmFscPayInfoReqBO));
        if (StringUtils.isBlank(rspStr)) {
            return false;
        }
        BaseRspBo baseRspBo = BcmRuUtil.jss(rspStr, BaseRspBo.class);
        return BaseRspConstant.RSP_CODE_SUCCESS.equals(baseRspBo.getRespCode());
    }

    /**
     * 同步记录
     *
     * @return
     */
    private BcmSyncLogPO getSyncLog() {
        BcmSyncLogPO bcmSyncLogPO = new BcmSyncLogPO();
        bcmSyncLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.PAYMENT_RECORD);
        return bcmSyncLogMapper.getModelBy(bcmSyncLogPO);
    }

    /**
     * 待同步付款记录获取
     *
     * @return
     */
    private Page<BcmFmisPaymentRecordPO> getFmisPaymentRecordList() {
        BcmFmisPaymentRecordPO bcmFmisPaymentRecordPO = new BcmFmisPaymentRecordPO();
        bcmFmisPaymentRecordPO.setDealResult(BcmDealResultEnum.PENDING.getDealResult());
        Page<BcmFmisPaymentRecordPO> page = new Page<>(1, dealSize);
        List<BcmFmisPaymentRecordPO> pageList = bcmFmisPaymentRecordMapper.getPageList(bcmFmisPaymentRecordPO, page);
        page.setResult(pageList);
        return page;
    }

    /**
     * 更新付款记录临时表
     *
     * @param idList
     * @param dealResult
     */
    private void updateFmisPaymentRecord(List<Long> idList, String dealResult) {
        BcmQueryFmisPaymentRecordPO bcmQueryFmisPaymentRecordPO = new BcmQueryFmisPaymentRecordPO();
        bcmQueryFmisPaymentRecordPO.setIdList(idList);
        bcmQueryFmisPaymentRecordPO.setDealResult(dealResult);
        bcmQueryFmisPaymentRecordPO.setDealTime(new Date());
        bcmFmisPaymentRecordMapper.updateBatch(bcmQueryFmisPaymentRecordPO);
    }

    /**
     * 新增同步记录
     */
    private void addSyncLog() {
        BcmSyncLogPO bcmSyncLogPO = new BcmSyncLogPO();
        bcmSyncLogPO.setId(BcmIdUtil.nextId());
        bcmSyncLogPO.setLastDate(new Date());
        bcmSyncLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.PAYMENT_RECORD);
        bcmSyncLogMapper.insert(bcmSyncLogPO);
    }

    /**
     * 更新同步记录
     *
     * @param id
     */
    private void updateSyncLog(Long id) {
        BcmSyncLogPO bcmSyncLogPO = new BcmSyncLogPO();
        bcmSyncLogPO.setId(id);
        bcmSyncLogPO.setLastDate(new Date());
        bcmSyncLogMapper.updateById(bcmSyncLogPO);
    }
}
