package com.tydic.bcm.personal.task.impl;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.tydic.bcm.personal.constants.BcmPersonalCommonConstant;
import com.tydic.bcm.personal.dao.BcmIpmpSupplierBankMapper;
import com.tydic.bcm.personal.dao.BcmSyncLogMapper;
import com.tydic.bcm.personal.dao.TEspSupplierBankMapper;
import com.tydic.bcm.personal.po.*;
import com.tydic.bcm.personal.task.api.BcmIpmpSupplierBankSyncTaskService;
import com.tydic.bcm.personal.task.bo.BcmIpmpSyncSupplierBankReqBO;
import com.tydic.bcm.personal.task.bo.BcmIpmpSyncSupplierBankRspBO;
import com.tydic.bcm.personal.utils.BcmIdUtil;
import com.tydic.dyc.base.constants.BaseRspConstant;
import lombok.extern.ohaotian.HTServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 类名： BcmIpmpSupplierBankSyncTaskServiceImpl
 * 说明：于(IPMP)供应商银行的同步与比对实现
 *
 * @author： lishiqing
 * 时间： 2023/9/11 15:29
 */
@HTServiceImpl
public class BcmIpmpSupplierBankSyncTaskServiceImpl implements BcmIpmpSupplierBankSyncTaskService {

    /**
     * 同步日志表--Mapper
     */
    @Autowired
    private BcmSyncLogMapper bcmSyncLogMapper;

    /**
     * 供应商银行账号临时表Mapper
     */
    @Autowired
    private TEspSupplierBankMapper tEspSupplierBankMapper;

    /**
     * 供应商银行账号Mapper
     */
    @Autowired
    private BcmIpmpSupplierBankMapper bcmIpmpSupplierBankMapper;

    @Override
    @Async("global-thread-pool")
    public BcmIpmpSyncSupplierBankRspBO syncIpmpSupplierBank(BcmIpmpSyncSupplierBankReqBO reqBO) {
        BcmIpmpSyncSupplierBankRspBO rspBO = new BcmIpmpSyncSupplierBankRspBO();
        // 判断有没有数据写入
        TEspSupplierBankQueryPO qryPO = new TEspSupplierBankQueryPO();
        qryPO.setDealResult(BcmPersonalCommonConstant.DealResult.PENDING);
        List<TEspSupplierBankPO> pageList = tEspSupplierBankMapper.getList(qryPO);
        if (ObjectUtil.isEmpty(pageList)) {
            rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
            rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
            return rspBO;
        }

        Date lastDate = pageList.get(0).getPushTime();

        // 根据字符串“IPMP_SUPPLIER_BANK”查询SYNC_LOG表中的记录，获取上次执行时间LAST_DATE，如果没有就为空
        BcmSyncLogPO syncLogPO = new BcmSyncLogPO();
        syncLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.IPMP_SUPPLIER_BANK);
        BcmSyncLogPO bcmSyncLogPO = bcmSyncLogMapper.getModelBy(syncLogPO);
        if (ObjectUtil.isNotEmpty(bcmSyncLogPO)) {
            // 已经同步过数据，查询以前存在的数据
            TEspSupplierBankQueryPO qryOldPO = new TEspSupplierBankQueryPO();
            qryOldPO.setDealResult(BcmPersonalCommonConstant.DealResult.SUCCESS);
            qryOldPO.setPushTime(bcmSyncLogPO.getLastDate());
            List<TEspSupplierBankPO> oldBankPOList = tEspSupplierBankMapper.getList(qryOldPO);
            Map<String, TEspSupplierBankPO> oldBankMap = oldBankPOList.stream().collect(Collectors.
                    toMap(TEspSupplierBankPO::getBankId, Function.identity(), (e1, e2) -> e2));

            // 循环、分页查询ORG_INFO_GUWP表中的数据：DEAL_RESULT字段为待同步状态的数据，数据处理完或无数据终止循环
            qryPO.setPushStartTime(bcmSyncLogPO.getLastDate());
            pageList = tEspSupplierBankMapper.getList(qryPO);
            if (ObjectUtil.isEmpty(pageList)) {
                rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
                rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
                return rspBO;
            }
            // 成功的就是要同步给能力平台的
            List<Long> successIdList = new ArrayList<>();
            // 新增的数据
            List<BcmIpmpSupplierBankPO> insertPOList = new ArrayList<>();
            // 未改变的ID集合
            List<Long> skipIdList = new ArrayList<>();
            // 遍历对比
            for (TEspSupplierBankPO newBankPO : pageList ) {
                // 如果不存在，则是新增
                if (!oldBankMap.containsKey(newBankPO.getBankId())) {
                    successIdList.add(newBankPO.getId());
                    BcmIpmpSupplierBankPO insertPO = JSON.parseObject(JSON.toJSONString(newBankPO),BcmIpmpSupplierBankPO.class);
                    insertPO.setId(BcmIdUtil.nextId());
                    insertPOList.add(insertPO);
                    newBankPO.setDealResult(BcmPersonalCommonConstant.DealResult.SUCCESS);
                    continue;
                }
                // 得到存在的OldBankPO
                TEspSupplierBankPO oldBankPO = oldBankMap.get(newBankPO.getBankId());
                // 获得新老对象的Hash值
                int oldHashCode = oldBankPO.hashCode();
                int newHashCode = newBankPO.hashCode();
                // hash码相同，对象的值也是相同的，所以没变化
                if (oldHashCode == newHashCode) {
                    skipIdList.add(newBankPO.getId());
                } else { // 设置成功
                    successIdList.add(newBankPO.getId());
                    BcmIpmpSupplierBankPO queryPO = new BcmIpmpSupplierBankPO();
                    queryPO.setBankId(newBankPO.getBankId());
                    BcmIpmpSupplierBankPO supplierBankPO = bcmIpmpSupplierBankMapper.getModelBy(queryPO);
                    if (ObjectUtil.isNotEmpty(supplierBankPO)) {
                        BcmIpmpSupplierBankPO setSupBankPO = JSON.parseObject(JSON.toJSONString(newBankPO),BcmIpmpSupplierBankPO.class);
                        BcmIpmpSupplierBankPO whereSupBankPO = new BcmIpmpSupplierBankPO();
                        whereSupBankPO.setId(supplierBankPO.getId());
                        bcmIpmpSupplierBankMapper.updateBy(setSupBankPO,whereSupBankPO);
                    } else {
                        BcmIpmpSupplierBankPO insertPO = JSON.parseObject(JSON.toJSONString(newBankPO),BcmIpmpSupplierBankPO.class);
                        insertPO.setId(BcmIdUtil.nextId());
                        insertPOList.add(insertPO);
                    }
                }
            }
            //判断无需同步的ID集合
            if (ObjectUtil.isNotEmpty(skipIdList)) {
                changeDealResult(skipIdList,BcmPersonalCommonConstant.DealResult.SKIP);
            }
            // 如果成功的数据不未空，同步给能力平台
            if (ObjectUtil.isNotEmpty(successIdList)) {
                changeDealResult(successIdList,BcmPersonalCommonConstant.DealResult.SUCCESS);
            }
            // 新增的数据
            if (ObjectUtil.isNotEmpty(insertPOList)) {
                bcmIpmpSupplierBankMapper.insertBatch(insertPOList);
            }
            // 数据处理完了，修改处理记录
            BcmSyncLogPO updateLogPO = new BcmSyncLogPO();
            updateLogPO.setId(bcmSyncLogPO.getId());
            updateLogPO.setLastDate(lastDate);
            bcmSyncLogMapper.updateById(updateLogPO);
        } else {
            // 对象转换
            List<BcmIpmpSupplierBankPO> insertPOList = pageList.stream().map(item -> {
                BcmIpmpSupplierBankPO insertPO = JSON.parseObject(JSON.toJSONString(item), BcmIpmpSupplierBankPO.class);
                insertPO.setId(BcmIdUtil.nextId());
                return insertPO;
            }).collect(Collectors.toList());
            bcmIpmpSupplierBankMapper.insertBatch(insertPOList);
            changeDealResult(pageList.stream().map(TEspSupplierBankPO::getId).collect(Collectors.toList()), BcmPersonalCommonConstant.DealResult.SUCCESS);

            // 在sync_log里面新增一条记录
            BcmSyncLogPO insertLogPO = new BcmSyncLogPO();
            insertLogPO.setId(BcmIdUtil.nextId());
            insertLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.IPMP_SUPPLIER_BANK);
            insertLogPO.setLastDate(lastDate);
            bcmSyncLogMapper.insert(insertLogPO);
        }

        rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
        rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
        return rspBO;
    }

    /**
     * 更新状态
     * @param ids
     */
    private void changeDealResult(List<Long> ids,String dealResult) {
        TEspSupplierBankUpdatePO setPO = new TEspSupplierBankUpdatePO();
        setPO.setDealResult(dealResult);
        setPO.setDealTime(new Date());
        TEspSupplierBankUpdatePO wherePO = new TEspSupplierBankUpdatePO();
        wherePO.setIdList(ids);
        tEspSupplierBankMapper.updateBy(setPO,wherePO);
    }
}
