package com.tydic.bcm.personal.task.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSONObject;
import com.ohaotian.plugin.db.Page;
import com.tydic.bcm.personal.constants.BcmDealResultEnum;
import com.tydic.bcm.personal.constants.BcmPersonalCommonConstant;
import com.tydic.bcm.personal.constants.BcmPersonalSettleConstant;
import com.tydic.bcm.personal.dao.BcmFmisPaymentRecordMapper;
import com.tydic.bcm.personal.dao.BcmSyncLogMapper;
import com.tydic.bcm.personal.po.BcmFmisPaymentRecordPO;
import com.tydic.bcm.personal.po.BcmQueryFmisPaymentRecordPO;
import com.tydic.bcm.personal.po.BcmSyncLogPO;
import com.tydic.bcm.personal.task.api.BcmFmisPaymentRecordSyncTaskService;
import com.tydic.bcm.personal.task.bo.BcmFmisPaymentRecordSyncTaskReqBO;
import com.tydic.bcm.personal.task.bo.BcmFmisPaymentRecordSyncTaskRspBO;
import com.tydic.bcm.personal.task.bo.BcmFscPayInfoBackAbilityReqBO;
import com.tydic.bcm.personal.utils.BcmIdUtil;
import com.tydic.bcm.personal.utils.BcmRuUtil;
import com.tydic.dyc.base.bo.BaseRspBo;
import lombok.extern.ohaotian.HTServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 标题 BcmFmisPaymentRecordSyncTaskServiceImpl
 * 说明 付款记录同步对比服务实现
 * 时间 2023/8/18 10:14
 *
 * @author 唐富强
 */
@Slf4j
@HTServiceImpl
public class BcmFmisPaymentRecordSyncTaskServiceImpl implements BcmFmisPaymentRecordSyncTaskService {

    /**
     * 付款记录临时表Mapper
     */
    @Autowired
    private BcmFmisPaymentRecordMapper bcmFmisPaymentRecordMapper;

    /**
     * 同步记录Mapper
     */
    @Autowired
    private BcmSyncLogMapper bcmSyncLogMapper;

    /**
     * 每次处理条数
     */
    @Value("${dealSize:100}")
    private Integer dealSize;

    /**
     * 推送数据能力平台地址
     */
    @Value("${SYNC_PAYMENT_RECORD_URL:url}")
    private String syncPaymentRecordUrl;

    /**
     * 付款记录同步对比服务实现
     *
     * @param reqBO
     * @return
     */
    @Override
    @Async("global-thread-pool")
    public BcmFmisPaymentRecordSyncTaskRspBO syncPaymentRecord(BcmFmisPaymentRecordSyncTaskReqBO reqBO) {
        //数据同步对比
        sync();
        return BcmRuUtil.success(BcmFmisPaymentRecordSyncTaskRspBO.class);
    }

    /**
     * 数据同步对比
     */
    private void sync() {
        Page<BcmFmisPaymentRecordPO> syncPage = getFmisPaymentRecordList();
        List<BcmFmisPaymentRecordPO> syncList = syncPage.getResult();
        List<Long> succssIdList = new ArrayList<Long>(syncList.size());
        List<Long> failIdList = new ArrayList<Long>(syncList.size());
        if (CollectionUtils.isEmpty(syncList)) {
            return;
        }
        BcmSyncLogPO syncLogPO = getSyncLog();
        if (syncLogPO == null) {

            for (BcmFmisPaymentRecordPO po : syncList) {
                boolean check = pushAbility(po);
                if (check) {
                    succssIdList.add(po.getId());
                } else {
                    failIdList.add(po.getId());
                }
            }
            //新增同步记录
            addSyncLog();
        } else {
            //无需同步id集合
            List<Long> skipIdList = new ArrayList<Long>(syncList.size());
            //需要推送到能力平台的数据
            List<BcmFmisPaymentRecordPO> addList = new ArrayList<>(syncList.size());
            addList.addAll(syncList);
            //推送能力平台
            if (!CollectionUtils.isEmpty(addList)) {
                for (BcmFmisPaymentRecordPO po : addList) {
                    boolean check = pushAbility(po);
                    if (check) {
                        succssIdList.add(po.getId());
                    } else {
                        failIdList.add(po.getId());
                    }
                }

            }
            //更新付款记录临时表处理结果
            if (!CollectionUtils.isEmpty(syncList)) {
                updateFmisPaymentRecord(skipIdList, BcmDealResultEnum.SKIP.getDealResult());
            }
            //更新同步记录
            updateSyncLog(syncLogPO.getId());
        }
        //更新付款记录临时表处理结果
        if (!CollectionUtils.isEmpty(succssIdList)) {
            updateFmisPaymentRecord(succssIdList, BcmDealResultEnum.SUCCESS.getDealResult());
        }
        if (!CollectionUtils.isEmpty(failIdList)) {
            updateFmisPaymentRecord(failIdList, BcmDealResultEnum.FAIL.getDealResult());
        }
        if (syncPage.getTotalPages() > 1) {
            sync();
        }
    }

    /**
     * 对比PO获取
     *
     * @param po
     * @return
     */
    private void setComparePO(BcmFmisPaymentRecordPO po) {
        po.setId(null);
        po.setDealTime(null);
        po.setDealResult(null);
    }

    /**
     * 对比数据获取
     *
     * @param storeIdList
     * @param pushTime
     * @return
     */
    private Map<String, BcmFmisPaymentRecordPO> getThanMap(List<String> storeIdList, Date pushTime) {
        BcmQueryFmisPaymentRecordPO bcmQueryFmisPaymentRecordPO = new BcmQueryFmisPaymentRecordPO();
        bcmQueryFmisPaymentRecordPO.setPushTime(pushTime);
        bcmQueryFmisPaymentRecordPO.setStoreIdList(storeIdList);
        bcmQueryFmisPaymentRecordPO.setDealResult(BcmDealResultEnum.SUCCESS.getDealResult());
        // 获取前一天的日期
        LocalDate yesterday = LocalDate.now().minusDays(1);
        // 定义日期格式
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        // 格式化日期为字符串类型
        String yesterdayString = yesterday.format(formatter);
        bcmQueryFmisPaymentRecordPO.setPaymentTime(yesterdayString);
        List<BcmFmisPaymentRecordPO> list = bcmFmisPaymentRecordMapper.getList(bcmQueryFmisPaymentRecordPO);
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        return list.stream().collect(Collectors.toMap(BcmFmisPaymentRecordPO::getStoreId, o -> o, (k1, k2) -> k2));
    }

    /**
     * 数据推送到能力平台
     *
     * @param po
     */
    private Boolean pushAbility(BcmFmisPaymentRecordPO po) {
        log.info("付款记录po:{}", JSONObject.toJSONString(po));
        BcmFscPayInfoBackAbilityReqBO bcmFscPayInfoReqBO = new BcmFscPayInfoBackAbilityReqBO();
        bcmFscPayInfoReqBO.setBusiAmount(new BigDecimal(po.getPaymentAmount()));
//        bcmFscPayInfoReqBO.setBusiOrderNo(po.getStoreCode());
        bcmFscPayInfoReqBO.setBusiOrderId(Long.valueOf(po.getStoreId()));
        bcmFscPayInfoReqBO.setPayerName(po.getPayer());
        bcmFscPayInfoReqBO.setShouldPayId(Long.valueOf(po.getOrderItemId()));
        if (!BcmPersonalSettleConstant.PayResult.SUCCESS.equals(po.getPaymentResult())) {
            bcmFscPayInfoReqBO.setPayResult(BcmPersonalSettleConstant.PayResult.FAIL);
        } else {
            bcmFscPayInfoReqBO.setPayResult(po.getPaymentResult());
        }
        bcmFscPayInfoReqBO.setBusiTime(DateUtil.parse(po.getPaymentTime()));
        //id 作用仅是跳过防重校验(前一天可能支付多笔相同金额的支付记录，只校验是前一天的数据就行)
        bcmFscPayInfoReqBO.setId(po.getId());
        log.info("调用能力平台url:{}入参:{}", syncPaymentRecordUrl, JSONObject.toJSONString(bcmFscPayInfoReqBO));
        String rspStr = HttpUtil.post(syncPaymentRecordUrl, JSONObject.toJSONString(bcmFscPayInfoReqBO));
        log.info("调用能力平台出参:{}", rspStr);
        if (StrUtil.isBlank(rspStr)) {
            return false;
        }
        BaseRspBo baseRspBo = BcmRuUtil.jss(rspStr, BaseRspBo.class);
        return BcmPersonalCommonConstant.Code.SUCCESS.equals(baseRspBo.getCode());
    }

    /**
     * 同步记录
     *
     * @return
     */
    private BcmSyncLogPO getSyncLog() {
        BcmSyncLogPO bcmSyncLogPO = new BcmSyncLogPO();
        bcmSyncLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.PAYMENT_RECORD);
        return bcmSyncLogMapper.getModelBy(bcmSyncLogPO);
    }

    /**
     * 待同步付款记录获取
     *
     * @return
     */
    private Page<BcmFmisPaymentRecordPO> getFmisPaymentRecordList() {
        BcmFmisPaymentRecordPO bcmFmisPaymentRecordPO = new BcmFmisPaymentRecordPO();
        bcmFmisPaymentRecordPO.setDealResult(BcmDealResultEnum.PENDING.getDealResult());
        // 获取前一天的日期
        LocalDate yesterday = LocalDate.now().minusDays(1);
        // 定义日期格式
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        // 格式化日期为字符串类型
        String yesterdayString = yesterday.format(formatter);
        bcmFmisPaymentRecordPO.setPaymentTime(yesterdayString);
        Page<BcmFmisPaymentRecordPO> page = new Page<>(1, dealSize);
        List<BcmFmisPaymentRecordPO> pageList = bcmFmisPaymentRecordMapper.getPageList(bcmFmisPaymentRecordPO, page);
        page.setResult(pageList);
        return page;
    }

    /**
     * 更新付款记录临时表
     *
     * @param idList
     * @param dealResult
     */
    private void updateFmisPaymentRecord(List<Long> idList, String dealResult) {
        BcmQueryFmisPaymentRecordPO bcmQueryFmisPaymentRecordPO = new BcmQueryFmisPaymentRecordPO();
        bcmQueryFmisPaymentRecordPO.setIdList(idList);
        bcmQueryFmisPaymentRecordPO.setDealResult(dealResult);
        bcmQueryFmisPaymentRecordPO.setDealTime(new Date());
        bcmFmisPaymentRecordMapper.updateBatch(bcmQueryFmisPaymentRecordPO);
    }

    /**
     * 新增同步记录
     */
    private void addSyncLog() {
        BcmSyncLogPO bcmSyncLogPO = new BcmSyncLogPO();
        bcmSyncLogPO.setId(BcmIdUtil.nextId());
        bcmSyncLogPO.setLastDate(new Date());
        bcmSyncLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.PAYMENT_RECORD);
        bcmSyncLogMapper.insert(bcmSyncLogPO);
    }

    /**
     * 更新同步记录
     *
     * @param id
     */
    private void updateSyncLog(Long id) {
        BcmSyncLogPO bcmSyncLogPO = new BcmSyncLogPO();
        bcmSyncLogPO.setId(id);
        bcmSyncLogPO.setLastDate(new Date());
        bcmSyncLogMapper.updateById(bcmSyncLogPO);
    }

    public static void main(String[] args) {
        // 获取前一天的日期
        LocalDate yesterday = LocalDate.now().minusDays(1);
        // 定义日期格式
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        // 格式化日期为字符串类型
        String yesterdayString = yesterday.format(formatter);
        System.out.println(yesterdayString);

    }
}
