package com.tydic.bcm.personal.task.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSONObject;
import com.ohaotian.plugin.db.Page;
import com.tydic.bcm.personal.constants.BcmPersonalCommonConstant;
import com.tydic.bcm.personal.dao.BcmGuwpOrgInfoMapper;
import com.tydic.bcm.personal.dao.BcmSyncLogMapper;
import com.tydic.bcm.personal.po.BcmGuwpOrgInfoPO;
import com.tydic.bcm.personal.po.BcmGuwpOrgInfoQueryPO;
import com.tydic.bcm.personal.po.BcmGuwpOrgInfoUpdatePO;
import com.tydic.bcm.personal.po.BcmSyncLogPO;
import com.tydic.bcm.personal.task.api.BcmOrgSyncTaskService;
import com.tydic.bcm.personal.task.bo.BcmGuwpSyncOrgReqBO;
import com.tydic.bcm.personal.task.bo.BcmGuwpSyncOrgRspBO;
import com.tydic.bcm.personal.utils.BcmIdUtil;
import com.tydic.bcm.personal.utils.BcmRuUtil;
import com.tydic.dyc.base.bo.BaseRspBo;
import com.tydic.dyc.base.constants.BaseRspConstant;
import lombok.extern.ohaotian.HTServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 标题 BcmOrgSyncTaskServiceImpl
 * 说明 机构同步比对任务实现
 * 时间 2023/8/16 14:07
 *
 * @author 唐富强
 */
@HTServiceImpl
@Slf4j
public class BcmOrgSyncTaskServiceImpl implements BcmOrgSyncTaskService {

    /**
     * 同步日志表--Mapper
     */
    @Autowired
    private BcmSyncLogMapper bcmSyncLogMapper;

    /**
     * 机构信息表Mapper
     */
    @Autowired
    private BcmGuwpOrgInfoMapper bcmGuwpOrgInfoMapper;

    /**
     * 每次处理条数
     */
    @Value("${dealSize:100}")
    private Integer dealSize;

    /**
     * 调用能力平台地址
     */
    @Value("${SYNC_ORG_URL:url}")
    private String syncOrgUrl;

    private static final Integer delPage = 1;

    @Override
    @Async("global-thread-pool")
    public BcmGuwpSyncOrgRspBO syncOrg(BcmGuwpSyncOrgReqBO reqBO) {
        BcmGuwpSyncOrgRspBO rspBO = new BcmGuwpSyncOrgRspBO();
        // 查询有没有待处理的数据
        BcmGuwpOrgInfoQueryPO qryPO = new BcmGuwpOrgInfoQueryPO();
        qryPO.setDealResult(BcmPersonalCommonConstant.DealResult.PENDING);
        Page<BcmGuwpOrgInfoPO> page = new Page<>(delPage, dealSize);
        List<BcmGuwpOrgInfoPO> pageList = bcmGuwpOrgInfoMapper.getPageList(qryPO, page);
        if (ObjectUtil.isEmpty(pageList)) {
            rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
            rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
            return rspBO;
        }
        Date lastDate = pageList.get(0).getPushTime();

        // 根据字符串“GUWP_ORG”查询SYNC_LOG表中的记录，获取上次执行时间LAST_DATE，如果没有就为空
        BcmSyncLogPO syncLogPO = new BcmSyncLogPO();
        syncLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.GUWP_ORG);
        BcmSyncLogPO bcmSyncLogPO = bcmSyncLogMapper.getModelBy(syncLogPO);
        int syncNum = 0;
        if (ObjectUtil.isNotEmpty(bcmSyncLogPO)) {

            // 循环、分页查询ORG_INFO_GUWP表中的数据：DEAL_RESULT字段为待同步状态的数据，数据处理完或无数据终止循环
            qryPO.setPushStartTime(bcmSyncLogPO.getLastDate());
            while (true) {
                Page<BcmGuwpOrgInfoPO> pageNow = new Page<>(delPage, dealSize);
                pageList = bcmGuwpOrgInfoMapper.getPageList(qryPO, pageNow);
                if (ObjectUtil.isEmpty(pageList)) {
                    // 数据处理完了，修改处理记录
                    BcmSyncLogPO updateLogPO = new BcmSyncLogPO();
                    updateLogPO.setId(bcmSyncLogPO.getId());
                    updateLogPO.setLastDate(lastDate);
                    bcmSyncLogMapper.updateById(updateLogPO);

                    rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
                    rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
                    rspBO.setSyncNum(syncNum);
                    return rspBO;
                }
                // 成功的就是要同步给能力平台的
                List<BcmGuwpOrgInfoPO> successPOList = new ArrayList<>();
                // 未改变的ID集合
                List<Long> skipIdList = new ArrayList<>();
                // 遍历对比
                List<Long> orgIdList = pageList.stream().map(BcmGuwpOrgInfoPO::getOrgId).collect(Collectors.toList());
                // 已经同步过数据，查询以前存在的数据
                Map<Long, BcmGuwpOrgInfoPO> oldOrgMap = getOldOrgMap(bcmSyncLogPO.getLastDate(), orgIdList);
                for (BcmGuwpOrgInfoPO newOrgPO : pageList) {

                    // 得到存在的OldOrgPO
                    BcmGuwpOrgInfoPO oldOrgPO = oldOrgMap.get(newOrgPO.getOrgId());
                    // hash码相同，对象的值也是相同的，所以没变化
                    if (newOrgPO.equals(oldOrgPO)) {
                        skipIdList.add(newOrgPO.getId());
                    } else { // 设置未成功
                        successPOList.add(newOrgPO);
                    }
                }

                //判断无需同步的ID集合
                if (ObjectUtil.isNotEmpty(skipIdList)) {
                    changeDealResult(skipIdList, BcmPersonalCommonConstant.DealResult.SKIP);
                }

                // 如果成功的数据不未空，同步给能力平台
                if (ObjectUtil.isNotEmpty(successPOList)) {
                    doAbilityMethod(successPOList, syncOrgUrl);
                    syncNum = syncNum + successPOList.size();
                }
            }
        } else {
            while (true) {
                Page<BcmGuwpOrgInfoPO> pageNow = new Page<>(delPage, dealSize);
                pageList = bcmGuwpOrgInfoMapper.getPageList(qryPO, pageNow);
                if (ObjectUtil.isEmpty(pageList)) {
                    // 在sync_log里面新增一条记录
                    BcmSyncLogPO insertLogPO = new BcmSyncLogPO();
                    insertLogPO.setId(BcmIdUtil.nextId());
                    insertLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.GUWP_ORG);
                    insertLogPO.setLastDate(lastDate);
                    bcmSyncLogMapper.insert(insertLogPO);

                    rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
                    rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
                    rspBO.setSyncNum(syncNum);
                    return rspBO;
                }
                // 调用能力平台
                doAbilityMethod(pageList, syncOrgUrl);
                syncNum = syncNum + pageList.size();
            }
        }
    }

    /**
     * 已经同步过数据，查询以前存在的数据
     *
     * @param
     * @return
     */
    private Map<Long, BcmGuwpOrgInfoPO> getOldOrgMap(Date lastDate, List<Long> orgIdList) {
        if (CollectionUtil.isEmpty(orgIdList)) {
            return Collections.emptyMap();
        }
        BcmGuwpOrgInfoQueryPO qryOldPO = new BcmGuwpOrgInfoQueryPO();
        qryOldPO.setDealResult(BcmPersonalCommonConstant.DealResult.SUCCESS);
        qryOldPO.setPushTime(lastDate);
        qryOldPO.setOrgIdList(orgIdList);
        List<BcmGuwpOrgInfoPO> oldOrgPOList = bcmGuwpOrgInfoMapper.getList(qryOldPO);
        if (CollectionUtil.isEmpty(oldOrgPOList)) {
            return Collections.emptyMap();
        }
        return oldOrgPOList.stream().collect(Collectors.
                toMap(BcmGuwpOrgInfoPO::getOrgId, Function.identity(), (e1, e2) -> e2));
    }

    /**
     * 调用能力平台
     *
     * @param successPOList
     * @param url
     */
    private void doAbilityMethod(List<BcmGuwpOrgInfoPO> successPOList, String url) {
        List<Long> idList = successPOList.stream().map(BcmGuwpOrgInfoPO::getId).collect(Collectors.toList());
        String reqStr = JSONObject.toJSONString(successPOList);
        log.info("同步机构调用能力平台入参为：{}", reqStr);
        String rspStr = HttpUtil.post(url, reqStr);
        log.info("同步机构调用能力平台出参为：{}", rspStr);

        // 根据能力平台返回判断
        BaseRspBo baseRspBo = BcmRuUtil.jss(rspStr, BaseRspBo.class);
        if (baseRspBo == null) {
            changeDealResult(idList, BcmPersonalCommonConstant.DealResult.FAIL);
        } else if ("0".equals(baseRspBo.getCode())) {
            changeDealResult(idList, BcmPersonalCommonConstant.DealResult.SUCCESS);
        } else {
            changeDealResult(idList, BcmPersonalCommonConstant.DealResult.FAIL);
        }
    }

    /**
     * 更新状态
     *
     * @param ids
     */
    private void changeDealResult(List<Long> ids, String dealResult) {
        BcmGuwpOrgInfoUpdatePO setPO = new BcmGuwpOrgInfoUpdatePO();
        setPO.setDealResult(dealResult);
        setPO.setDealTime(new Date());
        BcmGuwpOrgInfoUpdatePO wherePO = new BcmGuwpOrgInfoUpdatePO();
        wherePO.setIdList(ids);
        bcmGuwpOrgInfoMapper.updateBy(setPO, wherePO);
    }
}
