package com.tydic.bcm.personal.task.impl;

import com.ohaotian.plugin.db.Page;
import com.tydic.bcm.personal.constants.BcmDealResultEnum;
import com.tydic.bcm.personal.constants.BcmPersonalCommonConstant;
import com.tydic.bcm.personal.dao.BcmBillTellerMapper;
import com.tydic.bcm.personal.dao.BcmSmUserMapper;
import com.tydic.bcm.personal.po.BcmBillTellerPO;
import com.tydic.bcm.personal.po.BcmQueryBillTellerPO;
import com.tydic.bcm.personal.po.BcmQuerySmUserPO;
import com.tydic.bcm.personal.po.BcmSmUserPO;
import com.tydic.bcm.personal.task.api.BcmSmUserSyncTaskService;
import com.tydic.bcm.personal.task.bo.BcmSmUserSyncTaskReqBO;
import com.tydic.bcm.personal.task.bo.BcmSmUserSyncTaskRspBO;
import com.tydic.bcm.personal.utils.BcmRuUtil;
import lombok.extern.ohaotian.HTServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 标题 BcmSmUserSyncTaskServiceImpl
 * 说明 报账员同步对比服务实现
 * 时间 2023/8/18 14:27
 *
 * @author 唐富强
 */
@HTServiceImpl
public class BcmSmUserSyncTaskServiceImpl implements BcmSmUserSyncTaskService {

    /**
     * 报账员临时表Mapper
     */
    @Autowired
    private BcmSmUserMapper bcmSmUserMapper;

    /**
     * 报账员表Mapper
     */
    @Autowired
    private BcmBillTellerMapper bcmBillTellerMapper;

    /**
     * 每次处理条数
     */
    @Value("${dealSize:100}")
    private Integer dealSize;

    /**
     * 报账员同步对比服务实现
     *
     * @param reqBO
     * @return
     */
    @Override
    @Async("global-thread-pool")
    public BcmSmUserSyncTaskRspBO syncSmUser(BcmSmUserSyncTaskReqBO reqBO) {
        //数据同步对比
        sync();
        return BcmRuUtil.success(BcmSmUserSyncTaskRspBO.class);
    }

    /**
     * 数据同步对比
     */
    private void sync() {
        Page<BcmSmUserPO> syncPage = getSmUser();
        List<BcmSmUserPO> syncList = syncPage.getResult();
        if (CollectionUtils.isEmpty(syncList)) {
            return;
        }
        //新增成功id集合
        List<Long> successIdList = new ArrayList<>(syncList.size());
        Map<String, BcmBillTellerPO> thanMap = getBillTellerMap(syncList);
        if (CollectionUtils.isEmpty(thanMap)) {
            addBillTeller(syncList);
            successIdList.addAll(syncList.stream().map(BcmSmUserPO::getId).collect(Collectors.toList()));
        } else {
            //无需同步id集合
            List<Long> skipIdList = new ArrayList<>(syncList.size());
            //需要新增数据
            List<BcmSmUserPO> addList = new ArrayList<>(syncList.size());
            //需要更新数据
            List<BcmBillTellerPO> updateList = new ArrayList<>(syncList.size());
            for (BcmSmUserPO smUserPO : syncList) {
                BcmBillTellerPO than = thanMap.get(smUserPO.getCuserId());
                if (than == null) {
                    addList.add(smUserPO);
                    successIdList.add(smUserPO.getId());
                } else {
                    BcmBillTellerPO newThan = setBillTeller(smUserPO);
                    Long id = than.getId();
                    //排除不需要比对的值
                    than.setId(null);
                    than.setCreateTime(null);
                    than.setUpdateTime(null);
                    than.setBillTellerId(null);
                    if (than.hashCode() != newThan.hashCode()) {
                        newThan.setId(id);
                        newThan.setUpdateTime(smUserPO.getPushTime());
                        updateList.add(newThan);
                        successIdList.add(smUserPO.getId());
                    } else {
                        skipIdList.add(smUserPO.getId());
                    }
                }
            }
            //报账员新增
            if (!CollectionUtils.isEmpty(addList)) {
                addBillTeller(addList);
            }
            //报账员新增
            if (!CollectionUtils.isEmpty(updateList)) {
                bcmBillTellerMapper.updateBatch(updateList);
            }
            //更新同步结果
            if (!CollectionUtils.isEmpty(skipIdList)) {
                updateSmUser(skipIdList, BcmDealResultEnum.SKIP.getDealResult());
            }
        }
        //更新同步结果
        if (!CollectionUtils.isEmpty(successIdList)) {
            updateSmUser(successIdList, BcmDealResultEnum.SUCCESS.getDealResult());
        }
        if (syncPage.getTotalPages() > 1) {
            sync();
        }
    }

    /**
     * 新增报账员数据
     *
     * @param syncList
     */
    private void addBillTeller(List<BcmSmUserPO> syncList) {
        List<BcmBillTellerPO> list = new ArrayList<>(syncList.size());
        for (BcmSmUserPO smUserPO : syncList) {
            BcmBillTellerPO po = setBillTeller(smUserPO);
            po.setBillTellerId(smUserPO.getCuserId());
            po.setId(smUserPO.getId());
            po.setCreateTime(smUserPO.getPushTime());
            po.setUpdateTime(smUserPO.getPushTime());
            list.add(po);
        }
        bcmBillTellerMapper.insertBatch(list);
    }

    /**
     * 待同步数据获取
     *
     * @return
     */
    private Page<BcmSmUserPO> getSmUser() {
        BcmSmUserPO bcmSmUserPO = new BcmSmUserPO();
        bcmSmUserPO.setDealResult(BcmDealResultEnum.PENDING.getDealResult());
        Page<BcmSmUserPO> page = new Page<>(1, dealSize);
        List<BcmSmUserPO> pageList = bcmSmUserMapper.getPageList(bcmSmUserPO, page);
        page.setResult(pageList);
        return page;
    }

    /**
     * 报账员数据设置
     *
     * @param smUserPO
     * @return
     */
    private BcmBillTellerPO setBillTeller(BcmSmUserPO smUserPO) {
        BcmBillTellerPO po = new BcmBillTellerPO();
        po.setBillTellerCode(smUserPO.getUserCode());
        po.setBillTellerName(smUserPO.getUserName());
        po.setBillTellerOrgId(smUserPO.getOaOrg());
        po.setBillTellerOrgCode(smUserPO.getOaOrg());
        po.setBillTellerOrgName(smUserPO.getOrgName());
        po.setBillOrgId(smUserPO.getOaDept());
        po.setBillOrgCode(smUserPO.getOaDept());
        po.setBillOrgName(smUserPO.getDeptName());
        po.setDelFlag(BcmPersonalCommonConstant.DelFlag.NO);
        return po;
    }

    /**
     * 比对数据获取
     *
     * @param syncList
     * @return
     */
    private Map<String, BcmBillTellerPO> getBillTellerMap(List<BcmSmUserPO> syncList) {
        BcmQueryBillTellerPO bcmQueryBillTellerPO = new BcmQueryBillTellerPO();
        bcmQueryBillTellerPO.setDelFlag(BcmPersonalCommonConstant.DelFlag.NO);
        bcmQueryBillTellerPO.setBillTellerIdList(syncList.stream().map(BcmSmUserPO::getCuserId).collect(Collectors.toList()));
        List<BcmBillTellerPO> list = bcmBillTellerMapper.getList(bcmQueryBillTellerPO);
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        return list.stream().collect(Collectors.toMap(BcmBillTellerPO::getBillTellerId, o -> o, (k1, k2) -> k2));
    }

    /**
     * 修改临时表同步结果
     *
     * @param idList
     * @param dealResult
     */
    private void updateSmUser(List<Long> idList, String dealResult) {
        BcmQuerySmUserPO bcmQuerySmUserPO = new BcmQuerySmUserPO();
        bcmQuerySmUserPO.setIdList(idList);
        bcmQuerySmUserPO.setDealResult(dealResult);
        bcmQuerySmUserPO.setDealTime(new Date());
        bcmSmUserMapper.updateBatch(bcmQuerySmUserPO);
    }
}
