package com.tydic.bcm.personal.task.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSONObject;
import com.ohaotian.plugin.db.Page;
import com.tydic.bcm.personal.constants.BcmPersonalCommonConstant;
import com.tydic.bcm.personal.dao.BcmSyncLogMapper;
import com.tydic.bcm.personal.dao.BcmUserInfoIpspMapper;
import com.tydic.bcm.personal.po.BcmSyncLogPO;
import com.tydic.bcm.personal.po.BcmUserInfoIpspPO;
import com.tydic.bcm.personal.po.BcmUserInfoIpspQueryPO;
import com.tydic.bcm.personal.po.BcmUserInfoIpspUpdatePO;
import com.tydic.bcm.personal.task.api.BcmUserSyncTaskService;
import com.tydic.bcm.personal.task.bo.BcmGuwpSyncUserReqBO;
import com.tydic.bcm.personal.task.bo.BcmGuwpSyncUserRspBO;
import com.tydic.bcm.personal.utils.BcmIdUtil;
import com.tydic.bcm.personal.utils.BcmRuUtil;
import com.tydic.dyc.base.bo.BaseRspBo;
import com.tydic.dyc.base.constants.BaseRspConstant;
import lombok.extern.ohaotian.HTServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 类名： BcmSaasUserSyncTaskServiceImpl
 * 说明：用户同步比对任务实现
 *
 * @author： lishiqing
 * 时间： 2023/8/16 11:34
 */
@HTServiceImpl
@Slf4j
public class BcmUserSyncTaskServiceImpl implements BcmUserSyncTaskService {

    /**
     * 同步日志表--Mapper
     */
    @Autowired
    private BcmSyncLogMapper bcmSyncLogMapper;

    /**
     * 用户信息临时表Mapper
     */
    @Autowired
    private BcmUserInfoIpspMapper bcmUserInfoIpspMapper;

    /**
     * 每次处理条数
     */
    @Value("${DEAL_USER_COUNT:100}")
    private Integer dealCount;

    /**
     * 调用能力平台地址
     */
    @Value("${SYNC_USER_ALL_URL:}")
    private String syncUserAllUrl;

    private static final Integer delPage = 1;

    @Override
    @Async("global-thread-pool")
    public BcmGuwpSyncUserRspBO syncUser(BcmGuwpSyncUserReqBO reqBO) {
        BcmGuwpSyncUserRspBO rspBO = new BcmGuwpSyncUserRspBO();

        // 查询有没有待处理的数据
        BcmUserInfoIpspQueryPO qryPO = new BcmUserInfoIpspQueryPO();
        qryPO.setDealResult(BcmPersonalCommonConstant.DealResult.PENDING);
        Page<BcmUserInfoIpspPO> page = new Page<>(delPage, dealCount);
        List<BcmUserInfoIpspPO> pageList = bcmUserInfoIpspMapper.getPageList(qryPO, page);
        if (ObjectUtil.isEmpty(pageList)) {
            rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
            rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
            return rspBO;
        }
        Date lastDate = pageList.get(0).getPushTime();

        // 根据字符串“GUWP_USER”查询SYNC_LOG表中的记录，获取上次执行时间LAST_DATE，如果没有就为空
        BcmSyncLogPO syncLogPO = new BcmSyncLogPO();
        syncLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.GUWP_USER);
        BcmSyncLogPO bcmSyncLogPO = bcmSyncLogMapper.getModelBy(syncLogPO);
        int syncNum = 0;
        if (ObjectUtil.isNotEmpty(bcmSyncLogPO)) {

            // 循环、分页查询USER_INFO_GUWP表中的数据：DEAL_RESULT字段为待同步状态的数据，数据处理完或无数据终止循环
            qryPO.setPushStartTime(bcmSyncLogPO.getLastDate());
            while (true) {
                Page<BcmUserInfoIpspPO> pageNow = new Page<>(delPage, dealCount);
                pageList = bcmUserInfoIpspMapper.getPageList(qryPO, pageNow);
                if (ObjectUtil.isEmpty(pageList)) {
                    // 数据处理完了，修改处理记录
                    BcmSyncLogPO updateLogPO = new BcmSyncLogPO();
                    updateLogPO.setId(bcmSyncLogPO.getId());
                    updateLogPO.setLastDate(lastDate);
                    bcmSyncLogMapper.updateById(updateLogPO);

                    rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
                    rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
                    rspBO.setSyncNum(syncNum);
                    return rspBO;
                }
                // 成功的就是要同步给能力平台的
                List<BcmUserInfoIpspPO> successPOList = new ArrayList<>();
                // 未改变的ID集合
                List<Long> skipIdList = new ArrayList<>();

                List<Long> userIdList = pageList.stream().map(BcmUserInfoIpspPO::getUserId).collect(Collectors.toList());
                // 已经同步过数据，查询以前存在的数据
                Map<Long, BcmUserInfoIpspPO> oldUserMap = getOldUserMap(bcmSyncLogPO.getLastDate(), userIdList);
                // 遍历对比
                for (BcmUserInfoIpspPO newUserPO : pageList) {
                    // 得到存在的OldUserPO
                    BcmUserInfoIpspPO oldUserPO = oldUserMap.get(newUserPO.getUserId());
                    // hash码相同，对象的值也是相同的，所以没变化
                    if (newUserPO.equals(oldUserPO)) {
                        skipIdList.add(newUserPO.getId());
                    } else {
                        // 存在值的修改
                        successPOList.add(newUserPO);
                    }
                }
                //判断无需同步的ID集合
                if (ObjectUtil.isNotEmpty(skipIdList)) {
                    changeDealResult(skipIdList, BcmPersonalCommonConstant.DealResult.SKIP);
                }
                // 如果成功的数据不未空，同步给能力平台
                if (ObjectUtil.isNotEmpty(successPOList)) {
                    doAbilityMethod(successPOList, syncUserAllUrl);
                    syncNum = syncNum + successPOList.size();
                }
            }
        } else {
            // 循环、分页查询USER_INFO_GUWP表中的数据：DEAL_RESULT字段为待同步状态的数据，数据处理完或无数据终止循环
            while (true) {
                Page<BcmUserInfoIpspPO> pageNow = new Page<>(delPage, dealCount);
                pageList = bcmUserInfoIpspMapper.getPageList(qryPO, pageNow);
                if (ObjectUtil.isEmpty(pageList)) {
                    // 在sync_log里面新增一条记录
                    BcmSyncLogPO insertLogPO = new BcmSyncLogPO();
                    insertLogPO.setId(BcmIdUtil.nextId());
                    insertLogPO.setDataType(BcmPersonalCommonConstant.SyncLogDataType.GUWP_USER);
                    insertLogPO.setLastDate(lastDate);
                    bcmSyncLogMapper.insert(insertLogPO);

                    rspBO.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
                    rspBO.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
                    rspBO.setSyncNum(syncNum);
                    return rspBO;
                }
                // 调用能力平台
                doAbilityMethod(pageList, syncUserAllUrl);
                syncNum = syncNum + pageList.size();
            }
        }
    }

    /**
     * 已经同步过数据，查询以前存在的数据
     *
     * @param lastDate
     * @param userIdList
     * @return
     */
    private Map<Long, BcmUserInfoIpspPO> getOldUserMap(Date lastDate, List<Long> userIdList) {
        if (CollectionUtil.isEmpty(userIdList)) {
            return Collections.emptyMap();
        }
        BcmUserInfoIpspQueryPO qryOldPO = new BcmUserInfoIpspQueryPO();
        qryOldPO.setDealResult(BcmPersonalCommonConstant.DealResult.SUCCESS);
        qryOldPO.setPushTime(lastDate);
        qryOldPO.setUserIdList(userIdList);
        List<BcmUserInfoIpspPO> oldUserPOList = bcmUserInfoIpspMapper.getList(qryOldPO);
        if (CollectionUtil.isEmpty(oldUserPOList)) {
            return Collections.emptyMap();
        }
        return oldUserPOList.stream().collect(Collectors.
                toMap(BcmUserInfoIpspPO::getUserId, Function.identity(), (e1, e2) -> e2));
    }

    /**
     * 调用能力平台
     *
     * @param successPOList
     * @param url
     */
    private void doAbilityMethod(List<BcmUserInfoIpspPO> successPOList, String url) {
        List<Long> idList = successPOList.stream().map(BcmUserInfoIpspPO::getId).collect(Collectors.toList());
        String reqStr = JSONObject.toJSONString(successPOList);
        log.info("同步用户调用能力平台入参为：{}", reqStr);
        String rspStr = HttpUtil.post(url, reqStr);
        log.info("同步用户调用能力平台出参为：{}", rspStr);
        // 根据能力平台返回判断
        BaseRspBo baseRspBo = BcmRuUtil.jss(rspStr, BaseRspBo.class);
        if (baseRspBo == null) {
            changeDealResult(idList, BcmPersonalCommonConstant.DealResult.FAIL);
        } else if ("0".equals(baseRspBo.getCode())) {
            changeDealResult(idList, BcmPersonalCommonConstant.DealResult.SUCCESS);
        } else {
            changeDealResult(idList, BcmPersonalCommonConstant.DealResult.FAIL);
        }
    }

    /**
     * 更新状态
     *
     * @param ids
     */
    private void changeDealResult(List<Long> ids, String dealResult) {
        BcmUserInfoIpspUpdatePO setPO = new BcmUserInfoIpspUpdatePO();
        setPO.setDealResult(dealResult);
        setPO.setDealTime(new Date());
        BcmUserInfoIpspUpdatePO wherePO = new BcmUserInfoIpspUpdatePO();
        wherePO.setIdList(ids);
        bcmUserInfoIpspMapper.updateBy(setPO, wherePO);
    }
}
