package com.tydic.dict.system.repository.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ohaotian.authority.common.rsp.DictPage;
import com.ohaotian.authority.common.rsp.DictResult;
import com.tydic.dict.system.repository.dao.DictAnnouncementMapper;
import com.tydic.dict.system.repository.po.DictAnnouncementPo;
import com.tydic.dict.system.repository.service.search.DictAnnouncementSearchService;
import com.tydic.dict.system.service.bo.DictAnnouncementPageReqBo;
import com.tydic.dict.system.service.bo.DictAnnouncementRspBo;
import com.tydic.dict.system.service.constants.HomeConstant;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class DictAnnouncementSearchServiceImpl extends ServiceImpl<DictAnnouncementMapper, DictAnnouncementPo> implements DictAnnouncementSearchService {

    @Autowired
    private DictAnnouncementMapper dictAnnouncementMapper;

    @Autowired
    private RedisTemplate redisTemplate;

    // 公告缓存key
    private static final String ANNOUNCEMENT_CACHE_KEY = "announcement_list";

    @Override
    public DictResult<DictAnnouncementRspBo> detail(Long id) {
        if (id == null) {
            return DictResult.error(null, "参数为空");
        }
        DictAnnouncementPo po = dictAnnouncementMapper.selectById(id);
        if (po == null) {
            return DictResult.error(null, "查询失败");
        }
        return DictResult.success(BeanUtil.copyProperties(po, DictAnnouncementRspBo.class));
    }

    @Override
    public DictResult<DictPage<DictAnnouncementRspBo>> queryByCondition(DictAnnouncementPageReqBo req) {
        Page<DictAnnouncementPo> page = dictAnnouncementMapper.selectPage(
                new Page<>(req.getPageNo(), req.getPageSize()),
                Wrappers.<DictAnnouncementPo>lambdaQuery()
                        .like(StringUtils.isNotBlank(req.getTitle()), DictAnnouncementPo::getTitle, req.getTitle())
                        .eq(ObjectUtil.isNotEmpty(req.getType()), DictAnnouncementPo::getType, req.getType())
                        .eq(ObjectUtil.isNotEmpty(req.getStatus()), DictAnnouncementPo::getStatus, req.getStatus())
                        .between(ObjectUtil.isNotEmpty(req.getBeginTime()) && ObjectUtil.isNotEmpty(req.getFinishTime()), DictAnnouncementPo::getCreatedTime, req.getBeginTime(), req.getFinishTime())
                        .orderByDesc(DictAnnouncementPo::getCreatedTime)
        );
        if (page == null) {
            return DictResult.error(null, "查询失败");
        }
        List<DictAnnouncementRspBo> rspBoList = new ArrayList<>();
        page.getRecords().forEach(po -> {
            DictAnnouncementRspBo rspBo = BeanUtil.copyProperties(po, DictAnnouncementRspBo.class);
            rspBoList.add(rspBo);
        });
        DictPage<DictAnnouncementRspBo> dictPage = DictPage.newInstance(req.getPageNo(), req.getPageSize(), rspBoList, page.getTotal());
        return DictResult.success(dictPage);
    }

    @Override
    public DictResult<List<DictAnnouncementRspBo>> selectList() {
        List<DictAnnouncementPo> poList = dictAnnouncementMapper.selectList(Wrappers.<DictAnnouncementPo>lambdaQuery()
                .eq(DictAnnouncementPo::getStatus, HomeConstant.ANNOUNCEMENT_STATUS_PUBLISHED)
                .orderByDesc(DictAnnouncementPo::getTop)
                .orderByDesc(DictAnnouncementPo::getPublishTime));
        if (CollectionUtil.isEmpty(poList)) {
            return DictResult.error(null, "暂无公告");
        }
        // 置顶时间在置顶生效时间与置顶失效时间之间
        Date currentDate = new Date();
        for (DictAnnouncementPo po : poList) {
            if (po.getStartTime() != null && po.getEndTime() != null) {
                if (currentDate.after(po.getStartTime()) && currentDate.before(po.getEndTime())) {
                    // 置顶时间有效，设置为置顶
                    po.setTop(HomeConstant.ANNOUNCEMENT_TOP);
                } else {
                    // 置顶时间无效，取消置顶
                    po.setTop(HomeConstant.ANNOUNCEMENT_NO_TOP);
                    po.setStartTime(null); // 清空置顶开始时间
                    po.setEndTime(null);   // 清空置顶结束时间
                    // 更新数据库中的记录
                    dictAnnouncementMapper.updateById(po);
                }
            }
        }
        // 降序排序, 置顶优先, 发布时间优先
        poList = poList.stream()
                .sorted(Comparator.comparing(DictAnnouncementPo::getTop, Comparator.nullsFirst(Comparator.reverseOrder()))
                        .thenComparing(DictAnnouncementPo::getPublishTime, Comparator.nullsFirst(Comparator.reverseOrder())))
               .collect(Collectors.toList());
        List<DictAnnouncementRspBo> rspBoList = poList.stream()
                .map(po -> BeanUtil.copyProperties(po, DictAnnouncementRspBo.class))
                .collect(Collectors.toList());
        return DictResult.success(rspBoList);
    }


    @Override
    public DictResult<DictPage<DictAnnouncementRspBo>> queryAnnouncementPage(DictAnnouncementPageReqBo req) {
        // 分页查询
        Page<DictAnnouncementPo> page = dictAnnouncementMapper.selectPage(
                new Page<>(req.getPageNo(), req.getPageSize()),
                Wrappers.<DictAnnouncementPo>lambdaQuery()
                        .eq(DictAnnouncementPo::getStatus, HomeConstant.ANNOUNCEMENT_STATUS_PUBLISHED)
                        .like(StringUtils.isNotBlank(req.getTitle()), DictAnnouncementPo::getTitle, req.getTitle())
                        .eq(ObjectUtil.isNotEmpty(req.getType()), DictAnnouncementPo::getType, req.getType())
                        .between(ObjectUtil.isNotEmpty(req.getBeginTime()) && ObjectUtil.isNotEmpty(req.getFinishTime()), DictAnnouncementPo::getPublishTime, req.getBeginTime(), req.getFinishTime())
                        .orderByDesc(DictAnnouncementPo::getTop)
                        .orderByDesc(DictAnnouncementPo::getPublishTime));
        if (page == null) {
            return DictResult.error(null, "查询失败");
        }
        List<DictAnnouncementRspBo> rspBoList = new ArrayList<>();
        page.getRecords().forEach(po -> {
            DictAnnouncementRspBo rspBo = BeanUtil.copyProperties(po, DictAnnouncementRspBo.class);
            rspBoList.add(rspBo);
        });
        DictPage<DictAnnouncementRspBo> dictPage = DictPage.newInstance(req.getPageNo(), req.getPageSize(), rspBoList, page.getTotal());
        return DictResult.success(dictPage);
    }
}
