package com.tydic.dict.system.repository.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.metadata.OrderItem;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ohaotian.authority.common.rsp.DictPage;
import com.ohaotian.authority.common.rsp.DictResult;
import com.tydic.dict.system.repository.dao.DictMessageTemplateMapper;
import com.tydic.dict.system.repository.po.DictMessageTemplatePO;
import com.tydic.dict.system.repository.service.search.DictMessageTemplateSearchService;
import com.tydic.dict.system.service.bo.DictMessageTemplateQueryBo;
import com.tydic.dict.system.service.bo.DictMessageTemplateReaderBo;
import com.tydic.dict.system.service.bo.DictMessageTemplateRspBO;
import com.tydic.dict.system.service.common.ExcelUtil;
import freemarker.template.Configuration;
import freemarker.template.Template;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.util.FileCopyUtils;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.net.URLEncoder;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Service
@Slf4j
public class DictMessageTemplateSearchServiceImpl extends ServiceImpl<DictMessageTemplateMapper, DictMessageTemplatePO> implements DictMessageTemplateSearchService {

	@Autowired
	private DictMessageTemplateMapper dictMessageTemplateMapper;
	@Autowired
	private Configuration freemarkerConfig;
	@Autowired
	ExcelUtil excelUtil;

	private static final int BATCH_SIZE = 100;
	private static int ROW_NUM = 1;
	private static final String BASE_PATH = "dict-manage-infrastructure/src/main/resources/messageTemplate";
	private static final String ERROR_MESSAGE_BASE_PATH = "dict-manage-infrastructure/src/main/resources/messageTemplate/errorMessage.xlsx";
	private static final String MESSAGE_TEMPLATE_FILENAME = "messageTemplate.xlsx";

	/**
	 * 消息模版分页查询
	 *
	 * @return 分页结果
	 */
	@Override
	public DictResult<DictPage> pageQuery(DictMessageTemplateQueryBo reqBo) {
		// 分页查询
		Page<DictMessageTemplatePO> pageParam = new Page<DictMessageTemplatePO>(reqBo.getPageNo(), reqBo.getPageSize()).addOrder(OrderItem.desc("created_time"));
		Page<DictMessageTemplatePO> page = dictMessageTemplateMapper.selectPage(
				pageParam, Wrappers.<DictMessageTemplatePO>lambdaQuery()
						.eq(DictMessageTemplatePO::getDeleted, false)
						.like(ObjectUtil.isNotEmpty(reqBo.getName()), DictMessageTemplatePO::getName, reqBo.getName())
						.eq(ObjectUtil.isNotEmpty(reqBo.getCode()), DictMessageTemplatePO::getCode, reqBo.getCode())
						.eq(ObjectUtil.isNotEmpty(reqBo.getType()), DictMessageTemplatePO::getType, reqBo.getType())
						.eq(ObjectUtil.isNotEmpty(reqBo.getScene()), DictMessageTemplatePO::getScene, reqBo.getScene()));
		if (page != null) {
			DictPage<DictMessageTemplatePO> poDictPage = DictPage.newInstance(reqBo.getPageNo(), reqBo.getPageSize(), page.getRecords(), page.getTotal());
			return DictResult.success(poDictPage);
		}
		return DictResult.error(null, "查询失败");
	}

	/**
	 * 消息模版类型列表
	 *
	 * @return
	 */
	@Override
	public List<String> getTypeList() {
		List<DictMessageTemplatePO> dictMessageTemplatePOS = dictMessageTemplateMapper.selectList(
				Wrappers.<DictMessageTemplatePO>lambdaQuery().eq(DictMessageTemplatePO::getDeleted, false)
		);

		List<String> types = dictMessageTemplatePOS.stream()
				.map(DictMessageTemplatePO::getType)
				.filter(Objects::nonNull)
				.distinct()
				.collect(Collectors.toList());
		return types;
	}

	/**
	 * 消息模版场景列表
	 *
	 * @return 场景列表
	 */
	@Override
	public List<String> getSceneList() {
		List<DictMessageTemplatePO> dictMessageTemplatePOS = dictMessageTemplateMapper.selectList(
				Wrappers.<DictMessageTemplatePO>lambdaQuery().eq(DictMessageTemplatePO::getDeleted, false)
		);

		List<String> scenes = dictMessageTemplatePOS.stream()
				.map(DictMessageTemplatePO::getScene)
				.filter(Objects::nonNull)
				.distinct()
				.collect(Collectors.toList());
		return scenes;
	}

	/**
	 * 消息模版内容渲染, 通过freemarker实现
	 *
	 * @param readerBo 消息模版id
	 * @return rsp
	 */
	@Override
	public DictMessageTemplateRspBO render(DictMessageTemplateReaderBo readerBo) {
		if (readerBo.getId() == null || ObjectUtil.isEmpty(readerBo.getParams())) {
			return null;
		}
		// 从数据库获取模版内容
		DictMessageTemplatePO dictMessageTemplatePO = dictMessageTemplateMapper.selectById(readerBo.getId());
		// JSON转map
		ObjectMapper mapper = new ObjectMapper();
		Map<String, Object> map = null;
		try {
			map = mapper.readValue(readerBo.getParams(), Map.class);
		} catch (JsonProcessingException e) {
			log.error("JSON转map出现error: {}", e.getMessage());
		}

		if (ObjectUtil.isEmpty(map)) {
			// 无需渲染
			return BeanUtil.copyProperties(dictMessageTemplatePO, DictMessageTemplateRspBO.class);
		}

		String templateContent = dictMessageTemplatePO.getContent();

		// 将模版内容转换为freemarker可解析的字符串
		templateContent = convertTemplateSyntax(templateContent);

		// 渲染数据
		Template template = null;
		try {
			template = new Template("templateMessage", templateContent, freemarkerConfig);
		} catch (IOException e) {
			log.error("模版对象创建error: {}", e.getMessage());
		}
		StringWriter writer = new StringWriter();
		try {
			template.process(map, writer);
		} catch (Exception e) {
			log.error("模版渲染error: {}", e.getMessage());
		}
		templateContent = writer.toString();
		templateContent = evaluateExpressions(templateContent);

		// 渲染后的内容
		dictMessageTemplatePO.setContent(templateContent);

		return BeanUtil.copyProperties(dictMessageTemplatePO, DictMessageTemplateRspBO.class);
	}

	// 将 "{}" 替换为 "${}"
	private String convertTemplateSyntax(String template) {
		// 定义正则表达式来匹配大括号中的内容
		Pattern pattern = Pattern.compile("\\{([^{}]+)\\}");
		Matcher matcher = pattern.matcher(template);
		StringBuffer result = new StringBuffer();

		while (matcher.find()) {
			String content = matcher.group(1);
			// 替换为 ${...} 格式
			matcher.appendReplacement(result, "\\$\\{" + content + "\\}");
		}
		matcher.appendTail(result);
		return result.toString();
	}

	// 计算表达式并替换
	private static String evaluateExpressions(String input) {
		// 使用 JavaScript 计算
		ScriptEngine engine = new ScriptEngineManager().getEngineByName("JavaScript");

		// 正则表达式匹配数学表达式（如：2025+5）
		Pattern pattern = Pattern.compile("\\d+\\s*[-+*/]\\s*\\d+");
		Matcher matcher = pattern.matcher(input);
		StringBuffer result = new StringBuffer();

		while (matcher.find()) {
			String expression = matcher.group();

			try {
				// 计算表达式的值
				Object evalResult = engine.eval(expression);
				matcher.appendReplacement(result, evalResult.toString());
			} catch (Exception e) {
				matcher.appendReplacement(result, matcher.group());
				log.error("计算表达式失败: {}", e.getMessage());
			}
		}

		matcher.appendTail(result);
		return result.toString();
	}


	/**
	 * 导出消息模版为excel
	 */
	@Override
	public Boolean exportMessageTemplate(HttpServletResponse response) {
		try {
			List<DictMessageTemplatePO> templatePOList = dictMessageTemplateMapper.selectList(Wrappers.<DictMessageTemplatePO>lambdaQuery().eq(DictMessageTemplatePO::getDeleted, false));
			List<String> excludeColumns = Arrays.asList("id", "updatedBy", "updatedTime", "createdTime", "createdBy", "deleted");
			// 使用swagger 会导致各种问题，请使用浏览器测试
			response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
			response.setCharacterEncoding("utf-8");
			String fileName = URLEncoder.encode("消息模板", "UTF-8").replaceAll("\\+", "%20");
			response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
			EasyExcel.write(response.getOutputStream(), DictMessageTemplatePO.class)
					.sheet(fileName)
					.excludeColumnFieldNames(excludeColumns)
					.doWrite(templatePOList);
			return true;
		} catch (Exception e) {
			log.error("导出消息模版失败: {}", e.getMessage());
			return false;
		}
	}


	/**
	 * 下载消息模版
	 *
	 * @return
	 */
	@Override
	public Boolean downloadMessageTemplate(HttpServletResponse response) {
		// 构建文件路径
		Path filePath = Paths.get(BASE_PATH, MESSAGE_TEMPLATE_FILENAME);
		File file = filePath.toFile();

		// 检查文件是否存在
		if (!file.exists()) {
			response.setStatus(HttpStatus.NOT_FOUND.value());
			try {
				response.getWriter().write("文件未找到");
			} catch (Exception e) {
				log.error("下载消息模版失败: {}", e.getMessage());
				return false;
			}
		}

		response.setContentType("application/octet-stream");
		response.setHeader(HttpHeaders.CONTENT_DISPOSITION, "attachment; filename=\"" + file.getName() + "\"");

		try {
			FileCopyUtils.copy(new FileSystemResource(file).getInputStream(), response.getOutputStream());
		} catch (IOException e) {
			log.error("下载消息模版失败: {}", e.getMessage());
			return false;
		}
		return true;
	}

	@Override
	public Boolean downloadErrorMessageTemplate(HttpServletResponse response) {
		// 构建文件路径
		Path filePath = Paths.get(ERROR_MESSAGE_BASE_PATH);
		File file = filePath.toFile();

		// 检查文件是否存在
		if (!file.exists()) {
			response.setStatus(HttpStatus.NOT_FOUND.value());
			try {
				response.getWriter().write("文件未找到");
			} catch (Exception e) {
				log.error("错误报告下载失败: {}", e.getMessage());
				return false;
			}
		}

		response.setContentType("application/octet-stream");
		response.setHeader(HttpHeaders.CONTENT_DISPOSITION, "attachment; filename=\"" + file.getName() + "\"");

		try {
			FileCopyUtils.copy(new FileSystemResource(file).getInputStream(), response.getOutputStream());
		} catch (IOException e) {
			log.error("错误报告下载失败: {}", e.getMessage());
			return false;
		}
		return true;
	}
}
