package com.tydic.dict.system.repository.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.read.listener.PageReadListener;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ohaotian.authority.common.bo.BaseReqBO;
import com.ohaotian.authority.common.rsp.DictResult;
import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.tydic.dict.system.repository.dao.DictMessageTemplateMapper;
import com.tydic.dict.system.repository.po.DictMessageTemplatePO;
import com.tydic.dict.system.repository.service.update.DictMessageTemplateUpdateService;
import com.tydic.dict.system.repository.util.ValidationUtil;
import com.tydic.dict.system.service.bo.DictMessageTemplateErrorRspBO;
import com.tydic.dict.system.service.bo.DictMessageTemplateReqBO;
import com.tydic.dict.system.service.common.ExcelUtil;
import freemarker.template.Configuration;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Service
@Slf4j
public class DictMessageTemplateUpdateServiceImpl extends ServiceImpl<DictMessageTemplateMapper, DictMessageTemplatePO> implements DictMessageTemplateUpdateService {

	@Autowired
	private DictMessageTemplateMapper dictMessageTemplateMapper;
	@Autowired
	private Configuration freemarkerConfig;
	@Autowired
	ExcelUtil excelUtil;

	private static final int BATCH_SIZE = 100;
	private static int ROW_NUM = 1;
	private static final String BASE_PATH = "dict-manage-infrastructure/src/main/resources/messageTemplate";
	private static final String ERROR_MESSAGE_BASE_PATH = "dict-manage-infrastructure/src/main/resources/messageTemplate/errorMessage.xlsx";
	private static final String MESSAGE_TEMPLATE_FILENAME = "messageTemplate.xlsx";


	/**
	 * 消息模版添加
	 *
	 * @return 添加结果
	 */
	@Override
	public int add(DictMessageTemplateReqBO req) {
		List<DictMessageTemplatePO> templatePOList = dictMessageTemplateMapper.selectList(
				Wrappers.<DictMessageTemplatePO>lambdaQuery()
						.eq(DictMessageTemplatePO::getDeleted, false)
						.eq(DictMessageTemplatePO::getCode, req.getCode())
						.or()
						.eq(DictMessageTemplatePO::getName, req.getName()));
		if (CollectionUtil.isNotEmpty(templatePOList)) {
			return -1;
		}
		if (req.getCode().length() > 20) {
			throw new ZTBusinessException("消息模版标识 不超过20个字符");
		}
		if (req.getTitle().length() > 20) {
			throw new ZTBusinessException("消息标题 不超过20个字符");
		}
		if (req.getContent().length() > 500) {
			throw new ZTBusinessException("模版內容 不超过500个字符");
		}
		if (req.getName().length() > 20) {
			throw new ZTBusinessException("消息模版名称 不超过20个字符");
		}
		DictMessageTemplatePO messageTemplatePO = BeanUtil.toBean(req, DictMessageTemplatePO.class);
		if (ObjectUtil.isEmpty(messageTemplatePO.getTitle())) {
			messageTemplatePO.setTitle(messageTemplatePO.getName());
		}
		// todo 后续需要修改
//        messageTemplatePO.setUpdatedBy(req.getName());
//        messageTemplatePO.setCreatedBy(req.getName());
		messageTemplatePO.setUpdatedBy("admin");
		messageTemplatePO.setCreatedBy("admin");
		return dictMessageTemplateMapper.insert(messageTemplatePO);
	}

	/**
	 * 消息模版修改
	 *
	 * @return 修改结果
	 */
	@Override
	public int updateMessageTemplate(DictMessageTemplateReqBO req) {
		if (req.getId() != null) {
			// 校验code和name是否重复
			List<DictMessageTemplatePO> templatePOList = dictMessageTemplateMapper.selectList(
					Wrappers.<DictMessageTemplatePO>lambdaQuery()
							.ne(DictMessageTemplatePO::getId, req.getId())
							.eq(DictMessageTemplatePO::getDeleted, false)
							.and(innerWrapper -> innerWrapper
									.eq(DictMessageTemplatePO::getCode, req.getCode())
									.or()
									.eq(DictMessageTemplatePO::getName, req.getName())
							)
			);
			if (CollectionUtil.isNotEmpty(templatePOList)) {
				return -1;
			}
			if (req.getCode().length() > 20) {
				throw new ZTBusinessException("消息模版标识 不超过20个字符");
			}
			if (req.getTitle().length() > 20) {
				throw new ZTBusinessException("消息标题 不超过20个字符");
			}
			if (req.getContent().length() > 500) {
				throw new ZTBusinessException("模版內容 不超过500个字符");
			}
			if (req.getName().length() > 20) {
				throw new ZTBusinessException("消息模版名称 不超过20个字符");
			}
			DictMessageTemplatePO po = dictMessageTemplateMapper.selectById(req.getId());

			po.setUpdatedTime(new Date());
			// todo 后续需要修改
//            po.setUpdatedBy(req.getName());
			po.setUpdatedBy("admin");
			BeanUtil.copyProperties(req, po);
			return dictMessageTemplateMapper.update(po, Wrappers.<DictMessageTemplatePO>lambdaUpdate()
					.eq(DictMessageTemplatePO::getId, req.getId()));
		}
		return 0;
	}


	// 将 "{}" 替换为 "${}"
	private String convertTemplateSyntax(String template) {
		// 定义正则表达式来匹配大括号中的内容
		Pattern pattern = Pattern.compile("\\{([^{}]+)\\}");
		Matcher matcher = pattern.matcher(template);
		StringBuffer result = new StringBuffer();

		while (matcher.find()) {
			String content = matcher.group(1);
			// 替换为 ${...} 格式
			matcher.appendReplacement(result, "\\$\\{" + content + "\\}");
		}
		matcher.appendTail(result);
		return result.toString();
	}

	// 计算表达式并替换
	private static String evaluateExpressions(String input) {
		// 使用 JavaScript 计算
		ScriptEngine engine = new ScriptEngineManager().getEngineByName("JavaScript");

		// 正则表达式匹配数学表达式（如：2025+5）
		Pattern pattern = Pattern.compile("\\d+\\s*[-+*/]\\s*\\d+");
		Matcher matcher = pattern.matcher(input);
		StringBuffer result = new StringBuffer();

		while (matcher.find()) {
			String expression = matcher.group();

			try {
				// 计算表达式的值
				Object evalResult = engine.eval(expression);
				matcher.appendReplacement(result, evalResult.toString());
			} catch (Exception e) {
				matcher.appendReplacement(result, matcher.group());
				log.error("计算表达式失败: {}", e.getMessage());
			}
		}

		matcher.appendTail(result);
		return result.toString();
	}


	/**
	 * 导入消息模版
	 *
	 * @param file
	 * @param req
	 */
	@Override
	public DictResult<String> uploadMessageTemplate(MultipartFile file, BaseReqBO req) {
		if (file == null) {
			return DictResult.error(null, "内容为空");
		}
		HashMap<String, String> errorMap = new HashMap<>();
		List<String> errorList = new ArrayList<>();
		try {
			excelUtil.readExcel(file.getInputStream(),
					DictMessageTemplatePO.class,
					new PageReadListener<>(dataList -> {
						// 校验每条数据
						for (DictMessageTemplatePO po : dataList) {
							String error = ValidationUtil.validateDictMessageTemplate(po);
							if (StringUtils.isNoneBlank(error)) {
								errorMap.put(po.getCode(), error);
							}
						}

						errorMap.forEach((k, v) -> {
							if (StringUtils.isNoneBlank(v)) {
								String error = "数据校验失败";
								errorList.add(error);
							}
						});
						if (errorList.isEmpty()) {
							// 保存数据
							saveBatch(dataList, req);
						}
						generateErrorReport(dataList, errorMap, errorList);
					})
			);
			if (errorList.size() > 0) {
				return DictResult.error(null, "导入失败，共有 " + errorList.size() + " 条数据有误，请修改后重新导入。, ");
			}
			return DictResult.success("导入成功");
		} catch (Exception e) {
			log.error("导入消息模版失败: {}", e.getMessage());
			return DictResult.error(null, "导入失败");
		}
	}

	/**
	 * 保存数据
	 */
	public void saveBatch(List<DictMessageTemplatePO> dataList, BaseReqBO req) {
		List<DictMessageTemplatePO> batchList = new ArrayList<>();
		batchList.addAll(dataList);
		// 数据达到设置的上限，执行插入操作
		if (batchList.size() >= BATCH_SIZE) {
			batchList.forEach(po -> {
				// todo
//                po.setUpdatedBy(req.getName());
//                po.setCreatedBy(req.getName());
				po.setUpdatedBy("admin");
				po.setCreatedBy("admin");
			});
			this.saveBatch(batchList, req);
			batchList.clear();
		}
		// 插入剩余的数据
		if (!batchList.isEmpty()) {
			this.saveBatch(batchList, req);
		}
	}

	/**
	 * 生成错误报告
	 */
	public void generateErrorReport(List<DictMessageTemplatePO> dataList, HashMap<String, String> errorMap, List<String> errorList) {
		// 生成并保存错误报告，可以生成一个 Excel文件，并将下载链接提供给用户
		List<DictMessageTemplateErrorRspBO> errorRspBOList = new ArrayList<>();
		List<String> excludeColumns = Arrays.asList("id");
		dataList.forEach(po -> {
			DictMessageTemplateErrorRspBO errorRspBO = new DictMessageTemplateErrorRspBO();
			BeanUtil.copyProperties(po, errorRspBO);
			String errorMessage = errorMap.get(po.getCode());
			errorRspBO.setErrorMessage(errorMessage);
			errorRspBOList.add(errorRspBO);
		});
		try {
			String fileName = URLEncoder.encode("错误报告", "UTF-8").replaceAll("\\+", "%20");
			EasyExcel.write(Files.newOutputStream(Paths.get(ERROR_MESSAGE_BASE_PATH)), DictMessageTemplateErrorRspBO.class)
					.sheet(fileName)
					.excludeColumnFieldNames(excludeColumns)
					.doWrite(errorRspBOList);
		} catch (Exception e) {
			log.error("生成错误报告失败: {}", e.getMessage());
		}

	}

}
