package com.tydic.dyc.base.utils.gendoc;

import cn.hutool.core.util.ObjectUtil;
import com.tydic.dyc.base.utils.gendoc.bo.GenericObject;
import com.tydic.dyc.base.utils.gendoc.bo.ParamBo;
import com.tydic.utils.generatedoc.annotation.DocField;
import com.tydic.utils.generatedoc.annotation.DocInterface;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;

/**
 * 描述 根据接口和出入参属性上的注解生成confluence标准文档工具类
 *
 * @author tgy
 * @date 2020/3/15 15:43
 */
public class GenDocUtil {
    /**
     * 当属性未使用注解时，使用默认值
     */
    private static final String DEFAULT_FIELD_DESC = " ";
    /**
     * 必传
     */
    private static final String REQUIRED_YES = "Y";
    /**
     * 非必传
     */
    private static final String REQUIRED_NO = "N";

    /**
     * 出参为空
     */
    private static final String VOID_RETURN = "void";

    /**
     * 生成入参
     */
    private static final boolean GEN_INPUT = true;

    /**
     * 生成出参
     */
    private static final boolean GEN_OUTPUT = false;

    /**
     * html文件的名称
     */
    private static final String HTML_NAME = "code2Confluence.html";

    /**
     * 排除值，他们会让wiki输入报错
     */
    private static Set<String> exclude = new HashSet<>();

    static {
        exclude.add("hostName");
    }


    public static String genInput(Class parameterType) {
        StringBuilder sb = new StringBuilder();
        genParam(sb, parameterType, GEN_INPUT);
        return sb.toString();
    }

    public static String genOutput(Class parameterType) {
        StringBuilder sb = new StringBuilder();
        genParam(sb, parameterType, GEN_OUTPUT);
        return sb.toString();
    }

    /**
     * 描述: 解析接口对象为wiki字符串，如果传入Class不是接口、没有DocInterface注解、接口没有方法，则跳过(返回空字符串)，且只解析从接口上能解析到的第一个方法
     *
     * @param interfaceClass:需要解析的接口的Class类
     * @return 解析的wiki字符串，可直接拷贝到confluence里面
     * @author tgy
     * @date 2020/3/15 15:55
     */
    public static String parseInterface(Class interfaceClass) {
        if (!interfaceClass.isInterface()) {
            System.err.println("手工警告: 接口[" + interfaceClass.getName() + "]不是接口，已跳过生成其文档");
            return null;
        }
        StringBuilder sb = new StringBuilder();
        //如果此接口没有接口注解，直接返回
        /*if (!interfaceClass.isAnnotationPresent(DocInterface.class)) {
            System.err.println("手工警告: 接口[" + interfaceClass.getName() + "]没有DocInterface注解，已跳过生成其文档");
            return null;
        }*/
        //获取接口的所有方法
        Method[] methods = interfaceClass.getDeclaredMethods();
        if (methods.length == 0) {
            System.err.println("手工警告: 接口[" + interfaceClass.getName() + "]一个方法也没有，已跳过生成其文档");
            return null;
        }

        //从Class对象上获取DocInterface注解对象
        DocInterface annotation = null;

        for (Method method : methods) {
            DocInterface tmpAnnotation = method.getAnnotation(DocInterface.class);
            if (tmpAnnotation != null) {
                annotation = tmpAnnotation;
                break;
            }
        }

        if (annotation == null) {
            System.err.println("手工警告: 接口[" + interfaceClass.getName() + "]没有方法有文档生成注解");
            return null;
        }

        String interfaceSimpleName = interfaceClass.getSimpleName();
        String interfaceName = interfaceClass.getName();
        String interfaceDesc = annotation.value();
        String[] logic = annotation.logic();
        String[] keyDataChanges = annotation.keyDataChanges();

        Method method = methods[0];
        String methodName = method.getName();
        //生成描述
        sb.append(genDesc(interfaceSimpleName + "接口的" + methodName + "方法，用于" + interfaceDesc));

        //生成使用场景
        sb.append(genUsed(interfaceDesc));
        //生成接口基本信息
        sb.append(genInterBaseInfo(interfaceDesc, interfaceSimpleName));

        //是否有入参和出参
        boolean noInput = false;
        boolean noReturn = false;

        //出入参定义
        Class parameterType = null;
        Class returnType = method.getReturnType();
        if (method.getParameterTypes().length == 0) {
            noInput = true;
        } else {
            parameterType = method.getParameterTypes()[0];
        }

        //出参为void
        if (VOID_RETURN.equals(returnType.getName())) {
            noReturn = true;
        }

        String inputClassName = noInput ? DEFAULT_FIELD_DESC : parameterType.getName();
        String outputClassName = noReturn ? DEFAULT_FIELD_DESC : returnType.getName();

        //生成部署依据,入参分别是：服务全类名，服务入参全类名，服务出参全类名
        sb.append(genDeploy(interfaceName, inputClassName, outputClassName, methodName));

        if (!noInput) {
            //生成入参(支持递归生成入参上用户自定义的对象和List中用户自定义的对象)
//            genInputParam(sb, parameterType);
            genParam(sb, parameterType, GEN_INPUT);
        }
        if (!noReturn) {
            //生成出参
//            genOutPutParam(sb, returnType);
            genParam(sb, returnType, GEN_OUTPUT);
        }

        //生成返回码
        sb.append(genReturnCode());
        //生成业务逻辑
        sb.append(genLogic(logic));
        //生成关键数据变化
        sb.append(genKey(keyDataChanges));
        //生成使用实例
        sb.append(genUseDemo());

        return sb.toString();
    }


    //统一参数生成方法
    private static void genParam(StringBuilder sb, Class parameterType, boolean isInput) {
        String inputParam = "\n||入参：" + parameterType.getName() + "\n" +
                "|名称|说明|类型|必填|默认值|长度|常量定义\n";
        String outPutParam = "\n||出参：" + parameterType.getName() + "\n" +
                "|名称|说明|类型|默认值|长度|常量定义\n";

        //添加各自的表头
        if (isInput) {
            sb.append(inputParam);
        } else {
            sb.append(outPutParam);
        }

        //得到所有字段
        List<Field> fields = getAllFields(parameterType);
        //获取父类的泛型对象
        List<Class> allGeneric = getAllGeneric(parameterType);
        //参数类，一个属性也没有，返回
        if (fields.size() == 0) {
            return;
        }
        //用LinkedHashSet把需要递归加载的类装起来，保证顺序和不重复生成对象上属性的表格
        Set<Class> classHashSet = new LinkedHashSet<>(allGeneric);
        for (Field field : fields) {
            ParamBo paramBo = new ParamBo();
            paramBo.setName(field.getName());
            //排除一些值
            if (exclude.contains(field.getName())) {
                paramBo.setName(field.getName() + "_");
            }
            paramBo.setDesc(DEFAULT_FIELD_DESC);
            paramBo.setRequired(REQUIRED_NO);
//            paramBo.setRequired(" ");
            Class typeClass = field.getType();
            //如果属性上有DocField注解，获取属性
            if (field.isAnnotationPresent(DocField.class)) {
                DocField annotation = field.getAnnotation(DocField.class);
                paramBo.setDesc(annotation.value());
                if (annotation.required()) {
                    paramBo.setRequired(REQUIRED_YES);
                }
            } else {
                if("code".equals(field.getName())){
                    paramBo.setDesc("应答返回状态码");
                }
                if("message".equals(field.getName())){
                    paramBo.setDesc("应答返回描述");
                }
            }
            //获得可能的泛型类型
            GenericObject genericType = getGenericType(field);
            //打印字段对应的simple字符串，例如Map<String,Object>
            String fieldTypeStr = genericType.getGenericStr();

            paramBo.setType(fieldTypeStr);

            if (isInput) {
                sb.append("| ").append(paramBo.getName()).append("| ").append(paramBo.getDesc()).append(" | ").append(paramBo.getType()).append("|").append(paramBo.getRequired()).append("| | | |\n");
            } else {
                sb.append("| ").append(paramBo.getName()).append("| ").append(paramBo.getDesc()).append(" | ").append(paramBo.getType()).append("| | | |\n");
            }

            //是否是用户自定义的类，如果是，则需要放到set里面，进行递归解析
            if (!isJavaClass(typeClass)) {
                classHashSet.add(typeClass);
            }

            //如果是泛型属性，把泛型里面的属性继续解析
            if (genericType.isGeneric()) {
                classHashSet.addAll(genericType.getGenericClasses());
            }
            //不等于空，说明是个List属性，应该把list那个泛型对象拿出来
            /*if (fieldTypeStr != null) {
                classHashSet.add(getListFieldClass(field));
            }*/
        }
        sb.append("\n\n");
        for (Class aClass : classHashSet) {
            //不跟当前类一样，且不是java的类，才会继续解析
            if (!parameterType.equals(aClass) && !isJavaClass(aClass)) {
                genParam(sb, aClass, isInput);
            }

        }
    }


    /**
     * 描述: 判断field是否是list，如果是list，得到类似List<Person>的泛型字符串
     *
     * @param field:入参
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:42
     */
    private static String isListField(Field field) {
        Class fieldClazz = field.getType();
        //基本类型，返回空
        if (fieldClazz.isPrimitive()) {
            return null;
        }
        //是否是List的子类Class对象
        if (fieldClazz.isAssignableFrom(List.class)) {
            // 关键的地方，如果是List类型，得到其Generic的类型
            Type fc = field.getGenericType();
            if (fc == null) {
                return null;
            }
            // 如果是泛型参数的类型
            if (fc instanceof ParameterizedType) {
                ParameterizedType pt = (ParameterizedType) fc;
                //得到泛型里的class类型对象。
                Class genericClazz = (Class) pt.getActualTypeArguments()[0];
                return "List<" + genericClazz.getSimpleName() + ">";
            }
        }
        return null;
    }

    /**
     * 描述: 判断field是否是list，如果是list，得到List泛型的Class对象
     *
     * @param field: 入参
     * @return java.lang.Class
     * @author tgy
     * @date 2020/3/15 16:44
     */
    private static Class getListFieldClass(Field field) {
        Class fieldClazz = field.getType();
        //基本类型，返回空
        if (fieldClazz.isPrimitive()) {
            return null;
        }
        //是List泛型的
        if (fieldClazz.isAssignableFrom(List.class)) {
            // 如果是List类型，得到其Generic的类型
            Type fc = field.getGenericType();
            if (fc == null) {
                return null;
            }
            if (fc instanceof ParameterizedType) {// 如果是泛型参数的类型
                ParameterizedType pt = (ParameterizedType) fc;
                return (Class) pt.getActualTypeArguments()[0]; //【4】 得到泛型里的class类型对象。
            }
        }
        return null;
    }

    /**
     * 描述: 判断是不是java自己加载的类，java自己加载的类获取不到类加载器
     *
     * @param clz:需要判断入参
     * @return boolean
     * @author tgy
     * @date 2020/3/15 15:54
     */
    private static boolean isJavaClass(Class<?> clz) {
        return clz != null && clz.getClassLoader() == null;
    }


    /**
     * 描述: 生成描述
     *
     * @param desc:描述
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:23
     */
    private static String genDesc(String desc) {
        return "||*描述*\n" +
                "|" + desc + "\n";
    }

    /**
     * 描述: 生成使用场景
     *
     * @param used:使用场景
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:23
     */
    private static String genUsed(String used) {
        return "\nh2. 使用场景：\n" +
                "||*使用场景*\n" +
                "|" + used + "\n";
    }


    /**
     * 描述: 生成接口基本信息
     *
     * @param desc：接口名称，
     * @param intName:名称
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:24
     */
    private static String genInterBaseInfo(String desc, String intName) {
        return "\nh2. 接口基本信息：\n" +
                "\n" +
                "||接口基本信息\n" +
                "|接口名称|" + desc + "\n" +
                "|HTTP请求方式|HTTP-POST\n" +
                "|接口地址|正式：[http://ip:port/ ] \n测试：[http://ip:port/ ]\n" +
                "|rap调用地址| |\n";
    }


    /**
     * 描述: 生成部署依据
     *
     * @param intName:服务全类名
     * @param inputName：服务入参全类名（可能为空）
     * @param outPutParam：服务出参全类名（可能为空）
     * @param methodName:               方法名
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:25
     */
    private static String genDeploy(String intName, String inputName, String outPutParam, String methodName) {
        return "\nh2. 平台组部署依据：\n" +
                "\n" +
                "||平台组部署依据|\n" +
                "|服务定义|值\n" +
                "|服务名称|" + intName + "\n" +
                "|入参类名|" + inputName + "\n" +
                "|出参类名|" + outPutParam + "\n" +
                "|方法名称|" + methodName + "\n" +
                "\nh2. 参数说明：\n";
    }


    private static List<Class> getAllGeneric(Class clazz) {
        List<Class> list = new LinkedList<>();
        if (clazz.getGenericSuperclass() instanceof ParameterizedType) {
            //有泛型了，把泛型class得到
            ParameterizedType pt = (ParameterizedType) clazz.getGenericSuperclass();
            Type[] actualTypeArguments = pt.getActualTypeArguments();
            for (Type actualTypeArgument : actualTypeArguments) {
                try {
                    Class aClazz = Class.forName(actualTypeArgument.getTypeName());
                    if (!aClazz.isInterface()) {
                        list.add(aClazz);
                    }
                } catch (ClassNotFoundException e) {
                    System.err.println("获取类得泛型时有不明确的泛型类：[" + actualTypeArgument.getTypeName() + "],已忽略掉递归解析");
                }
            }
        }

//        clazz.getGenericInterfaces();
        return list;
    }

    /**
     * 描述: 获取对象的属性和所有父类的属性，会过滤serialVersionUID字段
     *
     * @param inputClass:要获取所有属性的Class对象
     * @return java.util.List<java.lang.reflect.Field>
     * @author tgy
     * @date 2020/3/15 16:28
     */
    private static List<Field> getAllFields(Class inputClass) {
        List<Field> retList = new ArrayList<>();
        Class clazz = inputClass;
        //用栈把类放进去，先放子类，再放父类，然后重新取出来，放到list里面，这样类的顺序就是从服务属性开始展示
        Stack<Class> classStack = new Stack<>();
        List<Class> classList = new LinkedList<>();
        while (clazz != null) {
            classStack.push(clazz);
            clazz = clazz.getSuperclass();
        }
        while (!classStack.empty()) {
            classList.add(classStack.pop());
        }
        for (Class aClass : classList) {
            //Object对象跳过
            if (!"java.lang.Object".equals(aClass.getName())) {
                Field[] fieldList = aClass.getDeclaredFields();
                for (Field field : fieldList) {
                    //serialVersionUID字段跳过
                    if (!"serialVersionUID".equals(field.getName()) && field.getAnnotation(DocField.class) != null) {
                        retList.add(field);
                    }
                    //兼容应用层haotian基类，code，message字段
                    if ("code".equals(field.getName()) || "message".equals(field.getName())) {
                        if ("com.tydic.dyc.base.bo.RspBO".equals(aClass.getName())) {
                            retList.add(field);
                        }
                    }
                }
            }


        }
        return retList;
    }


    /**
     * 描述: 生成返回编码
     *
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:26
     */
    private static String genReturnCode() {
        return "\n||返回码\n" +
                "|返回编码|返回描述|返回场景描述\n" +
                "|0000|成功| |\n" +
                "|8888|系统异常| |\n";
    }

    /**
     * 描述: 业务逻辑生成
     *
     * @param logic:业务逻辑字符串数组
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:27
     */
    private static String genLogic(String[] logic) {
        StringBuilder logicSb = new StringBuilder("\nh2. 业务逻辑：\n" +
                "\n" +
                "||业务逻辑\n" +
                "|步骤|描述\n");

        for (int i = 0; i < logic.length; i++) {
            logicSb.append("|").append(i + 1).append("|").append(logic[i]).append(" |\n");
        }
        return logicSb.toString();
    }


    /**
     * 描述: 业务关键数据变化
     *
     * @param keyChange:业务逻辑字符串数组
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:27
     */
    private static String genKey(String[] keyChange) {
        StringBuilder keyChangeSb = new StringBuilder("\nh2. 关键数据变化：\n" +
                "\n" +
                "||关键数据变化\n" +
                "|表名|变化描述\n");

        for (int i = 0; i < keyChange.length; i++) {
            if (ObjectUtil.isNotEmpty(keyChange[i])) {
                String[] split = keyChange[i].split(":");
                keyChangeSb.append("|").append(split[0]).append("|").append(split[1]).append(" |\n");
            }
        }
        return keyChangeSb.toString();
    }

    /**
     * 描述: 生成报文
     *
     * @return java.lang.String
     * @author tgy
     * @date 2020/3/15 16:27
     */
    private static String genUseDemo() {
        return "\nh2. 使用示例：\n" +
                "\n" +
                "||示例报文| |\n" +
                "|入参报文| |\n" +
                "|出参报文| |";
    }


    //得到field的泛型抽象对象
    private static GenericObject getGenericType(Field field) {
        GenericObject rsp = new GenericObject();
        Class type = field.getType();
        rsp.setSelfClass(type);
        List<Class> genericClasses = new LinkedList<>();
        rsp.setGenericClasses(genericClasses);
        Type genericType = field.getGenericType();
        StringBuilder sb = new StringBuilder();
        if (genericType instanceof ParameterizedType) {
            rsp.setGeneric(true);
            ParameterizedType pt = (ParameterizedType) genericType;
            Type[] actualTypeArguments = pt.getActualTypeArguments();
            sb.append(type.getSimpleName()).append("<");
            for (Type actualTypeArgument : actualTypeArguments) {
                try {
                    Class clazz = Class.forName(actualTypeArgument.getTypeName());
                    if (!clazz.isInterface()) {
                        genericClasses.add(clazz);
                        sb.append(clazz.getSimpleName()).append(",");
                    }
                } catch (ClassNotFoundException e) {
                    System.err.println("有不明确的泛型类：[" + actualTypeArgument.getTypeName() + "],已忽略掉递归解析");
                    sb.append(actualTypeArgument.getTypeName()).append(",");
                }
            }
            String willStr = sb.toString();
            rsp.setGenericStr(willStr.substring(0, (willStr.length() - 1)) + ">");
        } else {
            rsp.setGeneric(false);
            sb.append(type.getSimpleName());
            rsp.setGenericStr(sb.toString());
        }

        return rsp;
    }


}
