package com.tydic.utils.npc.controller;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReflectUtil;
import com.alibaba.dubbo.rpc.service.GenericService;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.taobao.hsf.app.api.util.HSFApiConsumerBean;
import com.tydic.utils.npc.bo.NpcRpcControllerReqBo;
import com.tydic.utils.npc.bo.NpcRpcControllerRspBo;
import com.tydic.utils.npc.properties.NpcNativeProperties;
import lombok.extern.ohaotian.TempServiceInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

/**
 * 标题：NpcRpcController
 * 说明：RPC调用Controller，解决本机开发调用服务器，网络不通问题，借由此controller实现本机调用服务器上部署的服务或者任何Bean，暂不支持Dubbo
 * 时间：2021/9/3 16:33
 *
 * @author 何秀全
 */
@Slf4j
@RestController
@ConditionalOnProperty(name = "npc.native.is-open-rpc", havingValue = "true")
public class NpcRpcController {
    /**
     * HsfRpc类型名称
     */
    private static final String HSF_RPC_TYPE = "HSF";
    /**
     * SpringCloudRpc类型名称
     */
    private static final String SPRING_CLOUD_RPC_TYPE = "SpringCloud";
    /**
     * 没有符合条件的Bean类型
     */
    private static final String NO_QUALIFYING_BEAN_OF_TYPE = "No qualifying bean of type";
    /**
     * 应用上下文
     */
    @Autowired
    private ApplicationContext context;

    /**
     * RPC调用controller：URL可以由配置npc.url.suffix决定，默认：/npc/native/noauth/rpc
     *
     * @param reqBo RPC入参
     * @return rsp RPC出参
     */
    @PostMapping("${npc.native.url-suffix:/deploy/federate/noauth/rpc}")
    public NpcRpcControllerRspBo callNpcRpc(@RequestBody NpcRpcControllerReqBo reqBo) {
        String validateArg = validateArg(reqBo);
        if (null != validateArg) {
            return failed(validateArg);
        }
        String interClass = reqBo.getInterClass();
        String methodName = reqBo.getMethod();
        //获取服务的Class类
        Class<?> aClass;
        try {
            aClass = Class.forName(interClass);
        } catch (ClassNotFoundException e) {
            return failed("该class(" + interClass + ")不存在，请检查该maven依赖");
        }

        Object reqDataObj = reqBo.getData();
        String reqDataJson = JSON.toJSONString(reqDataObj);
        log.info("RPC调用服务入参：" + reqDataJson);

        //获取该方法名对应的Method对象
        Method method = ReflectUtil.getMethodByName(aClass, methodName);
        if (method == null) {
            return failed("该服务(" + interClass + ")的方法(" + methodName + ")不存在");
        }

        //获取入参类型，并转换成对应的对象，此处只考虑服务方法只有一个入参的情况
        Class<?>[] parameterTypes = method.getParameterTypes();
        Class<?> parameterType = parameterTypes[0];

        TempServiceInfo tempAnnotation = aClass.getAnnotation(TempServiceInfo.class);
        if (tempAnnotation == null) {
            //判断是否为RPC服务，不是调用本机Bean
            return invokeLocalBeanMethod(reqDataJson, aClass, parameterType, method);
        }
        String version = tempAnnotation.version();
        String group = tempAnnotation.group();
        String rpcType = tempAnnotation.invokeTypes()[0].toString();
        log.info("TempServiceInfo内容为:version:{};group:{};rpcType:{}", version, group, rpcType);

        if (HSF_RPC_TYPE.equalsIgnoreCase(rpcType)) {
            //HSF 逻辑,走HSF泛化调用
            try {
                HSFApiConsumerBean hsfApiConsumerBean = new HSFApiConsumerBean();
                hsfApiConsumerBean.setInterfaceName(interClass);
                hsfApiConsumerBean.setVersion(version);
                hsfApiConsumerBean.setGroup(group);
                // [设置] 泛化配置，将HSFConsumerBean配置generic为true，标识HSF客户端忽略加载不到接口的异常。
                hsfApiConsumerBean.init(true);
                //入参转换成map
                JSONObject reqJsonObject = JSON.parseObject(reqDataJson, JSONObject.class);
                Map<String, Object> reqMap = new HashMap<>(reqJsonObject.size());
                for (Map.Entry<String, Object> reqJsonEntry : reqJsonObject.entrySet()) {
                    Object value = reqJsonEntry.getValue();
                    if (ObjectUtil.isNotEmpty(value)) {
                        reqMap.put(reqJsonEntry.getKey(), value);
                    }
                }
                GenericService genericOrderService = (GenericService) hsfApiConsumerBean.getObject();
                Object rspObject = genericOrderService.$invoke(methodName, new String[]{parameterType.getName()}, new Object[]{reqMap});
                JSONObject rspJsonObject = JSON.parseObject(JSONObject.toJSONString(rspObject), JSONObject.class);
                rspJsonObject.remove("class");
                log.info("RPC调用服务出参：" + JSON.toJSONString(rspJsonObject));
                return success(rspJsonObject);
            } catch (Exception e) {
                return failed("[" + rpcType + "]服务调用异常：" + e.getMessage());
            }
        } else if (SPRING_CLOUD_RPC_TYPE.equalsIgnoreCase(rpcType)) {
            //SpringCloud逻辑，走bean调用，需要判断bean是否存在，从Spring容器中获取该真实可执行的类
            return invokeLocalBeanMethod(reqDataJson, aClass, parameterType, method);
        } else {
            return failed("该服务调用RPC类型(" + rpcType + ")不支持");
        }
    }

    /**
     * 入参校验
     *
     * @param reqBo 入参
     * @return 错误描述
     */
    private String validateArg(NpcRpcControllerReqBo reqBo) {
        if (null == reqBo) {
            return "入参对象不能为空";
        }
        if (ObjectUtil.isEmpty(reqBo.getInterClass())) {
            return "入参对象属性[interClass]不能为空";
        }
        if (ObjectUtil.isEmpty(reqBo.getMethod())) {
            return "入参对象属性[method]不能为空";
        }
        return null;
    }

    /**
     * 反射调用本机Bean方法
     *
     * @param reqDataJson   入参JSON字符串
     * @param aClass        Bean类
     * @param parameterType 参数类型
     * @param method        调用方法
     * @return 出参
     */
    private NpcRpcControllerRspBo invokeLocalBeanMethod(String reqDataJson, Class<?> aClass, Class<?> parameterType, Method method) {
        Object bean;
        try {
            bean = context.getBean(aClass);
        } catch (Exception e) {
            if (e.getMessage().contains(NO_QUALIFYING_BEAN_OF_TYPE)) {
                return failed("该服务(" + aClass.getName() + ")不存在");
            }
            return failed("getBean异常:" + e.getMessage());
        }

        Object invoke = ReflectUtil.invoke(bean, method, JSON.parseObject(reqDataJson, parameterType));
        String jsonStr = JSON.toJSONString(invoke);
        log.info("RPC调用服务出参：" + jsonStr);
        return success(invoke);
    }

    /**
     * 描述: 生成一个失败对象
     *
     * @param errorMessage 错误描述
     * @return com.tydic.order.controller.RpcControllerRspBo
     * @author tgy
     * @date 2021/2/23 18:57
     */
    private NpcRpcControllerRspBo failed(String errorMessage) {
        NpcRpcControllerRspBo rspBo = new NpcRpcControllerRspBo();
        rspBo.setCode("1");
        rspBo.setMessage(errorMessage);
        log.info("RPC调用服务失败原因：{}", errorMessage);
        return rspBo;
    }

    /**
     * 描述: 生成一个成功对象
     *
     * @param data 对象信息
     * @return com.tydic.order.controller.RpcControllerRspBo
     * @author tgy
     * @date 2021/2/23 18:57
     */
    private NpcRpcControllerRspBo success(Object data) {
        NpcRpcControllerRspBo rspBo = new NpcRpcControllerRspBo();
        rspBo.setCode("0");
        rspBo.setMessage("成功");
        rspBo.setData(data);
        return rspBo;
    }
}
