package com.tydic.extend.field.advice;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.ohaotian.plugin.cache.CacheClient;
import com.tydic.dyc.base.bo.BaseExtendBo;
import com.tydic.dyc.base.bo.BaseExtendFieldBo;
import com.tydic.extend.field.constants.ExtFieldConvertConstants;
import com.tydic.extend.field.model.ExtendFieldChildConfigDataBo;
import com.tydic.extend.field.model.ExtendFieldConfigDataBo;
import com.tydic.extend.field.utils.ObjectUtils;
import com.tydic.extend.field.utils.Sequence;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.springframework.jdbc.core.JdbcTemplate;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.tydic.extend.field.constants.ExtFieldConvertConstants.*;

/**
 * @ClassName HandleExtendFieldByPointCutAdvice
 * @Description 处理扩展字段（入库、查询）
 * @Author liugs
 * @Date 2022/10/9 14:52
 */
@Slf4j
public class HandleExtendFieldByPointCutAdvice implements MethodInterceptor {

    /** 占位符匹配正则 */
    public static final Pattern PLACE_PATTERN = Pattern.compile("(\\$\\{)([\\w]+)(})");
    /** 雪花占位符 */
    public static final String SNOWFLAKE_PLACEHOLDER = "SNOWFLAKE";

    private CacheClient cacheClient;

    private JdbcTemplate jdbcTemplate;

    public HandleExtendFieldByPointCutAdvice(CacheClient cacheClient, JdbcTemplate jdbcTemplate) {
        this.cacheClient = cacheClient;
        this.jdbcTemplate = jdbcTemplate;
    }

    @Override
    public Object invoke(MethodInvocation invocation) throws Throwable {
        // 切面一定要放在数据库事务切面内，保证插入失败可以事务回滚。
        // 考虑到有些表的主键要先存在，所以先执行业务，再处理扩展字段。
        Object rspObj = invocation.proceed();

        // 获取配置：服务名称+方法名称
        StringBuilder cacheKeyBuilder = new StringBuilder();
        cacheKeyBuilder.append(invocation.getMethod().getDeclaringClass().getName())
                .append(StrUtil.DOT)
                .append(invocation.getMethod().getName())
                .append(StrUtil.DOT)
                .append(ExtFieldConvertConstants.CONFIG_KEY_SUFFIX);

        Object config = cacheClient.get(cacheKeyBuilder.toString());
        // 没有配置，直接返回去了。
        if (ObjectUtil.isEmpty(config)) {
            log.info("没有获取到扩展配置");
            return rspObj;
        }

        // 出入参放入到map中，如果有相同字段，出参覆盖入参
        Map<String, Object> paramMap = new HashMap<>(16);
        // 入参放入 paramMap
        ObjectUtils.putRequestParamIntoParamMap(invocation.getArguments(), paramMap);
        // 出参放入 paramMap
        ObjectUtils.convertObjectToMap(rspObj, paramMap);

        // 配置信息
        ExtendFieldConfigDataBo configDataBo = JSON.parseObject(config.toString(), ExtendFieldConfigDataBo.class);
        // 判断操作类型
        if (EXT_FIELD_OPERATE_TYPE_ADD.equals(configDataBo.getOperateType())) {
            log.info("===开始处理扩展字段信息入库");
            extendFieldInsert(configDataBo, paramMap);
            log.info("===扩展字段信息入库完成");
        } else {
            log.info("===开始处理扩展字段查询");
            extendFieldQuery(configDataBo, paramMap, rspObj);
            log.info("===扩展字段信息查询完成");
        }

        return rspObj;
    }

    /**
     * 描述 扩展字段入库
     * @param configDataBo
     * @param paramMap
     * @return void
     * @author liugs
     * @date 2022/10/9 15:31
     */
    private void extendFieldInsert(ExtendFieldConfigDataBo configDataBo, Map<String, Object> paramMap) {
        // 顶层对象的扩展字段
        Object extendFieldBo = paramMap.remove(EXT_FILED_NAME);
        if (ObjectUtil.isNotEmpty(extendFieldBo)) {
            List<BaseExtendFieldBo> extendFieldBoList = JSON.parseArray(JSON.toJSONString(extendFieldBo), BaseExtendFieldBo.class);
            doInsert(configDataBo.getDynamicSql(), paramMap, extendFieldBoList);
        }

        // 子对象扩展字段配置
        if (CollectionUtil.isNotEmpty(configDataBo.getSubFieldConfig())) {
            for (ExtendFieldChildConfigDataBo configItem : configDataBo.getSubFieldConfig()) {
                Object childObj = paramMap.remove(configItem.getFieldCode());
                if (ObjectUtil.isEmpty(childObj)) {
                    continue;
                }
                // 对象子字段的扩展字段处理
                childExtendFieldInsert(childObj, configItem, paramMap);
            }
        }
    }

    /**
     * 描述 对象属性的扩展字段处理
     * @param childObj
     * @param configItem
     * @param paramMap
     * @return void
     * @author liugs
     * @date 2022/10/9 15:54
     */
    private void childExtendFieldInsert(Object childObj, ExtendFieldChildConfigDataBo configItem, Map<String, Object> paramMap) {
        // 判断此属性是列表，还是普通对象
        if (childObj instanceof List) {
            Class clazz = childObj.getClass();
            // 反射调用获取list的size方法
            Method method = ReflectUtil.getMethod(clazz, SIZE);
            // 获取集合大小
            int size = ReflectUtil.invoke(childObj, method);
            // 反射获取到list的get方法
            Method getMethod = ReflectUtil.getMethod(clazz, GET, int.class);
            // 调用get方法
            Object firstListItem = ReflectUtil.invoke(childObj, getMethod, 0);
            if (!(firstListItem instanceof BaseExtendBo)) {
                return;
            }
            for (int i = 0; i < size; i ++) {
                Object listItem = ReflectUtil.invoke(childObj, getMethod, i);
                childExtendFieldInsert(listItem, configItem, paramMap);
            }
        } else if (childObj instanceof BaseExtendBo) {
            // 当前层级对象的扩展字段
            Object extendFieldBo =  ReflectUtil.getFieldValue(childObj, EXT_FILED_NAME);
            if (ObjectUtil.isNotEmpty(extendFieldBo)) {
                List<BaseExtendFieldBo> extendFieldBoList = JSON.parseArray(JSON.toJSONString(extendFieldBo), BaseExtendFieldBo.class);
                doInsert(configItem.getDynamicSql(), paramMap, extendFieldBoList);
            }
        }
        if (CollectionUtil.isNotEmpty(configItem.getSubFieldConfig())) {
            for (ExtendFieldChildConfigDataBo config : configItem.getSubFieldConfig()) {
                Object subObj = ReflectUtil.getFieldValue(childObj, config.getFieldCode());
                if (ObjectUtil.isEmpty(subObj)) {
                    continue;
                }
                // 对象子字段的扩展字段处理
                childExtendFieldInsert(subObj, config, paramMap);
            }
        }
    }

    /**
     * 描述 入库
     * @param dynamicSql
     * @param paramMap
     * @param extendFieldBoList
     * @return void
     * @author liugs
     * @date 2022/10/9 15:38
     */
    private void doInsert(String dynamicSql, Map<String, Object> paramMap, List<BaseExtendFieldBo> extendFieldBoList) {
        // 语句列表
        List<String> dynamicSqlList = new ArrayList<>();
        for (BaseExtendFieldBo item : extendFieldBoList) {
            Map<String, Object> extendFieldValueMap =JSON.parseObject(JSON.toJSONString(item), Map.class);
            extendFieldValueMap.putAll(paramMap);
            dynamicSqlList.add(buildDynamicSql(dynamicSql, extendFieldValueMap));
        }
        log.info("===本次扩展字段入库执行的SQL列表：{}", JSON.toJSONString(dynamicSqlList));
        // 批量执行
        try {
            int[] result = jdbcTemplate.batchUpdate(dynamicSqlList.toArray(new String[dynamicSqlList.size()]));
            log.info("语句执行结果：{}", result);
        } catch (Exception e) {
            log.error("执行扩展字段入库语句异常：{}", e.getMessage());
            throw new ZTBusinessException("扩展字段信息入库异常：" + e.getMessage());
        }
    }

    /**
     * 描述 扩展字段查询
     * @param configDataBo
     * @param paramMap
     * @param rspObj
     * @return void
     * @author liugs
     * @date 2022/10/9 15:31
     */
    private void extendFieldQuery(ExtendFieldConfigDataBo configDataBo, Map<String, Object> paramMap, Object rspObj) throws Throwable {
        // 判断当前对象是否继承于扩展基类
        if (rspObj instanceof BaseExtendBo) {
            ReflectUtil.setFieldValue(rspObj, EXT_FILED_NAME, doQuery(null, paramMap, configDataBo.getDynamicSql()));
        }

        // 判断是否有子查询
        if (CollectionUtil.isNotEmpty(configDataBo.getSubFieldConfig())) {
            for (ExtendFieldChildConfigDataBo childConfigDataBo : configDataBo.getSubFieldConfig()) {
                childExtendFieldQuery(childConfigDataBo, paramMap, rspObj);
            }
        }
    }

    /**
     * 描述 子对象扩展字段查询
     * @param childConfigDataBo
     * @param paramMap
     * @param rspObj
     * @return void
     * @author liugs
     * @date 2022/10/10 10:09
     */
    private void childExtendFieldQuery(ExtendFieldChildConfigDataBo childConfigDataBo, Map<String, Object> paramMap, Object rspObj) throws Throwable {
        // 取子对象
        Object subObject = ReflectUtil.getFieldValue(rspObj, childConfigDataBo.getFieldCode());
        // 如果子对象为空、不是List、没有继承于扩展字段基类，不执行。
        if (ObjectUtil.isEmpty(subObject)) {
            return;
        }
        // 列表类型
        if (subObject instanceof List) {
            Class clazz = subObject.getClass();
            // 反射调用List的size方法
            Method sizeMethod = ReflectUtil.getMethod(clazz, SIZE);
            // 获取集合大小
            int size = ReflectUtil.invoke(subObject, sizeMethod);
            // 反射获取到List的get方法
            Method getMethod = ReflectUtil.getMethod(clazz, GET, int.class);
            // 调用get方法获取数据
            Object firstSubItem = ReflectUtil.invoke(subObject, getMethod, 0);
            // 如果列表元素不是继承于扩展字段基类，不执行后续流程
            if (!(firstSubItem instanceof BaseExtendBo)) {
                return;
            }
            for (int i = 0; i <size; i ++) {
                Object subItem = ReflectUtil.invoke(subObject, getMethod, i);
                ReflectUtil.setFieldValue(subItem, EXT_FILED_NAME, doQuery(subItem, paramMap, childConfigDataBo.getDynamicSql()));
            }
        } else {
            // 对象类型
            if (subObject instanceof BaseExtendBo) {
                ReflectUtil.setFieldValue(subObject, EXT_FILED_NAME, doQuery(subObject, paramMap, childConfigDataBo.getDynamicSql()));
            }
        }

        // 判断是否还有子对象
        if (CollectionUtil.isNotEmpty(childConfigDataBo.getSubFieldConfig())) {
            for (ExtendFieldChildConfigDataBo childConfigItem : childConfigDataBo.getSubFieldConfig()) {
                childExtendFieldQuery(childConfigItem, paramMap, subObject);
            }
        }
    }

    /**
     * 描述 执行查询
     * @param obj
     * @param paramMap
     * @param dynamicSql
     * @return java.util.List<com.tydic.dyc.base.bo.BaseExtendFieldBo>
     * @author liugs
     * @date 2022/10/10 10:39
     */
    private List<BaseExtendFieldBo> doQuery(Object obj, Map<String, Object> paramMap, String dynamicSql) throws Throwable {
        // 扩展字段列表
        List<BaseExtendFieldBo> extendFieldBoList = new ArrayList<>();
        // 将对象转为map，并将顶层的值添加进去
        Map<String, Object> objValueMap = new HashMap<>(16);
        if (ObjectUtil.isNotEmpty(obj)) {
            ObjectUtils.convertObjectToMap(obj, objValueMap);
        }
        objValueMap.putAll(paramMap);

        // 执行查询
        List<Map<String, Object>> queryForList = null;
        String queryDynamicSql = buildDynamicSql(dynamicSql, objValueMap);
        log.error("===本次执行的扩展字段查询SQL语句：{}", queryDynamicSql);
        try {
            queryForList = jdbcTemplate.queryForList(queryDynamicSql);
        } catch (Exception e) {
            log.error("执行扩展字段查询语句异常：{}", e.getMessage());
        }
        // 查询结果不为空，赋值到列表。
        if (CollectionUtil.isNotEmpty(queryForList)) {
            queryForList.forEach(item -> extendFieldBoList.add(JSON.parseObject(JSON.toJSONString(item), BaseExtendFieldBo.class)));
        }
        return extendFieldBoList;
    }

    /**
     * 描述 构建动态SQL
     * @param dynamicSql
     * @param paramMap
     * @return java.lang.String
     * @author liugs
     * @date 2022/10/9 15:43
     */
    private String buildDynamicSql(String dynamicSql, Map<String, Object> paramMap) {
        StringBuffer stringBuffer = new StringBuffer();
        Matcher matcher = PLACE_PATTERN.matcher(dynamicSql);
        while (matcher.find()) {
            String group = matcher.group(2);
            if (paramMap.get(group) == null) {
                if (SNOWFLAKE_PLACEHOLDER.equals(group)) {
                    matcher.appendReplacement(stringBuffer, String.valueOf(Sequence.nextId()));
                } else {
                    matcher.appendReplacement(stringBuffer, "null");
                }
                continue;
            }
            matcher.appendReplacement(stringBuffer, "'" + paramMap.get(group).toString() + "'");
        }
        matcher.appendTail(stringBuffer);

        return stringBuffer.toString().trim();
    }
}
