package com.tydic.dyc.oc.components.event;

import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.UocRspConstants;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 标题:EventConsumerManger
 * 说明:事件消费者管理类
 * 时间:2022/3/11 10:09
 * 作者:田桂银
 */
@Component
@Slf4j
public class EventConsumerManger {

    /**
     * 在系统启动时，缓存数据查询实现类
     */
    private final Map<String, EventConsumer> eventMap = new ConcurrentHashMap<>();

    /**
     * 描述:根据事件编码获取事件实现类
     *
     * @param eventName 入参
     * @return com.tydic.dyc.oc.components.event.EventConsumer
     * @author tgy
     * @date 2022/3/11 11:19
     */
    public EventConsumer getEventConsumer(String eventName) {
        EventConsumer intRet = eventMap.get(eventName);
        if (intRet == null) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_FAIL, "事件编码实现类(" + eventName + ")没有实现类！");
        }
        return intRet;
    }

    /**
     * 描述:注册实现类
     *
     * @param consumer 入参
     * @return void
     * @author tgy
     * @date 2022/3/1 12:50
     */
    public void register(EventConsumer consumer) {
        Assert.notNull(consumer, "consumer不能为空");
        String event = consumer.getEvent();
        Assert.notNull(event, "consumer的event不能为空");

        //此处获取到得必须为空，不为空则为重复
        EventConsumer intRet = eventMap.get(event);
        if (intRet != null) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_FAIL, "事件编码实现类(" + event + ")出现多个实现类！");
        }
        log.info(EventConsumerManger.class.getSimpleName() + ".register注册了事件编码实现类：" + event);
        eventMap.put(event, consumer);
    }

}


