package com.tydic.dyc.oc.service.domainservice;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.*;
import com.tydic.dyc.oc.model.common.IUocCommonModel;
import com.tydic.dyc.oc.model.common.UocCommonDo;
import com.tydic.dyc.oc.model.order.sub.UocOrderAccessory;
import com.tydic.dyc.oc.model.order.sub.UocOrderTaskInst;
import com.tydic.dyc.oc.model.saleorder.IUocSaleOrderModel;
import com.tydic.dyc.oc.model.saleorder.UocSaleOrderDo;
import com.tydic.dyc.oc.model.saleorder.qrybo.UocSaleOrderItemQryBo;
import com.tydic.dyc.oc.model.saleorder.sub.UocSaleOrderItem;
import com.tydic.dyc.oc.model.shiporder.IUocShipOrderModel;
import com.tydic.dyc.oc.model.shiporder.UocShipOrderDo;
import com.tydic.dyc.oc.model.shiporder.qrybo.UocShipOrderItemQryBo;
import com.tydic.dyc.oc.model.shiporder.sub.UocShipOrderItem;
import com.tydic.dyc.oc.model.shiporder.sub.UocShipOrderMap;
import com.tydic.dyc.oc.service.domainservice.bo.UocShipOrderArrivaConfirmOrRefuseReqBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocShipOrderArrivaConfirmOrRefuseRspBo;
import com.tydic.dyc.oc.service.saleorder.bo.UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo;
import com.tydic.dyc.oc.utils.IdUtil;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.ohaotian.HTServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 描述: 〈领域服务-发货单到货确认/拒收实现〉
 * @author : yuhao
 * @date : 2022/8/25 9:13
 */

@HTServiceImpl
@Slf4j
public class UocShipOrderArrivaConfirmOrRefuseServiceImpl implements UocShipOrderArrivaConfirmOrRefuseService {

    /**
     * 到货确认
     */
    private static final Integer OP_FLAG_CONFIRM = 1;
	/**
	 * 到货拒收
	 */
	private static final Integer OP_FLAG_REFUSE = 2;
	@Autowired
	private IUocShipOrderModel iUocShipOrderModel;
	@Autowired
	private IUocSaleOrderModel iUocSaleOrderModel;
	@Autowired
	private IUocCommonModel iUocCommonModel;

	@Override
	public UocShipOrderArrivaConfirmOrRefuseRspBo dealArriva(UocShipOrderArrivaConfirmOrRefuseReqBo reqBo) {
		//参数校验
		this.validateArg(reqBo);
		//获取销售单明细信息(供后面运算使用)
		List<UocSaleOrderItem> saleOrderItemInfos = this.getSaleOrderItemInfos(reqBo);
		//获取销售单所有已发货明细(供后面运算使用)
		List<UocShipOrderItem> saleOrderAllShipItemInfos = this.getSaleOrderAllShipItemInfos(reqBo);
		//获取本次要确认/拒收的发货单所有已发货明细(供后面运算使用)
		List<UocShipOrderItem> shipOrderAllItemInfos = this.getShipOrderAllItemInfos(reqBo);
		//更新发货单信息
		this.updateShipOrderInfo(reqBo,shipOrderAllItemInfos);
		//更新销售单状态
		this.updateSaleOrderStatus(reqBo, saleOrderItemInfos, saleOrderAllShipItemInfos, shipOrderAllItemInfos);
		//提交发货单任务
		this.submitShipOrderTask(reqBo);

		//返回结果
		UocShipOrderArrivaConfirmOrRefuseRspBo rspBo = new UocShipOrderArrivaConfirmOrRefuseRspBo();
		rspBo.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
		rspBo.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
		return rspBo;
	}

	/**
	 * 提交发货单任务
	 * @param reqBo
	 */
	private void submitShipOrderTask(UocShipOrderArrivaConfirmOrRefuseReqBo reqBo){
		for (String taskId : reqBo.getTaskIds()) {
			UocCommonDo uocCommonDo = new UocCommonDo();
			List<UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo> taskInfos = new ArrayList<>();
			UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo task = new UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo();
			task.setTaskId(taskId);
			taskInfos.add(task);
			uocCommonDo.setCompleteTaskInfos(taskInfos);
			uocCommonDo.setUserId(reqBo.getUserId());
			uocCommonDo.setOrderId(reqBo.getOrderId());
			this.iUocCommonModel.dealTask(uocCommonDo);
		}
	}

	/**
	 * 获取本次要确认/拒收的发货单所有已发货明细
	 * @param reqBo
	 * @return
	 */
	private List<UocShipOrderItem> getShipOrderAllItemInfos(UocShipOrderArrivaConfirmOrRefuseReqBo reqBo) {
		UocShipOrderItemQryBo uocShipOrderItemQryBo = new UocShipOrderItemQryBo();
		uocShipOrderItemQryBo.setShipOrderIdList(reqBo.getShipOrderIdList());
		uocShipOrderItemQryBo.setOrderId(reqBo.getOrderId());
		uocShipOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
		UocShipOrderDo shipOrderDo = this.iUocShipOrderModel.getListShipOrderItem(uocShipOrderItemQryBo);
		if (null != shipOrderDo && !CollectionUtils.isEmpty(shipOrderDo.getShipOrderItemBoList())) {
			return shipOrderDo.getShipOrderItemBoList();
		}else{
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "获取发货明细为空");
		}
    }

	/**
	 * 获取销售单所有已发货明细
	 * @param reqBo
	 * @return
	 */
	private List<UocShipOrderItem> getSaleOrderAllShipItemInfos(UocShipOrderArrivaConfirmOrRefuseReqBo reqBo) {
		UocShipOrderItemQryBo uocShipOrderItemQryBo = new UocShipOrderItemQryBo();
		uocShipOrderItemQryBo.setOrderId(reqBo.getOrderId());
		uocShipOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
		UocShipOrderDo shipOrderDo = this.iUocShipOrderModel.getListShipOrderItem(uocShipOrderItemQryBo);
		if (null != shipOrderDo && !CollectionUtils.isEmpty(shipOrderDo.getShipOrderItemBoList())) {
			return shipOrderDo.getShipOrderItemBoList();
		}else{
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "未获取到销售单所有已发货明细");
		}
	}

	/**
	 * 获取销售单明细信息
	 * @param reqBo
	 * @return
	 */
	private List<UocSaleOrderItem> getSaleOrderItemInfos(UocShipOrderArrivaConfirmOrRefuseReqBo reqBo){
		UocSaleOrderItemQryBo qryBo = new UocSaleOrderItemQryBo();
		qryBo.setOrderId(reqBo.getOrderId());
		qryBo.setSaleOrderId(reqBo.getSaleOrderId());
		UocSaleOrderDo uocSaleOrderDo = this.iUocSaleOrderModel.qryListSaleOrderItem(qryBo);
		if (null != uocSaleOrderDo && !CollectionUtils.isEmpty(uocSaleOrderDo.getSaleOrderItems())){
			return uocSaleOrderDo.getSaleOrderItems();
		}else{
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "未获取到销售单明细信息");
		}
	}

	/**
	 * 更新发货单信息
	 * @param reqBo
	 */
	private void updateShipOrderInfo(UocShipOrderArrivaConfirmOrRefuseReqBo reqBo,List<UocShipOrderItem> shipOrderAllItemInfos){
		UocShipOrderDo updateShipOrderDo = JSONObject.parseObject(JSON.toJSONString(reqBo),UocShipOrderDo.class);
		//设置更新人id
		updateShipOrderDo.setUpdateOperId(String.valueOf(reqBo.getUserId()));
		//设置更新时间
		updateShipOrderDo.setUpdateTime(new Date());
		//设置到货时间
		updateShipOrderDo.setArriveTime(new Date());
		//设置附件信息
		if (!CollectionUtils.isEmpty(reqBo.getOrderAccessoryBoList())) {
			List<UocOrderAccessory> orderAccessoryBoList = new ArrayList<>();
			reqBo.getOrderAccessoryBoList().forEach(orderAccessoryAddBo -> {
				UocOrderAccessory uocOrderAccessory = JSONObject.parseObject(JSON.toJSONString(orderAccessoryAddBo),UocOrderAccessory.class);
				//设置订单id
				uocOrderAccessory.setOrderId(reqBo.getOrderId());
				//单据id
				uocOrderAccessory.setObjId(reqBo.getSaleOrderId());
				//单据类型
				uocOrderAccessory.setObjType(UocDicConstant.OBJ_TYPE.SALE);
				//设置创建时间
				uocOrderAccessory.setCreateTime(new Date());
				//设置创建人id
				uocOrderAccessory.setCreateOperId(String.valueOf(reqBo.getUserId()));
				//设置附件id
				uocOrderAccessory.setId(IdUtil.nextId());
				//设置附件类型
				uocOrderAccessory.setAttachmentType(UocAttachementTypeConstants.ARRIVAL_CONFIRM);
				orderAccessoryBoList.add(uocOrderAccessory);
			});
			updateShipOrderDo.setOrderAccessoryBoList(orderAccessoryBoList);
		}

		if (OP_FLAG_CONFIRM.equals(reqBo.getOpFlag())) {
			//到货确认
			this.dealConfirm(updateShipOrderDo, shipOrderAllItemInfos);
		}else{
			//拒收
			this.dealRefuse(updateShipOrderDo, shipOrderAllItemInfos);
		}
	}

	/**
	 * 到货确认
	 */
	private void dealConfirm(UocShipOrderDo uocShipOrderDo, List<UocShipOrderItem> uocShipOrderItemList) {
		uocShipOrderItemList.forEach(uocShipOrderItem -> {
			if (uocShipOrderItem.getRefuseCount().compareTo(BigDecimal.ZERO) > 0) {
				throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "已拒收发货单不能做到货确认，请重新选择");
			}
			if (uocShipOrderItem.getArriveCount().compareTo(BigDecimal.ZERO) > 0) {
				throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "存在已确认到货发货单，请重新选择");
			}
		});
		//设置发货单状态
		uocShipOrderDo.setShipOrderState(UocStateConstants.ShipOrder.FH_FH_YSH);
		this.iUocShipOrderModel.arrivaConfirm(uocShipOrderDo);
    }

    /**
     * 到货拒收
     */
    private void dealRefuse(UocShipOrderDo uocShipOrderDo, List<UocShipOrderItem> uocShipOrderItemList) {
		uocShipOrderItemList.forEach(uocShipOrderItem -> {
			if (uocShipOrderItem.getRefuseCount().compareTo(BigDecimal.ZERO) > 0) {
				throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "存在已拒收发货单，请重新选择");
			}
			if (uocShipOrderItem.getArriveCount().compareTo(BigDecimal.ZERO) > 0) {
				throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "已确认到货发货单不能做拒收，请重新选择");
			}
		});
		//设置发货单状态
		uocShipOrderDo.setShipOrderState(UocStateConstants.ShipOrder.FH_FH_JS);
		//设置竣工标志
		uocShipOrderDo.setFinishFlag(UocConstant.SHIP_ORDER_FINISH_FLAG.FINISH);
		this.iUocShipOrderModel.arrivaRefuse(uocShipOrderDo);
    }


	/**
	 * 更新销售单状态
	 * @param reqBo
	 */
	private void updateSaleOrderStatus(UocShipOrderArrivaConfirmOrRefuseReqBo reqBo,List<UocSaleOrderItem> saleOrderItemInfos,
									   List<UocShipOrderItem> saleOrderAllShipItemInfos,List<UocShipOrderItem> shipOrderAllItemInfos){
		// 在到货确认/拒收节点会有 采购数量 发货数量 到货数量 拒收数量，不会有 退货数量 售后数量
		// XS_DH_DH   已到货(全部到货) 采购数量 = 到货数量
		// XS_DH_JS   已拒收(全部拒收) 采购数量 = 拒收数量
		// XS_DH_BFDH 部分到货 采购数量 > (到货数量 + 拒收数量) &&  到货数量 > 0 && 拒收数量 > 0

		//根据销售单明细，计算出采购总数量
		BigDecimal purchaseCount = new BigDecimal(0);
		for (UocSaleOrderItem uocSaleOrderItem : saleOrderItemInfos){
			purchaseCount = purchaseCount.add(uocSaleOrderItem.getPurchaseCount()==null?new BigDecimal(0):uocSaleOrderItem.getPurchaseCount());
		}
		//根据销售单已发货明细 计算出该销售单对应的已到货数量，已拒收数量
		BigDecimal arrivedCount = new BigDecimal(0);
		BigDecimal refusedCount = new BigDecimal(0);
		for (UocShipOrderItem uocShipOrderItem : saleOrderAllShipItemInfos){
			arrivedCount = arrivedCount.add(uocShipOrderItem.getArriveCount()==null?new BigDecimal(0):uocShipOrderItem.getArriveCount());
			refusedCount = refusedCount.add(uocShipOrderItem.getRefuseCount()==null?new BigDecimal(0):uocShipOrderItem.getRefuseCount());
		}
		//根据本次到货确认/拒收发货单明细，计算出此次拒收或者到货确认的总数量
		BigDecimal mutiplyCount = new BigDecimal(0);
		for (UocShipOrderItem uocShipOrderItem : shipOrderAllItemInfos){
			mutiplyCount = mutiplyCount.add(uocShipOrderItem.getSendCount()==null?new BigDecimal(0):uocShipOrderItem.getSendCount());
		}
		//计算出本次到货确认/拒收后的，拒收或者到货确认的总数量
		if (OP_FLAG_CONFIRM.equals(reqBo.getOpFlag())){
			arrivedCount = arrivedCount.add(mutiplyCount);
		}
		else{
			refusedCount = refusedCount.add(mutiplyCount);
		}
		//计算出销售单的状态
		String state = null;
		if (purchaseCount.compareTo(arrivedCount)==0){
			state = UocStateConstants.SaleOrder.XS_DH_DH;
		}
		else if (purchaseCount.compareTo(refusedCount)==0){
			state = UocStateConstants.SaleOrder.XS_DH_JS;
		}
		else if (purchaseCount.compareTo(arrivedCount.add(refusedCount))==0){
			state = UocStateConstants.SaleOrder.XS_DH_BFDH;
		}

		//更新销售单状态
		if (!StringUtils.isBlank(state)) {
			UocSaleOrderDo updateSaleOrderDo = new UocSaleOrderDo();
			//设置销售单状态
			updateSaleOrderDo.setSaleOrderState(state);
			//设置订单id
			updateSaleOrderDo.setOrderId(reqBo.getOrderId());
			//设置销售单id
			updateSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
			//设置更新时间
			updateSaleOrderDo.setUpdateTime(new Date());
			//设置更新人id
			updateSaleOrderDo.setUpdateOperId(String.valueOf(reqBo.getUserId()));
			this.iUocSaleOrderModel.updateSaleStatusAndProCode(updateSaleOrderDo);
		}
	}

	/**
	 * 参数校验
	 * @param reqBo
	 */
	private void validateArg(UocShipOrderArrivaConfirmOrRefuseReqBo reqBo) {
        if (null == reqBo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象不能为空");
        }
        if (null == reqBo.getOrderId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参订单ID不能为空");
        }
        if (null == reqBo.getSaleOrderId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参销售单ID不能为空");
        }
        if (CollectionUtils.isEmpty(reqBo.getShipOrderIdList())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参发货单ID数组不能为空");
		}
		if (null == reqBo.getOpFlag()) {
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参操作不能为空");
		}
		if (!OP_FLAG_CONFIRM.equals(reqBo.getOpFlag()) && !OP_FLAG_REFUSE.equals(reqBo.getOpFlag())) {
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "操作仅支持到货确认或拒收");
		}
		if (null == reqBo.getUserId()) {
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参用户ID为空");
		}
	}
}
